{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  This unit defines classes for search pattern edit controls.    }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_SearchEdit;

interface

uses
  ActnList, Graphics, Buttons, Classes, Controls, Dialogs, ExtCtrls, Forms,
  StdCtrls, SysUtils, Windows, uROR_GridView, uROR_Selector, uROR_Classes;

type

  {============================= TCCRSearchCommand =============================
    Overview:     Action codes for the TCCRSearchEdit advanced search mode.
    SeeAlso:      TCCRSearchEdit.OnSearch; TCCRSearchEdit.OnSearchEnd;
                  TCCRSearchEdit.OnSearchStart
  }
  TCCRSearchCommand = (

    cscContinue,  { Continue the search. }

    cscFinish,    { Complete the search. }

    cscCancel     { Cancel the search. }
    
  );

  {============================== TCCRValidateCode =============================
    Overview:     Result codes for the TCCRSearchEdit input validator.
    SeeAlso:      TCCRSearchEdit.OnValidate; TCCRSearchEdit.Validate
  }
  TCCRValidateCode = (

    vcOk,       { User input is valid. }

    vcInvalid,  { Invalid user input. }

    vcTooShort  { Input string is too short. }

  );

  {============================== TCCRSearchEvent ==============================
    Overview:     Event type for the search progress events.
    SeeAlso:      TCCRSearchEdit.OnSearch
    Description:
      This event is generated when the search is performed in enhanced mode.  
  }
  TCCRSearchEvent = procedure(aSender: TObject;
    var aCommand: TCCRSearchCommand) of object;

  {============================= TCCRValidateEvent =============================
    Overview:     Event type for the user input validation events.
    SeeAlso:      TCCRSearchEdit.OnValidate
    Description:
      This event is generated when user input should be validated.
  }
  TCCRValidateEvent = procedure(aSender: TObject; var aMessage: String) of object;

  {=============================== TCCRInlineEdit ==============================
    Overview:     Customized edit box for the search pattern edit control.
    SeeAlso:      TCCRSearchEdit.Edit
    Description:
      TCCRInlineEdit is a customized version of the single line edit control.
      It adds special processing of the Enter and Down Arrow keys that required
      by the search pattern edit control.
  }
  TCCRInlineEdit = class(TCustomEdit)
  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Respond to key press events.
      SeeAlso:      TWinControl.KeyDown
      Keywords:     KeyDown,TCCRInlineEdit
      Description:
        When a windowed control receives a key-down message (WM_KEYDOWN) from
        Windows, its message handler calls the DoKeyDown method. If DoKeyDown
        determines that the control should, in fact, process the character, it
        calls KeyDown, passing the key code and shift-key state in the
        <i>Key</i> and <i>Shift</i> parameters, respectively.
        <p>
        After calling the inherited method, TCCRInlineEdit checks if the user
        pressed Enter or Down Arrow key and the Owner property references an
        TCCRSearchEdit object. If this is true, then the owner's Search method
        is called in order to start the search.
    }
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;

  public

    property Text;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    //property Anchors;
    property AutoSelect;
    //property AutoSize;
    property BevelEdges;
    property BevelInner;
    property BevelKind default bkNone;
    property BevelOuter;
    property BiDiMode;
    property BorderStyle;
    property CharCase;
    property Color;
    //property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    //property Enabled;
    property Font;
    property HideSelection;
    property ImeMode;
    property ImeName;
    property MaxLength;
    property OEMConvert;
    property ParentBiDiMode;
    //property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    //property Visible;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property OnChange;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;

  end;

  {=============================== TCCRSearchEdit ==============================
    Overview:     Edit control with a button for search pattern.
    SeeAlso:      TCCRSearchEdit.CancelAction; TCCRSearchEdit.EnableCancel;
                  TCCRSearchEdit.OnSearch; TCCRSearchEdit.OnSearchEnd;
                  TCCRSearchEdit.OnSearchStart
    Description:
      TCCRSearchEdit is a specialized edit control with associated button. It
      is used for editing a search pattern and controlling a search. The
      search can be performed either in simple mode when all data that conforms
      to the search criteria is loaded at once or in advanced mode when records
      are loaded in portions. In the latter case, the user is able to cancel
      a long search.

      <b>Simple Mode</b>
      <p>
      Define the OnSearchStart event handler that should load all records that
      conform to the search pattern stored in the Text property into a data
      container or control. The handler should return cscFinish in case of
      success or cscCancel if no data was found.
      <p>
      Since all data is loaded at once, a search can take a lot of time. If you
      want to allow the user to cancel long searches, use advanced mode instead.

      <b>Enhanced Mode</b>
      <p>
      ** Optional: Define the OnSearchStart event handler if some preparations
      are needed before the search. The handler should return cscContinue to
      start the search or cscCancel to cancel it.
      <p>
      ** Required: Define the OnSearch event handler that will actually perform
      the search. The handler should load a portion of data that conforms the
      search criteria and return cscContinue. If there is no more data, the
      cscFinish should be returned. In order to cancel the search, the handler
      should return cscCancel.
      <p>
      ** Optional: Define the OnSearchEnd event handler if any post-processing
      is required.
      <p>
      ** Optional: Assign True to the EnableCancel property if you want to allow
      the user to cancel searches by clicking the internal button or by a
      keyboard shortcut.
      <p>
      ** Optional: Create an action, assign a keyboard shortcut that will be
      used for search cancellations, and link the action to the control using
      the CancelAction property.
  }
  TCCRSearchEdit = class(TCustomPanel)
  private

    fCancelAction:     TCustomAction;
    fIsSearchCanceled: Boolean;
    fDefaultResources: TCCRSingleton;
    fEdit:             TCCRInlineEdit;
    fEditColor:        TColor;
    fEnableCancel:     Boolean;
    fGoButton:         TSpeedButton;
    fMinLength:        Word;
    fOnSearch:         TCCRSearchEvent;
    fOnSearchEnd:      TCCRSearchEvent;
    fOnSearchStart:    TCCRSearchEvent;
    fOnValidate:       TCCRValidateEvent;
    fSearchCursor:     TCursor;
    fSearching:        Boolean;

    procedure ButtonClick(aSender: TObject);
    function  GetMaxLength: Word;
    procedure SetCancelAction(anAction: TCustomAction);
    procedure SetMaxLength(const aValue: Word);
    procedure SetSearching(const aValue: Boolean);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Respond to receiving input focus.
      SeeAlso:      TCCRSearchEdit.Edit; TWinControl.DoEnter
      Keywords:     DoEnter,TCCRSearchEdit
      Description:
        DoEnter is called automatically when the control receives the input
        focus. As implemented in TCCRSearchEdit, DoEnter sets focus to the
        internal edit control.
        <p>
        Descendant classes that override DoEnter should always call the
        inherited method.
    }
    procedure DoEnter; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Generates the OnSearch event.
      SeeAlso:      TCCRSearchEdit.OnSearch; TCCRSearchEdit.Search
      Description:
        DoSearch is called automatically when the search pattern edit control
        is performing the search in advanced mode and needs another portion of
        data that conforms to the search criteria. As implemented in
        TCCRSearchEdit, DoSearch calls the OnSearch event handler, if defined.
        <p>
        Code of the search control command returned by the event handler is
        assigned to the <i>aCommand</i> parameter. 
    }
    procedure DoSearch(var aCommand: TCCRSearchCommand); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Generates the OnSearchEnd event.
      SeeAlso:      TCCRSearchEdit.OnSearchEnd; TCCRSearchEdit.Search
      Description:
        DoSearchEnd is called automatically after the search is complete or
        cancelled by the user. As implemented in TCCRSearchEdit, DoSearchEnd
        calls the OnSearchEnd event handler, if defined.
        <p>
        Code of the search control command returned by the event handler is
        assigned to the <i>aCommand</i> parameter.
    }
    procedure DoSearchEnd(var aCommand: TCCRSearchCommand); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Generates the OnSearchStart event.
      SeeAlso:      TCCRSearchEdit.OnSearchStart; TCCRSearchEdit.Search
      Description:
        DoSearchStart is called automatically in order to prepare the control
        for the search in advanced mode or to perform the search in simple mode.
        As implemented in TCCRSearchEdit, DoSearchStart calls the OnSearchStart
        event handler, if defined.
        <p>
        Code of the search control command returned by the event handler is
        assigned to the <i>aCommand</i> parameter.
    }
    procedure DoSearchStart(var aCommand: TCCRSearchCommand); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the Modified property.
      SeeAlso:      TCCRSearchEdit.Edit; TCCRSearchEdit.Modified;
                    TCCRSearchEdit.SetModified
      Keywords:     GetModified,TCCRSearchEdit
      Description:
        GetModified is the read implementation of the Modified property. As
        implemented in TCCRSearchEdit, GetModified returns the value of the
        Modified property of the internal edit control.
    }
    function GetModified: Boolean; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the Text property.
      SeeAlso:      TCCRSearchEdit.Edit; TCCRSearchEdit.SetText;
                    TCCRSearchEdit.Text
      Keywords:     GetText,TCCRSearchEdit
      Description:
        GetText is the read implementation of the Text property. As
        implemented in TCCRSearchEdit, GetText returns the value of the
        Text property of the internal edit control.
    }
    function GetText: String; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Initializes the control after it is loaded from a stream.
      SeeAlso:      TControl.Loaded
      Keywords:     Loaded,TCCRSearchEdit
      Description:
        Do not call Loaded. The VCL streaming system calls Loaded automatically
        after the controls form is loaded into memory so that the control can
        complete any initializations that depend on other objects in the form.
    }
    procedure Loaded; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Respond to control resize.
      SeeAlso:      TControl.Resize
      Keywords:     Resize,TCCRSearchEdit
      Description:
        Resize is called automatically immediately after the controls
        dimensions change. As implemented in TCCRSearchEdit, Resize adjusts
        dimensions and position of internal edit and button controls after
        calling the inherited method.
    }
    procedure Resize; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Enabled property.
      SeeAlso:      TCCRSearchEdit.Edit; TControl.SetEnabled
      Keywords:     SetEnabled,TCCRSearchEdit
      Description:
        SetEnabled is the protected write implementation of the Enabled property.
        TCCRSearchEdit overrides this method in order to change the color of
        the internal edit control (see the Edit property) according to control
        state. Color of an enabled edit box is defined by its Color property.
        Disabling the control changes the edit box color to that of the
        TCCRSearchEdit object.
        <p>
        When overriding SetEnabled, be sure to call the inherited method.
    }
    procedure SetEnabled(aValue: Boolean); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Modified property.
      SeeAlso:      TCCRSearchEdit.Edit; TCCRSearchEdit.GetModified;
                    TCCRSearchEdit.Modified
      Keywords:     SetModified,TCCRSearchEdit
      Description:
        SetModified is the protected write implementation of the Modified
        property. As implemented in TCCRSearchEdit, SetModified assigns the
        <i>aValue</i> to the Modified property of the internal edit control.
    }
    procedure SetModified(aValue: Boolean); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the name of the control.
      SeeAlso:      TControl.Name; TControl.SetName
      Keywords:     SetName,TCCRSearchEdit
      Description:
        SetName is the protected write implementation of the Name property.
        As implemented in TCCRSearchEdit, it clears the Caption property after
        calling the inherited method.
    }
    procedure SetName(const NewName: TComponentName); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Text property.
      SeeAlso:      TCCRSearchEdit.Edit; TCCRSearchEdit.GetText;
                    TCCRSearchEdit.Text
      Keywords:     SetText,TCCRSearchEdit
      Description:
        SetText is the protected write implementation of the Text property.
        As implemented in TCCRSearchEdit, SetText assigns the <i>aValue</i> to
        the Text property of the internal edit control.
    }
    procedure SetText(aValue: String); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal button that initiates/cancels the search.
      SeeAlso:      TCCRSearchEdit.Edit; TCCRSearchEdit.Search
      Description:
        Use GoButton to access properties of the internal button that
        initiates/cancels the search.
    }
    property GoButton: TSpeedButton  read fGoButton;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRSearchEdit.
      SeeAlso:      TComponent.Owner; TCustomPanel.Create
      Keywords:     Create,TCCRSearchEdit
      Description:
        Call Create to instantiate a search pattern edit control at runtime.
        Controls placed in forms at design time do not need to be explicitly
        created because they are created automatically.
        <p>
        <i>anOwner</i> is the component, typically a form, that is responsible
        for freeing the search pattern edit control.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRSearchEdit.
      SeeAlso:      TCustomControl.Destroy; TObject.Free
      Keywords:     Destroy,TCCRSearchEdit
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the control is not nil, and only then calls Destroy.
        <p>
        Applications should only free controls explicitly when the constructor
        was called without assigning an owner to the control.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Cancel the search (advanced mode).
      SeeAlso:      TCCRSearchEdit.Search; TCCRSearchEdit.Searching
      Description:
        CancelSearch cancels the search performed in advanced mode.
    }
    procedure CancelSearch; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Performs the search.
      SeeAlso:      TCCRSearchEdit.OnSearch; TCCRSearchEdit.OnSearchEnd;
                    TCCRSearchEdit.OnSearchStart; TCCRSearchEdit.Searching
      Keywords:     Search,TCCRSearchEdit
      Description:
        Call Search to initiate search for the pattern specified by the Text
        property. It returns True if the search was completed successfully and
        False if it was cancelled.
        <p>
        Firstly, Search validates the search pattern by calling
        Validate. If the pattern is not valid, Search returns False.
        <p>
        Otherwise, it calls DoSearchStart to perform the search in simple mode
        or prepare for the search in advanced mode.
        <p>
        Then, if DoSearchStart returned cscContinue (advanced mode), the
        DoSearch handler is called one or more times until it returns cscFinish
        or cscCancel.
        <p>
        Finally, the DoSearchEnd is called to perform an optional
        post-processing. The corresponding event handler (OnSearchEnd) can
        change the final outcome of the search. If it returns cscCancel, the
        Search method returns False. Otherwise, True is returned.
    }
    function  Search: Boolean; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Validates the search criteria.
      SeeAlso:      TCCRSearchEdit.OnValidate; TCCRSearchEdit.Text
      Keywords:     Validate,TCCRSearchEdit
      Description:
        Call Validate to check if the search criteria in general and the search
        pattern in particular are valid. Firstly, it checks if the Text property
        holds no less than MinLength characters. Then it calls the OnValidate
        event handler, if defined.
        <p>
        Result value of the function characterizes the outcome of the validation.
        If it is not vcOk, then the error message is returned via the
        <i>aMessage</i> parameter.
    }
    function Validate(var aMessage: String): TCCRValidateCode; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies if the search pattern was modified.
      SeeAlso:      TCCRSearchEdit.Edit; TCCRSearchEdit.Text;
                    TCustomEdit.Modified
      Keywords:     Modified,TCCRSearchEdit
      Description:
        Use Modified property to get or set the value of the property of the
        same name of the internal edit control. True is automatically assigned
        to this property when the text of the control (search pattern) is
        modified.
    }
    property Modified: Boolean  read GetModified  write SetModified;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates that a search is in progress.
      SeeAlso:      TCCRSearchEdit.EnableCancel; TCCRSearchEdit.Search
      Keywords:     Searching,TCCRSearchEdit
      Description:
        Use Searching to get or set the "search in progress" status of the
        control. The Search method automatically updates the value of this
        property.
        <p>
        If the EnableCancel property value is True, then changing the value of
        the Searching property also changes the function and appearance of the
        internal button that initiates/cancels the search. When Searching is
        False, the button displays a magnifying glass and clicking on it will
        initiate the search. When True is assigned to the Searching property,
        the button image changes to a red 'X' and clicking on it will cancel
        the search.
    }
    property Searching: Boolean  read fSearching  write SetSearching;

  published

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property Align;
    //property Alignment;
    property Anchors;
    //property AutoSize;
    //property BevelInner;
    //property BevelOuter;
    //property BevelWidth;
    //property BiDiMode;
    //property BorderWidth;
    //property BorderStyle;
    //property Caption;
    property Color;
    property Constraints;
    property Ctl3D;
    //property UseDockManager default True;
    //property DockSite;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FullRepaint;
    property Font;
    //property Locked;
    //property ParentBiDiMode;
    {$IFDEF VERSION7}
    property ParentBackground;
    {$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    }
    property OnCanResize;
    property OnClick;
    property OnConstrainedResize;
    property OnContextPopup;
    //property OnDockDrop;
    //property OnDockOver;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    //property OnGetSiteInfo;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnStartDock;
    property OnStartDrag;
    //property OnUnDock;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Links the control to an action that cancels the search.
      SeeAlso:      TCCRSearchEdit.EnableCancel
      Description:
        Use CancelAction to link a control to an action with a keyboard shortcut
        that will be used to cancel searches. Usually, the same action can be
        linked to several search pattern edit controls.
    }
    property CancelAction: TCustomAction
      read fCancelAction  write SetCancelAction;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Internal edit control.
      SeeAlso:      TCCRSearchEdit.GoButton
      Keywords:     Edit,TCCRSearchEdit
      Description:
        Use Edit to access properties of the internal edit control that is used
        for editing the search pattern.
    }
    property Edit: TCCRInlineEdit  read fEdit;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Enables search cancellations.
      SeeAlso:      TCCRSearchEdit.Search; TCCRSearchEdit.Searching
      Description:
        Use EnableCancel to enable or disable cancellations of searches
        (advanced mode).
        <p>
        If the values of this property is True, then the value of the Searching
        property controls the appearance and function of the internal button
        that initiates/cancels the search.
    }
    property EnableCancel: Boolean
      read fEnableCancel  write fEnableCancel  default False;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies maximum length of a search pattern.
      SeeAlso:      TCCRSearchEdit.MinLength; TCCRSearchEdit.OnValidate
      Keywords:     MaxLength,TCCRSearchEdit
      Description:
        Use MaxLength to limit the number of characters in the search pattern.
        A value of 0 indicates that there is no application-defined limit on
        the length.
        <p>
        <b>Note:</b> Setting MaxLength will not truncate the existing text, it
        merely prevents the user from adding more text after reaching the limit
        of MaxLength characters.
        <p>
        <b>Note:</b> Even when MaxLength is 0, there may be limitations imposed
        by the operating system on the number of characters that may be entered
        into an edit control.
    }
    property MaxLength: Word  read GetMaxLength  write SetMaxLength;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies minimum length of a search pattern.
      SeeAlso:      TCCRSearchEdit.MaxLength; TCCRSearchEdit.OnValidate
      Keywords:     MinLength,TCCRSearchEdit
      Description:
        Use MinLength to get or set minimum number of characters allowed for a
        search pattern.
    }
    property MinLength: Word  read fMinLength  write fMinLength;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the image used to represent the mouse pointer
                    during the search.
      SeeAlso:      TControl.DragCursor; TScreen.Cursor
      Description:
        Change the value of Cursor to provide feedback to the user when the
        search is in progress. The value of Cursor is the index of the cursor
        in the list of cursors maintained by the global variable, Screen. In
        addition to the built-in cursors provided by TScreen, applications can
        add custom cursors to the list.
    }
    property SearchCursor: TCursor  read fSearchCursor  write fSearchCursor;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when a control needs another portion of data that
                    conforms to the search criteria.
      SeeAlso:      TCCRSearchEdit.DoSearch; TCCRSearchEdit.OnSearchEnd;
                    TCCRSearchEdit.OnSearchStart; TCCRSearchEdit.Search
      Description:
        Write an OnSearch event handler if you want a control to perform a
        search in advanced mode. The handler code should load a predetermined
        amount of data that conforms to the search criteria (search pattern and
        optional modifiers).
        <p>
        <i>aSender</i> is the search pattern edit control whose event handler
        is called. Initial value of the <i>aCommand</i> parameter is
        cscContinue. Assign cscFinish to the <i>aCommand</i> if the search
        returned no data. cscCancel aborts the search.
    }
    property OnSearch: TCCRSearchEvent  read fOnSearch  write fOnSearch;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs after the search is completed/cancelled.
      SeeAlso:      TCCRSearchEdit.DoSearchEnd; TCCRSearchEdit.OnSearch;
                    TCCRSearchEdit.OnSearchStart; TCCRSearchEdit.Search
      Description:
        Write an OnSearchEnd event handler if some additional processing is
        needed after a search is completed or cancelled. This event occurs
        both in simple and advanced search mode.
        <p>
        <i>aSender</i> is the search pattern edit control whose event handler
        is called. Initial value of the <i>aCommand</i> parameter indicates
        outcome of the search: cscFinish or cscCancel. Modify it if you want to
        change the final result. If cscCancel is returned via <i>aCommand</i>,
        then the Search method returns False. Otherwise, True is returned.
    }
    property OnSearchEnd: TCCRSearchEvent  read fOnSearchEnd  write fOnSearchEnd;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs before the search.
      SeeAlso:      TCCRSearchEdit.DoSearchStart; TCCRSearchEdit.OnSearch;
                    TCCRSearchEdit.OnSearchEnd; TCCRSearchEdit.Search
      Description:
        Write an OnSearchStart event handler to perform a search in simple mode
        or prepare for a search in advanced mode. <i>aSender</i> is the search
        pattern edit control whose event handler is called. Initial value of the
        <i>aCommand</i> parameter is cscContinue.
        <p>
        In simple mode, this handler should load all data that conforms to the
        search criteria search pattern and optional modifiers) and assign
        cscFinish to the <i>aCommand</i> parameter. cscCancel indicates that the
        search was aborted.
        <p>
        In advanced mode, OnSearchStart should prepare the environment for the
        OnSearch handler that will perform the search and return cscContinue
        via the <i>aCommand</i> parameter. cscCancel indicates that the search
        was aborted.
    }
    property OnSearchStart: TCCRSearchEvent
      read fOnSearchStart  write fOnSearchStart;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Occurs when a control needs to validate a search criteria.
      SeeAlso:      TCCRSearchEdit.Text; TCCRSearchEdit.Validate
      Keywords:     OnValidate,TCCRSearchEdit
      Description:
        Write an OnValidate event handler if some custom validation rules
        should be applied to the search criteria in general and/or the search
        pattern in particular.
        <p>
        <i>aSender</i> is the search pattern edit control whose event handler is
        called. Assign an error message to the <i>aMessage</i> parameter
        (initially, it is empty) to indicate a validation failure.
    }
    property OnValidate: TCCRValidateEvent  read fOnValidate  write fOnValidate;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies a search pattern.
      SeeAlso:      TCCRSearchEdit.Edit; TCCRSearchEdit.MaxLength;
                    TCCRSearchEdit.MinLength; TCCRSearchEdit.OnValidate
      Keywords:     Text,TCCRSearchEdit
      Description:
        Use the Text property to get or set the search pattern in the internal
        edit control.
    }
    property Text: String  read GetText  write SetText;

  end;

  {============================ TCCRVistASearchEdit ============================
    Overview:     CCR-specific version of a search pattern editor.
    SeeAlso:      ???
    Description:
      TCCRVistASearchEdit is a CCR-specific version of a search pattern editor.
      Use TCCRSearchEdit instead.
      <p>
      <b>Note</b>: If you choose to use this control, do this on your own risk.
      TCCRVistASearchEdit is not supported and can change at any time without
      warning.
  }
  TCCRVistASearchEdit = class(TCCRSearchEdit)
  private

    fRawData:            TStringList;
    fSearchFromIndex:    Integer;
    fSearchGrid:         TCCRGridView;
    fSearchMaxCount:     Integer;
    fSearchRPCName:      String;
    fSearchSelector:     TCCRSelector;
    fSearchStrPerItem:   Integer;
    fSuspendGridUpdates: Boolean;

    function  GetSearchFields: String;
    procedure SetSearchFields(const aValue: String);

  protected

    procedure DoSearchStart(var aCommand: TCCRSearchCommand); override;
    procedure Notification(aComponent: TComponent;
      Operation: TOperation); override;

  public

    SearchCount:       Integer;
    SearchFieldsArray: array of Integer;
    SearchParams:      array of String;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRVistASearchEdit.
      SeeAlso:      TComponent.Owner; TCustomPanel.Create
      Keywords:     Create,TCCRVistASearchEdit
      Description:
        Call Create to instantiate a search pattern edit control at runtime.
        Controls placed in forms at design time do not need to be explicitly
        created because they are created automatically.
        <p>
        <i>anOwner</i> is the component, typically a form, that is responsible
        for freeing the search pattern edit control.
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRVistASearchEdit.
      SeeAlso:      TCustomControl.Destroy; TObject.Free
      Keywords:     Destroy,TCCRVistASearchEdit
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the control is not nil, and only then calls Destroy.
        <p>
        Applications should only free controls explicitly when the constructor
        was called without assigning an owner to the control.
    }
    destructor Destroy; override;

    function Search: Boolean; override;
    procedure SetParams(Params: array of String);

    property RawData: TStringList  read fRawData;

  published

    property SearchFields: String  read GetSearchFields  write SetSearchFields;

    property SearchFromIndex: Integer
      read fSearchFromIndex  write fSearchFromIndex  default -1;

    property SearchGrid: TCCRGridView  read fSearchGrid  write fSearchGrid;

    property SearchMaxCount: Integer
      read fSearchMaxCount  write fSearchMaxCount  default 0;

    property SearchRPCName: String  read fSearchRPCName  write fSearchRPCName;

    property SearchSelector: TCCRSelector
      read fSearchSelector  write fSearchSelector;

    property SearchStrPerItem: Integer
      read fSearchStrPerItem  write fSearchStrPerItem  default 1;

    property SuspendGridUpdates: Boolean
      read fSuspendGridUpdates  write fSuspendGridUpdates  default True;

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  uROR_Resources, uROR_Utilities;

type

  {======================= TCCRSearchEditDefaultResources ======================
    Overview:     Singleton for default resources.
    Description:
      TCCRSearchEditDefaultResources handles a single copy of default resources
      for search edit controls.
  }
  TCCRSearchEditDefaultResources = class(TCCRSingleton)
  private

    fSearchCancel: Graphics.TBitmap;
    fSearchStart:  Graphics.TBitmap;

  protected

    procedure Finalize; override;
    procedure Initialize; override;

  public

    property SearchCancel: Graphics.TBitmap  read fSearchCancel;

    property SearchStart: Graphics.TBitmap  read fSearchStart;

  end;

///////////////////////////////// TCCRInlineEdit \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRInlineEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited;
  if (Key in [VK_RETURN, VK_DOWN]) and Assigned(Owner) then
    if Owner is TCCRSearchEdit then
      TCCRSearchEdit(Owner).Search;
end;

///////////////////////////////// TCCRSearchEdit \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRSearchEdit.Create(anOwner: TComponent);
begin
  inherited;
  fDefaultResources := TCCRSearchEditDefaultResources.Create;

  fEdit          := TCCRInlineEdit.Create(Self);
  fGoButton      := TSpeedButton.Create(Self);

  AutoSize       := False;
  BevelInner     := bvNone;
  BevelOuter     := bvNone;
  BorderStyle    := bsSingle;
  Caption        := '';
  DockSite       := False;
  ParentBiDiMode := True;

  Constraints.MinHeight := 16;
  Constraints.MinWidth  := 16;

  with Edit do
    begin
      SetSubComponent(True);
      Parent      := Self;
      BorderStyle := bsNone;
      Name        := 'Edit';
      Text        := '';
    end;

  with GoButton do
    begin
      SetSubComponent(True);
      Parent  := Self;

      Name           := 'Search';
      Caption        := '';
      Flat           := False;
      Glyph          := TCCRSearchEditDefaultResources(fDefaultResources).SearchStart;
      Hint           := RSC0012;
      OnClick        := ButtonClick;
      ParentShowHint := True;
      Visible        := True;
    end;

  SetBounds(Left, Top, Edit.Width + Edit.Height + 2, Edit.Height + 2);
end;

destructor TCCRSearchEdit.Destroy;
begin
  fCancelAction := nil;
  with fGoButton do
    begin
      Action := nil;
      Glyph  := nil;
    end;
  FreeAndNil(fEdit);
  FreeAndNil(fGoButton);
  
  FreeAndNil(fDefaultResources);
  inherited;
end;

procedure TCCRSearchEdit.ButtonClick(aSender: TObject);
begin
  if Searching then
    CancelSearch
  else
    Search;
end;

procedure TCCRSearchEdit.CancelSearch;
begin
  fIsSearchCanceled := True;
end;

procedure TCCRSearchEdit.DoEnter;
begin
  inherited;
  Edit.SetFocus;
end;

procedure TCCRSearchEdit.DoSearch(var aCommand: TCCRSearchCommand);
begin
  if Assigned(OnSearch) then
    OnSearch(Self, aCommand)
  else
    aCommand := cscFinish;
end;

procedure TCCRSearchEdit.DoSearchEnd(var aCommand: TCCRSearchCommand);
begin
  if Assigned(OnSearchEnd) then
    OnSearchEnd(Self, aCommand);
end;

procedure TCCRSearchEdit.DoSearchStart(var aCommand: TCCRSearchCommand);
begin
  if Assigned(OnSearchStart) then
    OnSearchStart(Self, aCommand);
end;

function TCCRSearchEdit.GetMaxLength: Word;
begin
  if Assigned(Edit) then
    Result := Edit.MaxLength
  else
    Result := 0;
end;

function TCCRSearchEdit.GetModified: Boolean;
begin
  if Assigned(Edit) then
    Result := Edit.Modified
  else
    Result := False;
end;

function TCCRSearchEdit.GetText: String;
begin
  if Assigned(Edit) then
    Result := Edit.Text
  else
    Result := '';
end;

procedure TCCRSearchEdit.Loaded;
begin
  inherited;
  if Assigned(Edit) and Edit.ParentColor then
    begin
      Edit.ParentColor := False;  // Enforce correct Parent Color
      Edit.ParentColor := True;
    end;
  Resize;
end;

procedure TCCRSearchEdit.Resize;
begin
  inherited;
  Edit.SetBounds(0, 0, ClientWidth - ClientHeight, ClientHeight);
  GoButton.SetBounds(Edit.Width, 0, ClientHeight, ClientHeight);
end;

function TCCRSearchEdit.Search: Boolean;
var
  vc: TCCRValidateCode;
  cmd: TCCRSearchCommand;
  oldCursor: TCursor;
  restoreCursor: Boolean;
  aMsg: String;
begin
  Result := False;

  vc := Validate(aMsg);
  case vc of
    vcTooShort:
      MessageDlg508('', Format(RSC0010, [MinLength]), mtWarning, [mbOK], 0);
    vcInvalid:
      MessageDlg508('', RSC0011 + #13 + aMsg, mtWarning, [mbOK], 0);
    vcOk:
      begin
        cmd := cscContinue;
        restoreCursor := False;
        fIsSearchCanceled := False;

        Searching := True;
        try
          oldCursor := Screen.Cursor;
          if Screen.Cursor <> SearchCursor then
            begin
              Screen.Cursor := SearchCursor;
              restoreCursor := True;
            end;
          DoSearchStart(cmd);
          try
            if cmd = cscContinue then
              repeat
                Application.ProcessMessages;
                if fIsSearchCanceled then
                  Break;
                DoSearch(cmd);
              until cmd <> cscContinue;
          finally
            if cmd = cscContinue then
              cmd := cscCancel;
            DoSearchEnd(cmd);
          end;
        finally
          if restoreCursor then
            Screen.Cursor := oldCursor;
          Searching := False;
        end;

        Result := (cmd <> cscCancel);
      end;
  end;

  if not Result then
    Edit.SetFocus;
end;

procedure TCCRSearchEdit.SetCancelAction(anAction: TCustomAction);
begin
  if anAction <> fCancelAction then
    begin
      fCancelAction := anAction;
      if Assigned(fCancelAction) then
        fCancelAction.Caption := '';
    end;
end;

procedure TCCRSearchEdit.SetEnabled(aValue: Boolean);
begin
  if aValue <> Enabled then
    begin
      inherited;
      if aValue then
        Edit.Color := fEditColor
      else
        begin
          fEditColor := Edit.Color;
          Edit.ParentColor := True;
        end;
      Edit.Enabled := aValue;
    end;
end;

procedure TCCRSearchEdit.SetMaxLength(const aValue: Word);
begin
  if Assigned(Edit) then
    Edit.MaxLength := aValue;
end;

procedure TCCRSearchEdit.SetModified(aValue: Boolean);
begin
  if Assigned(Edit) then
    Edit.Modified := aValue;
end;

procedure TCCRSearchEdit.SetName(const NewName: TComponentName);
begin
  inherited;
  Caption := '';
end;

procedure TCCRSearchEdit.SetSearching(const aValue: Boolean);
begin
  if aValue <> fSearching then
    if aValue then
      begin
        fSearching := True;
        if EnableCancel then
          begin
            with GoButton do
              begin
                Action := CancelAction;
                Glyph  := TCCRSearchEditDefaultResources(fDefaultResources).SearchCancel;
                Hint   := RSC0013;
              end;
          end;
      end
    else
      begin
        if EnableCancel then
          begin
            with GoButton do
              begin
                Action  := nil;
                Glyph   := TCCRSearchEditDefaultResources(fDefaultResources).SearchStart;
                Hint    := RSC0012;
                OnClick := ButtonClick;
              end;
          end;
        fSearching := False;
      end;
end;

procedure TCCRSearchEdit.SetText(aValue: String);
begin
  if Assigned(Edit) then
    Edit.Text := aValue;
end;

function TCCRSearchEdit.Validate(var aMessage: String): TCCRValidateCode;
begin
  aMessage := '';
  Result := vcOk;
  if Length(Text) < MinLength then
    Result := vcTooShort
  else if Assigned(OnValidate) then
    begin
      OnValidate(Self, aMessage);
      if aMessage <> '' then
        Result := vcInvalid;
    end;
end;

//////////////////////// TCCRSearchEditDefaultResources \\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRSearchEditDefaultResources.Finalize;
begin
  FreeAndNil(fSearchStart);
  FreeAndNil(fSearchCancel);
  inherited;
end;

procedure TCCRSearchEditDefaultResources.Initialize;
begin
  inherited;

  fSearchCancel := Graphics.TBitmap.Create;
  fSearchStart  := Graphics.TBitmap.Create;

  SearchCancel.LoadFromResourceName(HInstance, 'BTN_CANCEL');
  SearchStart.LoadFromResourceName(HInstance,  'BTN_SEARCH');
end;

////////////////////////////// TCCRVistASearchEdit \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRVistASearchEdit.Create(anOwner: TComponent);
begin
  inherited;
  fRawData := TStringList.Create;
  fSearchFromIndex  := -1;
  fSearchStrPerItem := 1;
  fSuspendGridUpdates := True;
end;

destructor TCCRVistASearchEdit.Destroy;
begin
  FreeAndNil(fRawData);
  inherited;
end;

procedure TCCRVistASearchEdit.DoSearchStart(var aCommand: TCCRSearchCommand);
begin
  SearchGrid.Clear;
  SearchCount := 0;
  inherited;
  if (SearchFromIndex >= 0) and (SearchFromIndex < Length(SearchParams)) then
    SearchParams[SearchFromIndex] := '';
end;

function TCCRVistASearchEdit.GetSearchFields: String;
var
  i: Integer;
begin
  if Length(SearchFieldsArray) > 0 then
    begin
      Result := IntToStr(SearchFieldsArray[0]);
      for i:=1 to High(SearchFieldsArray) do
        Result := Result + ',' + IntToStr(SearchFieldsArray[i]);
    end
  else
    Result := '';
end;

procedure TCCRVistASearchEdit.Notification(aComponent: TComponent; Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) then
    begin
      if aComponent = SearchGrid then
        SearchGrid := nil
      else if aComponent = SearchSelector then
        begin
          SearchSelector := nil;
          SearchGrid := nil;
        end;
    end;
end;

function TCCRVistASearchEdit.Search: Boolean;
begin
  if not Assigned(SearchGrid) and Assigned(SearchSelector) then
    SearchGrid := SearchSelector.SourceList;
  if Assigned(SearchGrid) then
    begin
      RawData.Clear;
      if SuspendGridUpdates then
        SearchGrid.Items.BeginUpdate;
      try
        Result := inherited Search;
      finally
        if SuspendGridUpdates then
          SearchGrid.Items.EndUpdate;
        RawData.Clear;
      end;
    end
  else
    Result := False;
end;

procedure TCCRVistASearchEdit.SetParams(Params: array of String);
var
  i, l: Integer;
begin
  l := Length(Params);
  SetLength(SearchParams, l);
  for i:=l-1 downto 0 do
    SearchParams[i] := Params[i];
end;

procedure TCCRVistASearchEdit.SetSearchFields(const aValue: String);
var
  i, n: Integer;
  fld: String;
begin
  i := 0;
  n := 0;
  while True do
    begin
      Inc(i);
      fld := Piece(aValue,',',i);
      if fld = '' then
        Break;
      if i >= n then
        begin
          Inc(n, 10);
          SetLength(SearchFieldsArray, n);
        end;
      SearchFieldsArray[i-1] := StrToIntDef(fld, 0);
    end;
  SetLength(SearchFieldsArray, i-1);
end;

end.
