unit uROR_ICD9Selector;

interface
{$IFNDEF NOVTREE}

uses
  Forms, Classes, Controls, uROR_SelectorTree, Buttons, VirtualTrees,
  uROR_TreeGrid, uROR_GridView, Graphics, Types;

const
  //--- Columns

  idscDummy = 0;
  idscName  = 1;
  idscCode  = 2;
  idscDescr = 3;
  idscId    = 4;
  idscVersion = 5; // ROR*1.5*19 2012-01-09

  //--- Tree Levels

  idslGroup   = 0;
  idslItem    = 1;

type

  TCCRICD9SelectorTree = class;

  TCCRICD9Selector = class(TCCRSelectorTree)
  private
    fProcMode:   Boolean;
    fICDVersion:    String;

    function  getResultTree: TCCRICD9SelectorTree;

  protected
    function  CreateResultControl: TWinControl; override;
    procedure DoUpdateButtons(var EnableAdd, EnableAddAll,
      EnableRemove, EnableRemoveAll: Boolean); override;
    procedure Loaded; override;
    procedure ResultTreePaintTextHandler(Sender: TBaseVirtualTree;
      const TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
      TextType: TVSTTextType);
    procedure ResultTreeSortHandler(aSender: TCCRTreeGrid;
      var aColumn: TColumnIndex; var aDirection: TSortDirection);

  public
    constructor Create(anOwner: TComponent); override;

    procedure AddSelectedItems; override;
    procedure Clear; override;
    procedure RemoveSelectedItems; override;

    property ResultTree: TCCRICD9SelectorTree       read    getResultTree;

    property ProcMode: Boolean                      read    fProcMode
                                                    write   fProcMode;

    property ICDVersion: String                     read    fICDVersion
                                                    write   fICDVersion;
  end;

  TCCRICD9SelectorTree = class(TCCRResultTree)
  private
    HasItems:     Boolean;

  protected
    function  DoDragOver(Source: TObject; Shift: TShiftState; State: TDragState;
      Pt: TPoint; Mode: TDropMode; var Effect: Integer): Boolean; override;

  public
    constructor Create(anOwner: TComponent); override;

    function  AddGroup(const aGroupName: String): PVirtualNode;
// ROR*1.5*19 2012-01-09 =====================================================
//    function  AddItem(aParent: PVirtualNode; const aName: String;
//      const aCode: String = ''): PVirtualNode;
    function  AddItem(aParent: PVirtualNode; const aSystem: String; const aName: String;
      const aCode: String = ''): PVirtualNode;
    procedure Clear; override;
    procedure DeleteGroup(aNode: PVirtualNode);
    procedure DeleteItem(aNode: PVirtualNode);
    procedure EndUpdate;
    function  FindGroup(const aName: String): PVirtualNode;
    function  GetFirstResultItem: PVirtualNode;
    function  GetGroup(aNode: PVirtualNode): PVirtualNode;

  end;

{$ENDIF}
implementation
{$IFNDEF NOVTREE}

uses
  Dialogs, SysUtils, uROR_Resources, ComCtrls, uROR_CustomSelector;

//////////////////////////////// TCCRICD9Selector \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRICD9Selector.Create(anOwner: TComponent);
begin
  inherited;

  with SourceList do
    begin
      SortField := -1;
      SortField := idscName;
    end;

  with ResultTree do
    begin
      with Header do
        begin
          Options := Options + [hoVisible,hoShowSortGlyphs];
          SortColumn  := idscName;
        end;

      with TreeOptions do
        begin
          AutoOptions  := AutoOptions  + [toAutoSpanColumns];
          MiscOptions  := MiscOptions  - [toAcceptOLEDrop];
          MiscOptions  := MiscOptions  + [toCheckSupport,toReportMode];
          PaintOptions := PaintOptions + [toShowHorzGridLInes,toShowVertGridLInes];
        end;

      OnPaintText := ResultTreePaintTextHandler;
      OnSort      := ResultTreeSortHandler;
    end;

  ResultChanged := False;
end;

procedure TCCRICD9Selector.AddSelectedItems;

  procedure copy_items(aGroup: PVirtualNode; fldLst: array of Integer);
  var
    srci: TCCRGridItem;
    dsti: PVirtualNode;
    oldCursor: TCursor;
    restoreCursor: Boolean;
    itemId: String;
    i: Integer;
  begin
    oldCursor := Screen.Cursor;
    if SourceList.SelCount > 30 then
      begin
        Screen.Cursor := crHourGlass;
        restoreCursor := True;
      end
    else
      restoreCursor := False;

    ResultTree.BeginUpdate;
    try
      srci := SourceList.Selected;
      while Assigned(srci) do
        begin
          itemId := srci.AsString[idscId];
          if (itemId = '') or (ResultTree.Find(itemId, idscId, aGroup) = nil) then
            begin
// ROR*1.5*19 2012-01-09 adding System to the grid ===========================
//              dsti := ResultTree.AddItem(aGroup, srci.AsString[idscName], itemId);
              dsti := ResultTree.AddItem(aGroup, srci.AsString[idscVersion], srci.AsString[idscName], itemId);
              ResultTree.AsString[dsti,idscDummy] := ' ';
              for i:=0 to High(fldLst) do
                ResultTree.AsString[dsti,fldLst[i]] := srci.AsString[fldLst[i]];
            end;
          srci := SourceList.GetNextItem(srci, sdAll, [isSelected]);
        end;
    finally
      ResultTree.EndUpdate;
      if restoreCursor then
        Screen.Cursor := oldCursor;
    end;
  end;

var
  groupNode: PVirtualNode;
  next: TCCRGridItem;

begin
  if SourceList.SelCount = 0 then
    Exit;

  try
    groupNode := ResultTree.GetGroup(ResultTree.FocusedNode);
    if not Assigned(groupNode) then
      raise Exception.Create('Group not found');
  except
    groupNode := ResultTree.GetFirstChild(nil);
    if not Assigned(groupNode) then
      Exit;
  end;

  SourceList.Items.BeginUpdate;
  try
    copy_items(groupNode, [idscDescr]);

    if AutoSort then
      ResultTree.SortData;
    with ResultTree do
      begin
        FullCollapse;
        Expanded[groupNode] := True;
      end;

    with SourceList do
      begin
        //--- Find the first item after selection
        next := GetItemFollowingSelection;
        //--- Remove the selected items from the source list
        DeleteSelected;
        //--- Select the item
        if Assigned(next) then
          begin
            Selected := next;
            ItemFocused := Selected;
          end
        else if Items.Count > 0 then
          begin
            Selected := Items[Items.Count-1];
            ItemFocused := Selected;
          end;
      end;
  finally
    SourceList.Items.EndUpdate;
  end;
end;

procedure TCCRICD9Selector.Clear;
begin
  inherited;
  if Assigned(ResultTree) then
    with ResultTree do
      begin
        BeginUpdate;
        try
          if ProcMode then
            AddGroup('Procedures')
          else
            AddGroup('Diagnoses');
          FullExpand;
        finally
          EndUpdate;
        end;
      end;
end;

function TCCRICD9Selector.CreateResultControl: TWinControl;
begin
  Result := TCCRICD9SelectorTree.Create(Self);
end;

procedure TCCRICD9Selector.DoUpdateButtons(var EnableAdd, EnableAddAll,
  EnableRemove, EnableRemoveAll: Boolean);
begin
  with ResultTree do
    begin
      EnableRemove     := HasItems and (SelectedCount > 0);
      EnableRemoveAll  := HasItems;
      if GetFirstChild(RootNode) = nil then
        begin
          EnableAdd    := False;
          EnableAddAll := False;
        end;
    end;
  inherited;
end;

function TCCRICD9Selector.getResultTree: TCCRICD9SelectorTree;
begin
  Result := TCCRICD9SelectorTree(GetResultControl);
end;

procedure TCCRICD9Selector.Loaded;
begin
  inherited;
  with SourceList.Fields do
    if Count = 0 then
      begin
        with TCCRGridField(Add) do
          begin
            Caption := 'ID';
            DataType := gftMUMPS;
            Visible := False;
            Width := 60;
          end;
        with TCCRGridField(Add) do
          begin
            Caption := 'Name';
            Width := 100;
          end;
        with TCCRGridField(Add) do
          begin
            Caption := 'Code';
            DataType := gftMUMPS;
            Width := 60;
          end;
        with TCCRGridField(Add) do
          begin
            Caption := 'Description';
            Width := 100;
          end;
      end;

// ROR*1.5*19 2012-01-09 =============================================== begin
    with SourceList.Fields do
      if SourceList.Fields.Count = 5 then
        with TCCRGridField(Add) do
          begin
            Caption := 'Version';
            Width := 60;
          end;
// ROR*1.5*19 2012-01-09 =============================================== end

  with ResultTree.Header do
    if Columns.Count = 0 then
      begin
        with TCCRTreeGridColumn(Columns.Add) do
          begin
            Text := 'ID';
            DataType := gftMUMPS;
            Width := 60;
            Options := Options - [coVisible];
          end;
        with TCCRTreeGridColumn(Columns.Add) do
          begin
            Text := 'Name';
            Width := 100;
          end;
        with TCCRTreeGridColumn(Columns.Add) do
          begin
            Text := 'Code';
            DataType := gftMUMPS;
            Width := 60;
          end;
        with TCCRTreeGridColumn(Columns.Add) do
          begin
            Text := 'Description';
            Width := 100;
          end;
        MainColumn := idscName;
      end;
end;

procedure TCCRICD9Selector.RemoveSelectedItems;

  function next_item(aNode: PVirtualNode): PVirtualNode;
  begin
    with ResultTree do
      begin
        Result := GetNext(aNode);
        while Assigned(Result) do
          if GetNodeLevel(Result) = idslItem then
            Break
          else
            Result := GetNext(Result);
      end;
  end;

  function prev_item(aNode: PVirtualNode): PVirtualNode;
  begin
    with ResultTree do
      begin
        Result := GetPrevious(aNode);
        while Assigned(Result) do
          if GetNodeLevel(Result) = idslItem then
            Break
          else
            Result := GetPrevious(Result);
      end;
  end;

var
  last, nextSel, node: PVirtualNode;
begin
  if ResultTree.SelectedCount = 0 then
    Exit;

  ResultTree.BeginUpdate;
  try
    with ResultTree do
      begin
        //--- Find the last selected node
        last := nil;
        node := GetFirstSelected;
        while Assigned(node) do
          begin
            last := node;
            node := GetNextSelected(node);
          end;
        //--- Find item that will be selected
        if Assigned(last) then
          begin
            nextSel := next_item(last);
            if not Assigned(nextSel) then
              nextSel := prev_item(last);
          end
        else
          nextSel := nil;
        //---
        node := GetFirstSelected;
        while Assigned(node) do
          begin
            case GetNodeLevel(node) of
              idslGroup:
                begin
                  Selected[node] := False;
                  DeleteChildren(node, True);
                end;
              idslItem:
                begin
                  DeleteNode(node);
                  node := nil;
                end;
            end;
            if Assigned(node) then
              Selected[node] := False;
            node := GetFirstSelected;
          end;
        //---
        ClearSelection;
        if Assigned(nextSel) then
          begin
            Selected[nextSel] := True;
            FocusedNode := nextSel;
            ScrollIntoView(nextSel, False);
          end;
      end;
  finally
    ResultTree.EndUpdate;
  end;
end;

procedure TCCRICD9Selector.ResultTreePaintTextHandler(Sender: TBaseVirtualTree;
  const TargetCanvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex;
  TextType: TVSTTextType);
begin
  if Sender.GetNodeLevel(Node) < idslItem then
    TargetCanvas.Font.Style := [fsBold];
end;

procedure TCCRICD9Selector.ResultTreeSortHandler(aSender: TCCRTreeGrid;
  var aColumn: TColumnIndex; var aDirection: TSortDirection);
var
  vn: PVirtualNode;
begin
  with ResultTree do
    begin
      vn := GetFirst();
      while Assigned(vn) do
        begin
          if GetNodeLevel(vn) = idslGroup then
            Sort(vn, aColumn, aDirection);
          vn := GetNext(vn);
        end;
    end;
end;

////////////////////////////// TCCRICD9SelectorTree \\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRICD9SelectorTree.Create(anOwner: TComponent);
begin
  inherited;
  Clear;
end;

function TCCRICD9SelectorTree.AddGroup(const aGroupName: String): PVirtualNode;
begin
  BeginUpdate;
  try
    Result := AddChild(RootNode);
    AsString[Result,idscName] := aGroupName;
    FullCollapse;
    FullExpand(Result);
  finally
    EndUpdate;
  end;
end;

function TCCRICD9SelectorTree.AddItem(aParent: PVirtualNode; const aSystem:String;
  const aName: String;
  const aCode: String = ''): PVirtualNode;
begin
  Result := AddChild(aParent);
  AsString[Result,idscDummy] := ' ';
  AsString[Result,idscName]  := aName;
  AsString[Result,idscCode]  := aCode;
  AsString[Result,idscId]    := aCode;
  AsString[Result,idscVersion]:=aSystem; // ROR*1.5*19 2012-01-09 ============
  HasItems := True;
end;

procedure TCCRICD9SelectorTree.Clear;
begin
  inherited;
  HasItems := False;
end;

procedure TCCRICD9SelectorTree.DeleteGroup(aNode: PVirtualNode);
begin
  DeleteNode(aNode);
  if UpdateCount = 0 then
    HasItems := (GetFirstResultItem <> nil);
end;

procedure TCCRICD9SelectorTree.DeleteItem(aNode: PVirtualNode);
begin
  if Assigned(aNode) then
    begin
      DeleteNode(aNode);
      if UpdateCount = 0 then
        HasItems := (GetFirstResultItem <> nil);
    end;
end;

function TCCRICD9SelectorTree.DoDragOver(Source: TObject; Shift: TShiftState;
  State: TDragState; Pt: TPoint; Mode: TDropMode; var Effect: Integer): Boolean;
begin
  Result := inherited DoDragOver(Source, Shift, State, Pt, Mode, Effect);
  if Result and (ChildCount[RootNode] = 0) then
    Result := False;
end;

procedure TCCRICD9SelectorTree.EndUpdate;
begin
  inherited;
  HasItems := (GetFirstResultItem <> nil);
end;

function TCCRICD9SelectorTree.FindGroup(const aName: String): PVirtualNode;
begin
  Result := GetFirstChild(RootNode);
  while Assigned(Result) do
    if AnsiSameText(AsString[Result,idscName], aName) then
      Break
    else
      Result := GetNextSibling(Result);
end;

function TCCRICD9SelectorTree.GetFirstResultItem: PVirtualNode;
begin
  Result := GetFirst;
  while Assigned(Result) do
    if GetNodeLevel(Result) = idslItem then
      Break
    else
      Result := GetNext(Result);
end;

function TCCRICD9SelectorTree.GetGroup(aNode: PVirtualNode): PVirtualNode;
begin
  Result := aNode;
  while Assigned(Result) and (GetNodeLevel(Result) > idslGroup) do
    Result := NodeParent[Result];
end;

{$ENDIF}
end.
