unit uROR_DebugBox;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls;

type
  TPositions = (poTopLeft,poBottomLeft,poTopRight,poBottomRight);

  TCCRDebugBox = class(TComponent)
  private
    CCRDebugForm : TForm;
    DebugList    : TListBox;
    FPosition    : TPositions;
    FVisible     : Boolean;
    FWidth       : Integer;
    FHeight      : Integer;
    FCaption     : String;
    FStamp       : Boolean;
    procedure FixList;
    procedure SetPosition(A: TPositions);

    procedure SetVisible(A: Boolean);
    procedure SetWidth(A: Integer);
    procedure SetHeight(A: Integer);
    procedure SetCaption(A: String);
  protected
    { Protected declarations }
  public
    constructor Create(AOwner: TComponent); override;
    procedure Add(A: String);
    procedure Clear;
    procedure LoadLines(FName: String);
    procedure SaveLines(FName: String);
  published
    property Caption: String read FCaption write SetCaption;
    property Position: TPositions read FPosition write SetPosition default poTopRight;
    property Visible: Boolean read FVisible write SetVisible default True;
    property Width: Integer read FWidth write SetWidth default 400;
    property Height: Integer read FHeight write SetHeight default 400;
    property TimeStamp: Boolean read FStamp write FStamp default False;
  end;

implementation

constructor TCCRDebugBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPosition := poTopRight;
  FVisible := True;
  FWidth := 400;
  FHeight := 400;
  FCaption := 'CCR Debug Box';
  if not (csDesigning in ComponentState) then
  begin
      CCRDebugForm := TForm.Create(Application);
      with CCRDebugForm do
      begin
        Visible := FVisible;
        Caption := FCaption;
        FormStyle := fsStayOnTop;
        BorderStyle := bsSizeable;
        BorderIcons := [biSystemMenu];
      end;
      DebugList := TListBox.Create(CCRDebugForm);
      with DebugList do
      begin
        Parent := CCRDebugForm;
        Align := alClient;
        Sorted := False;
        Font.Name := 'Arial';
        Font.Size := 9;
      end;
  end;
end;

procedure TCCRDebugBox.SetPosition(A: TPositions);
begin
  FPosition := A;
  if not (csDesigning in ComponentState) then with CCRDebugForm do
    case A of
      poTopLeft     : SetBounds(0,0,Width,Height);
      poBottomLeft  : SetBounds(0,Screen.Height-Height,Width,Height);
      poTopRight    : SetBounds(Screen.Width-Width,0,Width,Height);
      poBottomRight : SetBounds(Screen.Width-Width,Screen.Height-Height,Width,Height);

    end;
end;

procedure TCCRDebugBox.SetVisible(A: Boolean);
begin
  FVisible := A;
  if not (csDesigning in ComponentState) then
  begin
    CCRDebugForm.Hide;
    if A then
    begin
      Width := Self.Width;
      Height := Self.Height;
      SetPosition(FPosition);
      CCRDebugForm.Show;
    end;
  end;
end;

procedure TCCRDebugBox.SetWidth(A: Integer);

begin
  FWidth := A;
  if not (csDesigning in ComponentState) then
  begin
    CCRDebugForm.Width := FWidth;
    SetPosition(FPosition);
  end;
end;

procedure TCCRDebugBox.SetHeight(A: Integer);
begin
  FHeight := A;
  if not (csDesigning in ComponentState) then
  begin
    CCRDebugForm.Height := FHeight;
    SetPosition(FPosition);
  end;
end;

procedure TCCRDebugBox.SetCaption(A: String);
begin
  FCaption := A;
  if not (csDesigning in ComponentState) then
    CCRDebugForm.Caption := FCaption;
end;

procedure TCCRDebugBox.Add(A: String);
begin
  if TimeStamp then
    DebugList.Items.Add(TimeToStr(Time)+' - '+A)
  else
    DebugList.Items.Add(A);
  FixList;
end;

procedure TCCRDebugBox.Clear;
begin
  {Remove all items from the list box}
  DebugList.Items.Clear;
  FixList;
end;

procedure TCCRDebugBox.FixList;
begin
  DebugList.ItemIndex := DebugList.Items.Count-1;
end;

procedure TCCRDebugBox.LoadLines(FName: String);
begin
  DebugList.Items.Clear;
  DebugList.Items.LoadFromFile(FName);
end;

procedure TCCRDebugBox.SaveLines(FName: String);
begin
  DebugList.Items.SaveToFile(FName);
end;

end.
