{******************************************************************************}
{ Package:      Clinical Case Registries Custom Components                     }
{ Date Created: November 18, 2004                                              }
{ Site Name:    Hines OIFO                                                     }
{ Developers:   Sergey Gavrilov                                                }
{ Description:  This unit defines base classes for custom CCR list and grid    }
{               views.                                                         }
{ Note:                                                                        }
{******************************************************************************}

unit uROR_CustomListView;

{$I Components.inc}

interface

uses
  ComCtrls, Controls, Classes, Graphics, Variants, uROR_Utilities, Messages,
  {$IFNDEF NOORPHEUS}OvcFiler,{$ENDIF}
  CommCtrl, Windows, uROR_Classes;

type
  TCCRCustomListView = class;

  {============================= TCCRFindStringMode ============================
    Overview:     Search mode flags for the FindString method of custom
                  list/grid views.
    SeeAlso:      TCCRCustomListView.FindString
  }
  TCCRFindStringMode = set of (

    fsmCase,      { Perform case sensitive search. }

    fsmInclusive, { Include the start index in the search. Otherwise, the
                    search starts from the next item (and/or ends on the
                    previous item. }

    fsmPartial,   { Search for a string that starts from the search pattern.
                    Otherwise, the search is performed for the whole value of
                    the search pattern. }

    fsmWrap       { Continue the search from the beginning of the list if
                    nothing has been found. Otherwise, the search is performed
                    from the start index to the end of the list. }

  );

  {============================= TCCRCustomListItem ============================
    Overview:     Base class for custom list/grid view items.
    SeeAlso:      TCCRCustomListItems; TCCRCustomListView
    Description:
      The TCCRCustomListItem abstract class encapsulates basic functionality of
      a custom list/grid view item. Each item contains the same number of data
      fields. By default, all fields are visible and shown "as is" in the columns
      of the list/grid view but descendant classes can modify this behavior.
  }
  TCCRCustomListItem = class(TListItem)
  private

    fUpdateLock: Integer;

    function  GetListView: TCCRCustomListView;
    function  GetStringValue(aColumnIndex: Integer): String;
    procedure SetStringValue(aColumnIndex: Integer; const aValue: String);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns internal field value and its default printable
                    representation.
      Keywords:     GetFieldValue,TCCRCustomListItem
      Description:
        The GetFieldValue is called by custom list/grid views to get internal
        field values of an item and their default printable counterparts.
        This method is abstract and must be implemented by descendant classes.
        <p>
        Zero-based index of the field is passed via the <i>aFieldIndex</i> input
        parameter. Internal value of the field is returned via the
        <i>anInternalValue</i> output parameter. Printable field value is
        returned as the function result.
    }
    function GetFieldValue(const aFieldIndex: Integer;
      var anInternalValue: Variant): String; virtual; abstract;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates if updates of the item printable representation
                    are disabled (locked).
      SeeAlso:      TCCRCustomListItem.BeginUpdate; TCCRCustomListItem.EndUpdate;
                    TCCRCustomListItem.StringValues;
                    TCCRCustomListItem.UpdateStringValues
      Description:
        BeginUpdate increments value of this property and EndUpdate decrements
        it. When the value reaches 0, the EndUpdate method calls the
        UpdateStringValues in order to update the StringValues property.
    }
    property UpdateLock: Integer read fUpdateLock;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Tells the item object that field values are going to be
                    changed.
      SeeAlso:      TCCRCustomListItem.EndUpdate
      Keywords:     BeginUpdate,TCCRCustomListItem
      Description:
        Call BeginUpdate before modifying several field values to increase
        efficiency and avoid flickering. Then call EndUpdate after all values
        have been updated to repaint the item.
    }
    procedure BeginUpdate; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Tells the item object that field values has been updated.
      SeeAlso:      TCCRCustomListItem.BeginUpdate
      Keywords:     EndUpdate,TCCRCustomListItem
      Description:
        Call BeginUpdate before modifying several field values to increase
        efficiency and avoid flickering. Then call EndUpdate after all values
        have been updated to repaint the item.
    }
    procedure EndUpdate; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Updates StringValues according to corresponding internal
                    field values.
      SeeAlso:      TCCRCustomListItem.StringValues; TCCRCustomListItem.EndUpdate
      Keywords:     UpdateStringValues,TCCRCustomListItem
      Description:
        UpdateStringValues method is called internally (e.g. by the EndUpdate)
        in order to update element(s) of the StringValues property with new
        formatted field value(s).
        <p>
        If a valid field index is passed as the value of the optional
        <i>aFieldIndex</i> parameter, then only the column that corresponds
        to this field is updated. Otherwise, the whole StringValues array is
        updated.
    }
    procedure UpdateStringValues(const aFieldIndex: Integer = -1); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates the list/grid view control that displays the item.
      SeeAlso:      TCCRCustomListView
      Keywords:     ListView,TCCRCustomListItem
      Description:
        Do not confuse the ListView that displays the item with the Owner of the
        item. The Owner is the TCCRCustomListItems object that manages the
        collection of all list items. The list/grid view is the Owner of that
        TCCRCustomListItems object.
    }
    property ListView: TCCRCustomListView  read GetListView;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Formatted text shown in the grid/list view columns.
      SeeAlso:      TListItem.Caption; TListItem.SubItems
      Description:
        StringValues property unifies access to the text exposed by the ancestor
        class (TListItem) to the grid/list views. StringValue[0] represents the
        Caption property of the item. Elements with positive indexes correspond
        to the elements of the SubItems property as follow: StringValue[1] =
        SubItems[0], ..., StringValue[Columns.Count-1] = SubItems[SubItems.Count-1].
        <p>
        In other words, elements of StringValues represent formatted printable
        values of visible fields (those that are associated with the columns of
        the list/grid view).
    }
    property StringValues[aColumnIndex: Integer]: String
      read GetStringValue  write SetStringValue;

  end;

  {============================ TCCRCustomListItems ============================
    Overview:     Base class for internal list/grid view item collections.
    SeeAlso:      TCCRCustomListItem; TCCRCustomListView
    Description:
      TCCRCustomListItems class provides typecast for the Item property of
      the base TListItems collection.
  }
  TCCRCustomListItems = class(TListItems)
  private

    function  GetItem(anIndex: Integer): TCCRCustomListItem;
    procedure SetItem(anIndex: Integer; const aValue: TCCRCustomListItem);

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Lists all items managed by the TCCRCustomListItems object.
      SeeAlso:      TCCRCustomListItem; TListItems.Count
      Keywords:     Item,TCCRCustomListItems
      Description:
        Use Item to directly access a list/grid item, given its position in the
        list/grid view. The first item has an index of 0, the second an index
        of 1, and so on.
        <p>
        Item is the default property. So, if ListItems references an instance
        of the TCCRCustomListItems, then you can access the first item either
        as ListItems.Item[0] or as ListItems[0].
    }
    property Item[anIndex: Integer]: TCCRCustomListItem
      read GetItem  write SetItem;  default;

  end;

  {============================= TCCRCustomListView ============================
    Overview:     Base class for custom list/grid views.
    SeeAlso:      TCCRCustomListItem; TCCRCustomListItems; TCCRGridView;
                  TCCRListView; TListColumn;
    Description:
      The TCCRCustomListView class encapsulates basic functionality of a custom
      list/grid view. It adds automatic sorting functionality to the base
      TCustomListView class. It also customizes appearance of disabled list/grid
      view controls.
      <p>
      Do not create instances of TCCRCustomListView. To put a list/grid view on
      a form, use a descendant of TCCRCustomListView, such as TCCRListView.
  }
  TCCRCustomListView = class(TCustomListView)
  private

    fColor:            TColor;
    fDefaultResources: TCCRSingleton;
    fSortDescending:   Boolean;
    fSortField:        Integer;

    function  GetItems: TCCRCustomListItems;
    procedure SetColor(const aColor: TColor);

  protected

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Column click event dispatcher.
      SeeAlso:      TCustomListView.ColumnClick
                    TCustomListView.ShowColumnHeaders
                    TCCRCustomListView.SortColumn;
                    TCCRCustomListView.SortDescending;
                    TCCRCustomListView.SortField
      Keywords:     ColClick,TCCRCustomListView
      Description:
        ColClick is called automatically when the user clicks on a column header
        if the ColumnClick and ShowColumnHeaders properties are true.
        TCCRCustomListView overrides this method and automatically sorts the
        list/grid view by the field that corresponds to the column identified
        by the <i>aColumn</i> parameter.
    }
    procedure ColClick(aColumn: TListColumn); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     OnCompare event handler.
      SeeAlso:      TCustomListView.OnCompare;
                    TCCRCustomListView.GetFieldValue;
                    TCCRCustomListView.SortDescending;
                    TCCRCustomListView.SortField;
      Keywords:     CompareItems,TCCRCustomListView
      Description:
        The CompareItems method is used internally as the OnCompare event
        handler. It is called by the AlphaSort method (the <i>Data</i> parameter
        is 0), or when the CustomSort method is called without a SortProc
        parameter (the <i>Data</i> parameter is the value of the LParam
        parameter of CustomSort). The CompareItems method compares the field
        values of list/grid items passed as the <i>Item1</i> and
        <i>Item2</i> parameters. Index of the field is defined by the SortField
        property.

        If the field value in <i>Item1</i> is the same as that in <i>Item2</i>,
        set the <i>Compare</i> parameter to 0.
        <p>
        If the field value in <i>Item1</i> is less than that in <i>Item2</i>,
        set the <i>Compare</i> parameter to a value less than 0.
        <p>
        If the field value in <i>Item1</i> is greater than that in <i>Item2</i>,
        set the <i>Compare</i> parameter to a value greater than 0.
        <p>
        Then, if the SortDescending property is True, negate the sign of the
        <i>Compare</i> parameter.

        By default, the VarCompareValue function is used to compare internal
        field values returned by the GetFieldValue method. Descendant classes
        can override this method to customize the sort order for the list/grid.
    }
    procedure CompareItems(Sender: TObject; Item1, Item2: TListItem;
      Data: Integer; var Compare: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates the window used by the list/grid view.
      SeeAlso:      TCustomListView.CreateParams; TCustomListView.CreateWnd
      Keywords:     CreateWnd,TCCRCustomListView
      Description:
        CreateWnd creates the list/grid views window, based on the parameter
        values provided by the CreateParams method. CreateWnd then sends
        messages to the newly created window so that it reflects the current
        property settings of the list view object. TCCRCustomListView overrides
        this method in order to change some window parameters.
    }
    procedure CreateWnd; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Responds to receiving input focus.
      SeeAlso:      TCustomListView.DoEnter; TCCRCustomListView.EnsureSelection
      Keywords:     DoEnter,TCCRCustomListView
      Description:
        DoEnter is called automatically when the control receives the input
        focus. TCCRCustomListView overrides this method in order to ensure that
        at least one item is selected if the list is not empty.
    }
    procedure DoEnter; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns index of the column identified by its header.
      SeeAlso:      TCCRCustomListView.FindColumnWidth
      Description:
        FindColumnIndex returns zero-based index of the column that corresponds
        to the header referenced by the <i>pHeader</i> parameter. This method
        is used by descendant classes during processing of notifications
        generated by the column headers (e.g. column resize).
    }
    function FindColumnIndex(pHeader: PNMHdr): Integer;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns width of the column (in pixels) identified by its
                    header.
      SeeAlso:      TCCRCustomListView.FindColumnIndex
      Description:
        FindColumnWidth returns width in pixels of the column that corresponds
        to the header referenced by the <i>pHeader</i> parameter. This method
        is used by descendant classes during processing of notifications
        generated by the column headers (e.g. column resize).
    }
    function FindColumnWidth(pHeader: PNMHdr): Integer;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the FieldsCount property.
      SeeAlso:      TCCRCustomListView.FieldsCount
      Description:
        GetFieldCount returns number of fields defined in a list/grid view. As
        implemented in TCCRCustomListView, it returns number of columns.
    }
    function GetFieldsCount: Integer; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns the value of the SortColumn property.
      SeeAlso:      TCCRCustomListView.SortColumn; TCCRCustomListView.SortField
      Keywords:     GetSortColumn,TCCRCustomListView
      Description:
        The GetSortColumn method returns zero-based index of the column that the
        list/grid view is sorted by. If the control is not sorted, then this
        method return a negative value.
        <p>
        By default, all fields have corresponding columns and this method
        returns the value of the SortField property. Descendant classes can
        override this method in order to implement different behavior (e.g. to
        accommodate hidden fields).
    }
    function GetSortColumn: Integer; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Generates an OnInsert event.
      SeeAlso:      TCustomListView.InsertItem; TCustomListView.OnInsert;
                    TCCRCustomListItem
      Keywords:     InsertItem,TCCRCustomListView
      Description:
        List/grid view objects call InsertItem internally when new items are
        added to the Items property list. TCCRCustomListView overrides this
        method in order to properly initialize new items.
        The <i>anItem</i> parameter can be safely typecasted to
        TCCRCustomListItem. The same is true for the OnInsert event handler.
    }
    procedure InsertItem(anItem: TListItem); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Initializes the control after it is loaded from a stream.
      SeeAlso:      TControl.Loaded
      Keywords:     Loaded,TCCRCustomListView
      Description:
        The VCL streaming system calls Loaded automatically after the controls
        form is loaded into memory so that the control can complete any
        initializations that depend on other objects in the form.
        TCCRCustomListView overrides this method in order to complete the
        initialization (colors and sorting).
    }
    procedure Loaded; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the Enabled property.
      SeeAlso:      TControl.Color; TControl.SetEnabled
      Keywords:     SetEnabled,TCCRCustomListView
      Description:
        SetEnabled is the protected write implementation of the Enabled property.
        TCCRCustomListView overrides this method in order to change the color of
        the control according to its state. Color of an enabled list/grid view
        is defined by the Color property. Disabling the control changes its
        color to that of its parent.
    }
    procedure SetEnabled(aValue: Boolean); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the SortColumn property.
      SeeAlso:      TCCRCustomListView.SortColumn
      Description:
        By default, all data fields have corresponding columns in the grid/list
        view, the column index is identical to the field index, and this method
        just assigns the value of the <i>aValue</i> parameter to the SortField
        property. Descendant classes can override this method in order to modify
        this behavior (e.g. by implementing hidden fields) and/or add additional
        post-processing.
    }
    procedure SetSortColumn(const aValue: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the SortDescending property.
      SeeAlso:      TCCRCustomListView.SortDescending
      Description:
        Descendant classes can override this method in order to add additional
        post-processing.
    }
    procedure SetSortDescending(const aValue: Boolean); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Sets the value of the SortField property.
      SeeAlso:      TCCRCustomListView.SortField
      Description:
        The SetSortField method assign zero-based index of a data field passed
        via the <i>aValue</i> parameter to the SortField property, calls the
        UpdateSortField method, and re-sorts the list/grid view.
        Descendant classes can override this method in order to add additional
        post-processing.
    }
    procedure SetSortField(const aValue: Integer); virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Performs post-processing after the sort order changes.
      SeeAlso:      TCCRCustomListView.SortColumn; TCCRCustomListView.SortField
      Description:
        The UpdateSortField method is called internally after a valid index of
        a different field is assigned to the SortField property or the
        SortDescending property is modified. Note that this method is not called
        if the list/grid view becomes unsorted (SortField = -1).
        <p>
        By default, the UpdateSortField shows appropriate sort order indicator
        on the column header that corresponds to the data field that the
        list/grid view is sorted by. If the field is not associated with a
        column (hidden field), then the sort indicator is hidden. Descendant
        classes can override this method in order to add additional
        post-processing.
    }
    procedure UpdateSortField; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Specifies the background color of the control.
      SeeAlso:      TControl.Color
      Keywords:     Color,TCCRCustomListView
      Description:
        Use Color to read or change the background color of the control.
        Disabled list/grid view control automatically changes color to that of
        its parent. You cannot change color of disabled control. If a new value
        is assigned to the Color property while the control is disabled, this
        value will be used only after the control is re-enabled.
    }
    property Color: TColor  read fColor  write SetColor  default clWindow;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates number of fields in a list/grid view.
      SeeAlso:      TCustomListView.GetFieldsCount
      Description:
        Use FieldsCount to get the number of fields in a list/grid view. By
        default, the number of fields is the same as the number of columns.
    }
    property FieldsCount: Integer  read GetFieldsCount;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Contains the list of items displayed by the list/grid view.
      SeeAlso:      TCustomListView.Items; TCCRCustomListItems
      Keywords:     Items,TCCRCustomListView
      Description:
        Use Items to directly access the TCCRCustomListItems objects that
        represent the items in the list/grid view. TCCRCustomListView redefines
        this property in order to typecast it to TCCRCustomListItems.
    }
    property Items: TCCRCustomListItems  read GetItems;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Index of the column that the list/grid view is sorted by.
      SeeAlso:      TCCRCustomListView.SortField
      Description:
        Read the SortColumn property to get zero-based index of the column that
        the list/grid view is sorted by. If the list/grid is not sorted, then
        value of this property is less than 0. Assign a valid column index to
        this property in order to sort the list/grid by the corresponding data
        field.
        <p>
        By default, all data fields have corresponding columns and this property
        has the same value as the SortField. Descendant classes can change this
        (e.g. by implementing hidden data fields).
    }
    property SortColumn: Integer  read GetSortColumn  write SetSortColumn;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Indicates descending sort order.
      SeeAlso:      TCCRCustomListView.SortColumn; TCCRCustomListView.SortField
      Description:
        If the list/grid view is sorted and this field is True, then the data
        is sorted in descending order (i.e. from 'z' to 'a', numbers from bigger
        to smaller, etc.). Toggle the value of this property in order to change
        the sort direction.
    }
    property SortDescending: Boolean
      read fSortDescending  write SetSortDescending  default False;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Index of the field that the list/grid view is sorted by.
      SeeAlso:      TCCRCustomListView.SortColumn
      Description:
        Read the SortField property to get zero-based index of the data field
        that the list/grid view is sorted by. If the list/grid is not sorted,
        then value of this property is less than 0. Assign a valid field index
        to this property in order to sort the list/grid by this field.
    }
    property SortField: Integer  read fSortField  write SetSortField  default -1;

  public

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Creates and initializes an instance of TCCRCustomListView.
      SeeAlso:      TComponent.Owner; TCustomListView.Create
      Keywords:     Create,TCCRCustomListView
      Description:
        The Create constructor assigns default values to the properties and
        loads required resources (e.g. images of the sort order indicator).
    }
    constructor Create(anOwner: TComponent); override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Destroys an instance of TCCRCustomListView.
      SeeAlso:      TCustomListView.Destroy; TObject.Free
      Keywords:     Destroy,TCCRCustomListView
      Description:
        Do not call Destroy directly in an application. Instead, call Free.
        Free verifies that the control is not nil, and only then calls Destroy.
        <p>
        Applications should only free controls explicitly when the constructor
        was called without assigning an owner to the control.
    }
    destructor Destroy; override;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Activates the control.
      SeeAlso:      TCCRCustomListView.EnsureSelection; TWinControl.SetFocus
      Keywords:     Activate,TCCRCustomListView
      Description:
        If the list/grid view is not empty, the Activate method gives the input
        focus to the control and calls the EnsureSelection method.
    }
    procedure Activate;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns column index for the data field.
      SeeAlso:      TCCRCustomListView.FieldIndex
      Keywords:     ColumnIndex,TCCRCustomListView
      Description:
        The ColumnIndex method returns the zero-based index of the column that
        is associated with the field defined by the zero-based index passed via
        the <i>aFieldIndex</i> parameter. If the parameter has an invalid value
        or the field does not have corresponding column (hidden field),
        a negative number is returned.
        <p>
        By default, all fields are associated with the list/grid view columns
        and the latter are shown in the same sequence. So, the ColumnIndex just
        returns the value of its parameter (if it is valid). Descendant classes
        can override this method and change this behavior (e.g. to implement
        hidden data fields).
    }
    function ColumnIndex(const aFieldIndex: Integer): Integer; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Ensures that at least one item is selected and has focus
                    (if the list/grid view is not empty).
      SeeAlso:      TCustomListView.ItemFocused; TCustomListView.ItemIndex
      Description:
        If the list/grid view is empty, the EnsureSelection method does nothing.
        Otherwise, if there is no selected item, the first item of the list
        is selected. The focus is moved to the selected item as well.
    }
    procedure EnsureSelection;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns field index for the column.
      SeeAlso:      TCCRCustomListView.ColumnIndex
      Keywords:     FieldIndex,TCCRCustomListView
      Description:
        The FieldIndex method returns the zero-based index of the data field
        that is associated with the column defined by the zero-based index
        passed via the <i>aColumnIndex</i> parameter. If the parameter has an
        invalid value, a negative number is returned.
        <p>
        By default, all fields are associated with the list/grid view columns
        and the latter are shown in the same sequence. So, the FieldIndex just
        returns the value of its parameter (if it is valid). Descendant classes
        can override this method and change this behavior (e.g. to implement
        hidden data fields).
    }
    function FieldIndex(const aColumnIndex: Integer;
      const SortableOnly: Boolean = False): Integer; virtual;

    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Returns a list/grid view item, if any, with the specified
                    string value of the data field.
      SeeAlso:      TCCRFindStringMode
      Description:
        Call FindString to search for a list/grid view item that has the value
        specified as the <i>aValue</i> parameter in the data field referenced
        by the <i>aFieldIndex</i> parameter. This method searches among default
        string representations of field values. If no item is found, it returns
        nil.
        <p>
        By default, the search starts from the item with the index defined by
        the value of the <i>aStartIndex</i> parameter. Optional <i>aMode</i>
        parameter can be used to customize the process.
    }
    function FindString(aValue: String; aFieldIndex: Integer;
      aStartIndex: Integer = 0;
      aMode: TCCRFindStringMode = [fsmInclusive]): TCCRCustomListItem;

    {$IFNDEF NOORPHEUS}
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Loads a list view layout.
      SeeAlso:      TCCRCustomListView.SaveLayout; TCCRVistAStore
      Keywords:     LoadLayout,TCCRCustomListView
      Description:
        Call LoadLayout to restore list view layout (saved by the SaveLayout)
        from the data storage referenced by the <i>aStorage</i> parameter.
        <i>aSection</i> specifies the data section name (a key of the registry,
        an XML node, a section of an ini file, etc.) where the layout is stored.
        <p>
        Override this method in descendant classes to implement appropriate
        functionality. As implemented in TCCRCustomListView, LoadLayout does
        nothing.
    }
    procedure LoadLayout(aStorage: TOvcAbstractStore;
      const aSection: String); virtual;
    {$ENDIF}

    {$IFNDEF NOORPHEUS}
    {- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
      Overview:     Saves a list view layout.
      SeeAlso:      TCCRCustomListView.LoadLayout; TCCRVistAStore
      Keywords:     SaveLayout,TCCRCustomListView
      Description:
        Call SaveLayout to save a list view layout (it can be later restored by
        the LoadLayout) to the data storage referenced by the <i>aStorage</i>
        parameter. <i>aSection</i> specifies the data section name (a key of
        the registry, an XML node, a section of an ini file, etc.) where the
        layout is saved to.
        <p>
        Override this method in descendant classes to implement appropriate
        functionality. As implemented in TCCRCustomListView, SaveLayout does
        nothing.
    }
    procedure SaveLayout(aStorage: TOvcAbstractStore;
      const aSection: String); virtual;
    {$ENDIF}

  end;

///////////////////////////////// Implementation \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

implementation

uses
  uROR_Resources, ImgList, SysUtils, StrUtils;

const
  imgAscending  = 0;
  imgDescending = 1;

type

  {======================== TCCRListViewDefaultResources =======================
    Overview:     Singleton for default images.
    Description:
      TCCRListViewDefaultResources handles a single copy of default images for
      list/grid views.
  }
  TCCRListViewDefaultResources = class(TCCRSingleton)

    fImages: TImageList;

  protected

    procedure Finalize; override;
    procedure Initialize; override;

  public

    property Images: TImageList  read fImages;

  end;

///////////////////////// TCCRListViewDefaultResources \\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRListViewDefaultResources.Finalize;
begin
  FreeAndNil(fImages);
  inherited;
end;

procedure TCCRListViewDefaultResources.Initialize;
begin
  inherited;
  fImages := TImageList.Create(nil);
  with fImages do
    begin
      ResInstLoad(HInstance, rtBitmap, 'SORT_ASCENDING',  clFuchsia);
      ResInstLoad(HInstance, rtBitmap, 'SORT_DESCENDING', clFuchsia);
    end;
end;

/////////////////////////////// TCCRCustomListItem \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TCCRCustomListItem.BeginUpdate;
begin
  if fUpdateLock = 0 then
    SubItems.BeginUpdate;
  Inc(fUpdateLock);
end;

procedure TCCRCustomListItem.EndUpdate;
begin
  if fUpdateLock > 0 then
    begin
      Dec(fUpdateLock);
      if fUpdateLock = 0 then
        begin
          UpdateStringValues;
          SubItems.EndUpdate;
        end;
    end;
end;

function TCCRCustomListItem.GetListView: TCCRCustomListView;
begin
  Result := inherited ListView as TCCRCustomListView;
end;

function TCCRCustomListItem.GetStringValue(aColumnIndex: Integer): String;
begin
  if aColumnIndex = 0 then
    Result := Caption
  else if (aColumnIndex > 0) and (aColumnIndex <= SubItems.Count) then
    Result := SubItems[aColumnIndex-1]
  else
    Result := '';
end;

procedure TCCRCustomListItem.SetStringValue(aColumnIndex: Integer; const aValue: String);
var
  i: Integer;
begin
  if aColumnIndex = 0 then
   Caption := aValue
  else if (aColumnIndex > 0) and (aColumnIndex < ListView.Columns.Count) then
    begin
      if aColumnIndex > SubItems.Count then
        for i:=ListView.Columns.Count-2 to SubItems.Count do
          SubItems.Add('');
      SubItems[aColumnIndex-1] := aValue;
    end;
end;

procedure TCCRCustomListItem.UpdateStringValues(const aFieldIndex: Integer);
var
  i, icol, ifld, n: Integer;
  iv: Variant;
begin
  if Assigned(Data) and (UpdateLock = 0) then
    if (aFieldIndex >= 0) and (aFieldIndex < ListView.FieldsCount) then
      begin
        icol := ListView.ColumnIndex(aFieldIndex);
        if icol >= 0 then
          StringValues[icol] := GetFieldValue(aFieldIndex, iv)
      end
    else
      begin
        SubItems.BeginUpdate;
        try
          n := ListView.Columns.Count - 1;
          for i:=0 to n do
            begin
              ifld := ListView.FieldIndex(i);
              StringValues[i] := GetFieldValue(ifld, iv);
            end;
        finally
          SubItems.EndUpdate;
        end;
      end;
end;

/////////////////////////////// TCCRCustomListItems \\\\\\\\\\\\\\\\\\\\\\\\\\\\

function TCCRCustomListItems.GetItem(anIndex: Integer): TCCRCustomListItem;
begin
  Result := inherited Item[anIndex] as TCCRCustomListItem;
end;

procedure TCCRCustomListItems.SetItem(anIndex: Integer; const aValue: TCCRCustomListItem);
begin
  inherited Item[anIndex] := aValue;
end;

/////////////////////////////// TCCRCustomListView \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TCCRCustomListView.Create(anOwner: TComponent);
begin
  inherited;

  BevelKind         := bkNone;
  GridLines         := True;
  OnCompare         := CompareItems;
  ParentColor       := False;
  ReadOnly          := False;
  TabStop           := True;

  fColor            := clWindow;
  fSortDescending   := False;
  fSortField        := -1;
end;

destructor TCCRCustomListView.Destroy;
begin
  FreeAndNil(fDefaultResources);
  inherited;
end;

procedure TCCRCustomListView.Activate;
begin
  if Items.Count > 0 then
    begin
      SetFocus;
      EnsureSelection;
    end;
end;

procedure TCCRCustomListView.ColClick(aColumn: TListColumn);
var
  fldNdx: Integer;
begin
  fldNdx := FieldIndex(aColumn.Index, True);
  if fldNdx >= 0 then
    if fldNdx <> SortField then
      SortField := fldNdx
    else
      SortDescending := not SortDescending;
  inherited;
end;

function TCCRCustomListView.ColumnIndex(const aFieldIndex: Integer): Integer;
begin
  if (aFieldIndex >= 0) and (aFieldIndex < FieldsCount) then
    Result := aFieldIndex
  else
    Result := -1;
end;

procedure TCCRCustomListView.CompareItems(Sender: TObject; Item1, Item2: TListItem;
  Data: Integer; var Compare: Integer);
var
  iv1, iv2: Variant;
begin
  Compare := 0;
  if (SortField < 0) or (SortField >= FieldsCount) then
    Exit;

  TCCRCustomListItem(Item1).GetFieldValue(SortField, iv1);
  TCCRCustomListItem(Item2).GetFieldValue(SortField, iv2);

  case VarCompareValue(iv1, iv2) of
    vrLessThan:    Compare := -1;
    vrGreaterThan: Compare := 1;
  end;

  if SortDescending then
    Compare := -Compare;
end;

procedure TCCRCustomListView.CreateWnd;
var
  wnd: HWND;
begin
  inherited;
  wnd := GetWindow(Handle, GW_CHILD);
  SetWindowLong(wnd, GWL_STYLE,
    GetWindowLong(wnd, GWL_STYLE) and not HDS_FULLDRAG);
end;

procedure TCCRCustomListView.DoEnter;
begin
  inherited;
  EnsureSelection;
end;

procedure TCCRCustomListView.EnsureSelection;
begin
  if (Items.Count > 0) and (ItemIndex < 0) then
    ItemIndex := 0;
  if ItemIndex >= 0 then
    ItemFocused := Items[ItemIndex];
end;

function TCCRCustomListView.FieldIndex(const aColumnIndex: Integer;
  const SortableOnly: Boolean): Integer;
begin
  if (aColumnIndex >= 0) and (aColumnIndex < Columns.Count) then
    Result := aColumnIndex
  else
    Result := -1;
end;

function TCCRCustomListView.FindColumnIndex(pHeader: PNMHdr): Integer;
var
  hwndHeader: HWND;
  iteminfo: THdItem;
  ItemIndex: Integer;
  buf: array [0..128] of Char;
begin
  Result := -1;
  hwndHeader := pHeader^.hwndFrom;
  ItemIndex := pHDNotify(pHeader)^.Item;
  FillChar(iteminfo, SizeOf(iteminfo), 0);
  iteminfo.Mask := HDI_TEXT;
  iteminfo.pszText := buf;
  iteminfo.cchTextMax := SizeOf(buf) - 1;
  Header_GetItem(hwndHeader, ItemIndex, iteminfo);
  if CompareStr(Columns[ItemIndex].Caption, iteminfo.pszText) = 0 then
    Result := ItemIndex
  else
    begin
      for ItemIndex := 0 to Columns.Count - 1 do
        if CompareStr(Columns[ItemIndex].Caption, iteminfo.pszText) = 0 then
          begin
            Result := ItemIndex;
            Break;
          end;
    end;
end;

function TCCRCustomListView.FindColumnWidth(pHeader: pNMHdr): Integer;
begin
  Result := -1;
  if Assigned(PHDNotify(pHeader)^.pItem) and
    ((PHDNotify(pHeader)^.pItem^.mask and HDI_WIDTH) <> 0) then
    Result := PHDNotify(pHeader)^.pItem^.cxy;
end;

function TCCRCustomListView.FindString(aValue: String; aFieldIndex: Integer;
  aStartIndex: Integer; aMode: TCCRFindStringMode): TCCRCustomListItem;
var
  n: Integer;
  iv: Variant;

  function find(aStartIndex, anEndIndex: Integer; aMode: TCCRFindStringMode): TCCRCustomListItem;
  var
    i: Integer;
    Comparator: function(const aSubText, aText: String): Boolean;
  begin
    Result := nil;

    if aStartIndex < 0 then
      aStartIndex := 0;
    if anEndIndex >= Items.Count then
      anEndIndex := Items.Count - 1;

    if fsmPartial in aMode then
      if fsmCase in aMode then
        Comparator := AnsiStartsStr
      else
        Comparator := AnsiStartsText
    else
      if fsmCase in aMode then
        Comparator := AnsiSameStr
      else
        Comparator := AnsiSameText;

    for i:=aStartIndex to anEndIndex do
      if Comparator(aValue, Items[i].GetFieldValue(aFieldIndex, iv)) then
      begin
        Result := Items[i];
        Break;
      end;
  end;

begin
  Result := nil;
  if Items.Count > 0 then
    begin
      n := Items.Count - 1;
      if fsmInclusive in aMode then
        Result := find(aStartIndex, n, aMode)
      else
        Result := find(aStartIndex+1, n, aMode);

      if (not Assigned(Result)) and (fsmWrap in aMode) then
        begin
          if fsmInclusive in aMode then
            n := aStartIndex - 1
          else
            n := aStartIndex;
          if n >= 0 then
            Result := find(0, n, aMode);
        end;
    end;
end;

function TCCRCustomListView.GetFieldsCount: Integer;
begin
  Result := Columns.Count;
end;

function TCCRCustomListView.GetItems: TCCRCustomListItems;
begin
  Result := inherited Items as TCCRCustomListItems;
end;

function TCCRCustomListView.GetSortColumn: Integer;
begin
  Result := ColumnIndex(SortField);
end;

procedure TCCRCustomListView.InsertItem(anItem: TListItem);
var
  i: Integer;
begin
  with anItem do
    begin
      ImageIndex := -1;
      Indent     := -1;
      SubItems.BeginUpdate;
      try
        for i:=2 to Columns.Count do
          SubItems.Add('');
      finally
        SubItems.EndUpdate;
      end;
    end;
  inherited;
end;

procedure TCCRCustomListView.Loaded;
begin
  inherited;

  if not Assigned(SmallImages) then
    begin
      fDefaultResources := TCCRListViewDefaultResources.Create;
      SmallImages := TCCRListViewDefaultResources(fDefaultResources).Images;
    end;

  if ParentColor then
    begin
      ParentColor := False;  // Enforce correct Parent Color
      ParentColor := True
    end;

  if (SortField >= 0) and (SortField < FieldsCount) then
    UpdateSortField
  else
    SortField := -1;
end;

{$IFNDEF NOORPHEUS}
procedure TCCRCustomListView.LoadLayout(aStorage: TOvcAbstractStore;
  const aSection: String);
begin
end;
{$ENDIF}

{$IFNDEF NOORPHEUS}
procedure TCCRCustomListView.SaveLayout(aStorage: TOvcAbstractStore;
  const aSection: String);
begin
end;
{$ENDIF}

procedure TCCRCustomListView.SetColor(const aColor: TColor);
begin
  if aColor <> fColor then
    begin
      fColor := aColor;
      if Enabled then
        inherited Color := aColor;
    end;
end;

procedure TCCRCustomListView.SetEnabled(aValue: Boolean);
begin
  if aValue <> Enabled then
    begin
      inherited;
      if Assigned(Parent) then
        if aValue then
          inherited Color := fColor
        else
          inherited Color := Parent.Brush.Color;
    end;
end;

procedure TCCRCustomListView.SetSortColumn(const aValue: Integer);
begin
  if (aValue <> SortColumn) and (aValue < Columns.Count) then
    SortField := FieldIndex(aValue);
end;

procedure TCCRCustomListView.SetSortDescending(const aValue: Boolean);
begin
  if aValue <> fSortDescending then
    begin
      fSortDescending := aValue;
      if SortField >= 0 then
        begin
          UpdateSortField;
          AlphaSort;
        end;
    end;
end;

procedure TCCRCustomListView.SetSortField(const aValue: Integer);
begin
  if aValue <> fSortField then
    if aValue >= 0 then
      begin
        if SortColumn >= 0 then
          Columns[SortColumn].ImageIndex := -1;
        if not (csLoading in ComponentState) then
          fSortDescending := False;
        fSortField := aValue;
        UpdateSortField;
        if SortType <> stBoth then
          SortType := stBoth
        else
          AlphaSort;
      end
    else
      begin
        if SortColumn >= 0 then
          Columns[SortColumn].ImageIndex := -1;
        fSortField := -1;
        SortType   := stNone;
      end;
end;

procedure TCCRCustomListView.UpdateSortField;
begin
  if (SortColumn >= 0) and (SortColumn < Columns.Count) then
    begin
      if SortDescending then
        Columns[SortColumn].ImageIndex := imgDescending
      else
        Columns[SortColumn].ImageIndex := imgAscending;
    end;
end;

end.
