unit fZZ_EventLog;
{
================================================================================
*
*       Package:        development tool
*       Date Created:   5/08/12
*       Site:           Hines OIFO
*       Developers:
*                       PII                       
*
*       Description:    Event log  presentation window
*
================================================================================
}

interface

uses
  Windows, Messages, SysUtils
  , Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, ActnList, Menus, StdActns, Buttons, ImgList;

type
  TZZ_EventLog = class(TObject)
  private
    fLogLimit: Integer;
    fEventLog: TStringList;
    fChangeTracker: TNotifyEvent;
    procedure setChangeTracker(aTracker: TNotifyEvent);
    procedure setLogLimit(aLimit: Integer);
  public
    property ChangeTracker: TNotifyEvent read fChangeTracker write setChangeTracker;
    property EventLog: TStringList read fEventLog write fEventLog;
    property LogLimit: Integer read fLogLimit write setLogLimit;
    constructor Create;
    destructor Destroy; override;
  end;

  TfrmEventLog = class(TForm)
    pnlTimeline: TPanel;
    pnlMainTitle: TPanel;
    pnlBottom: TPanel;
    lbLog: TListBox;
    Splitter1: TSplitter;
    Panel1: TPanel;
    mm: TMemo;
    pnlTitle: TPanel;
    ActionList1: TActionList;
    acRefreshLog: TAction;
    MainMenu1: TMainMenu;
    WindowClose1: TWindowClose;
    File1: TMenuItem;
    Refresh1: TMenuItem;
    N1: TMenuItem;
    Close1: TMenuItem;
    SpeedButton1: TSpeedButton;
    acSaveAs: TAction;
    SpeedButton2: TSpeedButton;
    ImageList1: TImageList;
    Panel2: TPanel;
    OpenDialog1: TOpenDialog;
    acSetLimit: TAction;
    acClear: TAction;
    SpeedButton5: TSpeedButton;
    SpeedButton6: TSpeedButton;
    ckbWrap: TCheckBox;
    acWrap: TAction;
    pnlBottomDetails: TPanel;
    pnlLimit: TPanel;
    acLoad: TAction;
    SpeedButton3: TSpeedButton;
    edHigh: TEdit;
    procedure lbLogClick(Sender: TObject);
    procedure acRefreshLogExecute(Sender: TObject);
    procedure Close1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure acSaveAsExecute(Sender: TObject);
    procedure acSetLimitExecute(Sender: TObject);
    procedure acClearExecute(Sender: TObject);
    procedure acWrapExecute(Sender: TObject);
    procedure acLoadExecute(Sender: TObject);
    procedure lbLogDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure edHighChange(Sender: TObject);
  private
    { Private declarations }
    fEventLog: TStringList;
    Current: TObject;
    function findObject(anObject: TObject): Integer;
    procedure restorePosition;
    procedure RefreshLog;
    procedure ShowLogSizeInfo;
  public
    { Public declarations }
    procedure UpdateLog(aLog:TStringList=nil);
    procedure EventLogPresenterOnChange(Sender:TObject);
    procedure setViewMode(aParented:Boolean=false);
  end;

var
  frmEventLog: TfrmEventLog;

function newEventLog: TZZ_EventLog;
function addEvent(aDescription:String;anEvent:TObject):Boolean;
function addRPCEvent(anEvent:TObject): Boolean;
function addTracker(aTracker: TNotifyEvent):Boolean;

procedure ShowEventLog(aModal: Boolean=True);
function setParentedLog(aParent:TWinControl):Boolean;

implementation

uses
  uZZ_DescribedItem, uZZ_RPCEvent
  ;

{$R *.dfm}

var
  EL: TZZ_EventLog;

////////////////////////////////////////////////////////////////////////////////
function addRPCEvent(anEvent:TObject): Boolean;
var
  sDescription: String;
begin
  Result := False;
  if not Assigned(EL) then
    begin
      ShowMessage(TRPCEventItem(anEvent).RPC);
      FreeAndNil(anEvent);
      Exit;
    end;
  sDescription :=
    Format(' %-25.25s',[TRPCEventItem(anEvent).RPC]) +
    FormatDateTime(' hh:mm:ss.zzz', TRPCEventItem(anEvent).Start);
  EL.EventLog.InsertObject(0,sDescription,anEvent);
  Result := true;
end;


function addEvent(aDescription:String;anEvent:TObject): Boolean;
begin
  Result := False;
  if not Assigned(EL) then
    Exit;
  EL.EventLog.InsertObject(0,aDescription,anEvent);
  Result := true;
end;

function addTracker(aTracker: TNotifyEvent):Boolean;
begin
  Result := False;
  if not Assigned(EL) then
    Exit;
  EL.ChangeTracker := aTracker;
  Result := true;
end;

function newEventLog: TZZ_EventLog;
begin
  if not Assigned(EL) then
    EL := TZZ_EventLog.Create;
  result := EL;
end;


////////////////////////////////////////////////////////////////////////////////
procedure ShowEventLog(aModal: Boolean=True);
begin
  if not Assigned(frmEventLog) then
    Application.CreateForm(TfrmEventLog,frmEventLog);
  if not Assigned(frmEventLog) then Exit;

  frmEventLog.UpdateLog(EL.EventLog);
  frmEventLog.RefreshLog;

  if frmEventLog.Parent = nil  then
    if aModal then
      frmEventLog.ShowModal
    else
      frmEventLog.Show;

  frmEventLog.Left := 0;
  frmEventLog.Top := 0;
end;

procedure setFormParented(aForm: TForm; aParent: TWinControl);
begin
  if aForm.Parent <> aParent then
    begin
      aForm.BorderStyle := bsNone;
      aForm.Parent := aParent;
      aForm.Align := alClient;
      aForm.Menu := nil;
      aForm.Show;
    end;
end;

function setParentedLog(aParent:TWinCOntrol): Boolean;
begin
  Result := False;
  if Assigned(frmEventLog) then
    begin
      setFormParented(frmEventLog,aParent);
      frmEventLog.setViewMode(True);
      Result := True;
    end;
end;

////////////////////////////////////////////////////////////////////////////////

constructor TZZ_EventLog.Create;
begin
  inherited;
  fEventLog := TStringList.Create;
  fLogLimit := 500;
end;

destructor TZZ_EventLog.Destroy;
begin
  EventLog.OnChange := nil;
  if Assigned(EventLog) then
    while EventLog.Count > 0 do
      begin
        if Assigned(EventLog.Objects[0]) then
          try
            TObject(EventLog.Objects[0]).Free;
          except
          end;
        EventLog.Delete(0);
      end;
end;

procedure TZZ_EventLog.setChangeTracker(aTracker: TNotifyEvent);
begin
  fChangeTracker := aTracker;
  EventLog.OnChange := aTracker;
end;

procedure TZZ_EventLog.setLogLimit(aLimit: Integer);
begin
  if aLimit <= fLogLimit then
    begin
      while EventLog.Count > aLimit do
      begin
        if EventLog.Objects[EventLog.Count - 1] <> nil then
          EventLog.Objects[EventLog.Count - 1].Free;
        EventLog.Delete(EventLog.Count - 1);
      end;
    end;
  fLogLimit := aLimit;
end;

////////////////////////////////////////////////////////////////////////////////

procedure TfrmEventLog.FormCreate(Sender: TObject);
begin
  AddTracker(EventLogPresenterOnChange);
end;

procedure TfrmEventLog.RefreshLog;
begin
  if not Assigned(fEventLog) then
    exit;
  lbLog.Enabled := False;
  EL.LogLimit := EL.LogLimit;
  lbLog.Items.Assign(fEventLog);
  RestorePosition;
  lbLog.Enabled := True;
  Application.ProcessMessages;
end;

procedure TfrmEventLog.lbLogClick(Sender: TObject);
var
  X: ImDescribe;
begin
  if lbLog.ItemIndex >= 0 then
    begin
      Current := lbLog.Items.Objects[lbLog.ItemIndex];

      if Assigned(Current) and Current.GetInterface(ImDescribe,X) then
        mm.Text := X.getDescription.Text
      else
        begin
          mm.Text := '';
          pnlTitle.Caption := '';
        end;
    end
  else
    begin
      mm.Text := '';
      pnltitle.Caption := '';
      Current := nil;
    end;
end;

function TfrmEventLog.findObject(anObject:TObject): Integer;
var
  i: Integer;
begin
  Result := -1;
  for i := 0 to lbLog.Items.Count - 1 do
    begin
      if anObject = lbLog.Items.Objects[i] then
        begin
          Result := i;
          break;
        end;
    end;
end;

procedure TfrmEventLog.restorePosition;
var
  i: Integer;
begin
  if not Assigned(Current) then
    Exit;
  i := findObject(Current);
  if i >=0 then
    lbLog.ItemIndex := i;
end;

procedure TfrmEventLog.Close1Click(Sender: TObject);
begin
  Close;
end;

procedure TfrmEventLog.ShowLogSizeInfo;
begin
  if assigned(fEventLog) then
    pnlBottom.Caption := Format(' %d ',[fEventLog.Count]);
  if assigned(EL) then
    pnlLimit.Caption := Format('(%d) ',[EL.LogLimit]);
end;

procedure TfrmEventLog.UpdateLog(aLog:TStringList=nil);
begin
  if Assigned(aLog) then
    fEventLog := aLog;
  RefreshLog;
  lbLog.ItemIndex := 0;
  lbLogClick(nil);
  ShowLogSizeInfo;
  Application.ProcessMessages;
end;

procedure TfrmEventLog.acRefreshLogExecute(Sender: TObject);
begin
  UpdateLog(fEventLog);
end;

procedure TfrmEventLog.EventLogPresenterOnChange(Sender:TObject);
begin
  UpdateLog(TStringList(Sender));
end;

procedure TfrmEventLog.setViewMode(aParented:Boolean=false);
begin
  pnlTitle.ParentColor := aParented;
  pnlMainTitle.ParentColor := aParented;
  pnlBottom.ParentColor := aParented;
end;

procedure TfrmEventLog.acSaveAsExecute(Sender: TObject);
var
  i: Integer;
  s: String;
  sN: String;
  X: ImDescribe;
  SL : TStringList;
begin
  if OpenDialog1.Execute then
    begin
      sN := OpenDialog1.FileName;
      for i := lbLog.Items.Count -1 downto 0 do
        begin
          Current := lbLog.Items.Objects[i];
          if Assigned(Current) and Current.GetInterface(ImDescribe,X) then
            s := s + '--'+lbLog.Items[i] + #13#10 + X.getDescription.Text + #13#10
          else
            s := s + #13#10
        end;
      SL := TStringList.Create;
      SL.Text := s;
      SL.SaveToFile(sN);
      SL.Free;
    end;
end;

procedure TfrmEventLog.acSetLimitExecute(Sender: TObject);
var
  InputString: string;
begin
  InputString:= InputBox('Log Size Limit', 'Enter # of records to keep in the Log', IntToStr(EL.LogLimit));
  if InputString <> IntToStr(EL.LogLimit) then
    begin
      EL.LogLimit := StrToIntDef(InputString,EL.LogLimit);
      ShowLogSizeInfo;
    end;
end;

procedure TfrmEventLog.acClearExecute(Sender: TObject);
begin
  if MessageDlg('Erase Log now?',
    mtConfirmation, [mbYes, mbNo], 0) = mrYes then
    begin
      lbLog.Enabled := False;
      while fEventLog.Count > 0 do
      begin
        if fEventLog.Objects[fEventLog.Count - 1] <> nil then
          fEventLog.Objects[fEventLog.Count - 1].Free;
        fEventLog.Delete(fEventLog.Count - 1);
      end;
      lbLog.Enabled := True;
    end;
  Application.ProcessMessages;
end;

procedure TfrmEventLog.acWrapExecute(Sender: TObject);
begin
  mm.WordWrap := ckbWrap.Checked;
  if ckbWrap.Checked then
    mm.ScrollBars := ssVertical
  else
    mm.ScrollBars := ssBoth;
  pnlBottomDetails.Visible := ckbWrap.Checked;
end;

procedure TfrmEventLog.acLoadExecute(Sender: TObject);
var
  sEventName,sEventDescription,
  s, sN: String;
  i, iCount: integer;
  SL: TStringList;

  procedure LogEvent;
  begin
    if sEventName+sEventDescription <> '' then
      AddEvent(sEventName,getDescribedItem(sEventName,sEventDescription));
  end;

begin
  if OpenDialog1.Execute then
    begin
      sN := OpenDialog1.FileName;
      SL := TStringList.Create;
      SL.LoadFromFile(sN);
      i := 0;
      iCount := 0;
      sEventName := '-- File Import -- BEGIN';
      sEventDescription := 'File: '+sN;
      while i < SL.Count do
        begin
          s := SL[i];
          if pos('--',s)=1 then
            begin
              LogEvent;
              sEventName := '';
              sEventDescription := '';
              sEventName := copy(s,3,Length(s));
              inc(iCount);
            end
          else
            sEventDescription := sEventDescription + s + #13#10;
          inc(i);
        end;
      sEventName := '-- File Import -- END';
      sEventDescription := 'Total lines: '+IntToStr(SL.Count)+ #13#10+
        'Total events: ' + IntToStr(iCount-1);
      LogEvent;
      SL.Free;
    end;
end;

procedure TfrmEventLog.lbLogDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
var
  s: String;
begin
  inherited;
  s := edHigh.Text;
  with (Control as TListBox).Canvas do
    begin
      if (Length(s)>0) and (pos(s,(Control as TListBox).Items[Index])>0) then
        Font.Color := clRed
      else if (odSelected in State) then
        Font.Color := clWhite
      else
        Font.Color := clWindowtext;
      FillRect(Rect);
      TextOut(Rect.Left, Rect.Top, (Control as TListBox).Items[Index]);
      if odFocused In State then
        begin
          Brush.Color := lbLog.Color;
          DrawFocusRect(Rect);
        end;
    end;
end;

procedure TfrmEventLog.edHighChange(Sender: TObject);
begin
  inherited;
  lbLog.Invalidate;
  Application.ProcessMessages;
end;

initialization
  newEventLog;

finalization
  FreeAndNil(EL);
end.

