unit uWordSpell;

////////////////////////////////////////////////////////////////////////////////
//
// No Thread version of the uSpell.pas unit.
// Uses Word for spell and grammar checking.
//
//
////////////////////////////////////////////////////////////////////////////////

interface

uses
  Word2010
  , System.Classes
  ,Windows
  ,Messages
  , Vcl.StdCtrls
  ;

const
  UM_SPELLCHECK_OVER = WM_USER + 401;

var
  _Word: WordApplication;
  _Doc: WordDocument;
  _WordVersion: single;
  _DocDlg: OleVariant;
  _Dialog: OleVariant;
  _WordSettings: TList;
  _DocWindowHandle: HWnd;
  _Title: String;

  _NullStr: OleVariant;
  _FalseVar: OleVariant;

// Spell Check steps
function WordStart:WordApplication;
function WordCreateDocument:WordDocument;
procedure WordCreateDialogs;
function WordDoCheck(var aText:String; aSpell:Boolean=True):Boolean;
procedure WordSettingsSave;
procedure WordSettingsRestore;
procedure WordConfigWord;
procedure WordConfigDoc;
procedure WordGetDialogs;
procedure WordUserSettingsLoad;
procedure WordUserSettingsSave;
procedure WordExit;
procedure WordToNil;
procedure WordDocumentNil;

// Utility
function WordFindDocumentWindow: HWnd;

// Spell check
function WordSpellCheck(var aText:String;aSpell:Boolean = true): Boolean;

// Compatibility with original uSpell
procedure SpellCheckForControl(aMemo: TCustomMemo);
procedure GrammarCheckForControl(aMemo: TCustomMemo);

implementation


uses
  SysUtils,
  Dialogs,
  WinApi.ActiveX,
  Forms,
  uLog, fWordNotify;

var
  SpellCheckerSettings: string = '';


const
  TX_WINDOW_TITLE       = {'CPRS-Chart} 'Spell Checking #';

  RETRY_MAX = 3;

  usCheckSpellingAsYouType          = 1;
  usCheckGrammarAsYouType           = 2;
  usIgnoreInternetAndFileAddresses  = 3;
  usIgnoreMixedDigits               = 4;
  usIgnoreUppercase                 = 5;
  usCheckGrammarWithSpelling        = 6;
  usShowReadabilityStatistics       = 7;
  usSuggestFromMainDictionaryOnly   = 8;
  usSuggestSpellingCorrections      = 9;
  usHideSpellingErrors              = 10;
  usHideGrammarErrors               = 11;

  sTrueCode   = 'T';
  sFalseCode  = 'F';

    // AFAYT = AutoFormatAsYouType
  wsAFAYTApplyBorders             = 0;
  wsAFAYTApplyBulletedLists       = 1;
  wsAFAYTApplyFirstIndents        = 2;
  wsAFAYTApplyHeadings            = 3;
  wsAFAYTApplyNumberedLists       = 4;
  wsAFAYTApplyTables              = 5;
  wsAFAYTAutoLetterWizard         = 6;
  wsAFAYTDefineStyles             = 7;
  wsAFAYTFormatListItemBeginning  = 8;
  wsAFAYTInsertClosings           = 9;
  wsAFAYTReplaceQuotes            = 10;
  wsAFAYTReplaceFractions         = 11;
  wsAFAYTReplaceHyperlinks        = 12;
  wsAFAYTReplaceOrdinals          = 13;
  wsAFAYTReplacePlainTextEmphasis = 14;
  wsAFAYTReplaceSymbols           = 15;
  wsAutoFormatReplaceQuotes       = 16;
  wsTabIndentKey                  = 17;
  wsWindowState                   = 18;
  wsSaveInterval                  = 19;
  wsTrackRevisions                = 20;
  wsShowRevisions                 = 21;
  wsShowSummary                   = 22; // not used for Word 2010
////////////////////////////////////////////////////////////////////////////////

function WordSpellCheckInit: WordApplication;
begin
  if _Word <> nil then
    Result := _Word
  else
    begin
      WordNotifyShow;
      Result := WordStart;
      if Result = nil then
        exit;
      WordCreateDocument;
      WordSettingsSave;
      WordConfigWord;
      WordConfigDoc;
      WordCreateDialogs;
      WordUserSettingsLoad;
      WordNotifyHide;
    end;
end;

function WordSpellCheckClose: WordApplication;
begin
  if _Word <> nil then
    begin
      WordUserSettingsSave;
      WordSettingsRestore;
      WordExit;
      WordDocumentNil;
    end;
  Result := _Word;
end;


function WordStart:WordApplication;
var
  s: String;
begin
  LogEnterExit('WordStart');
  if _Word <> nil then
      s := 'WORD ' + _Word.Version + ' Already started'
  else
    try
      Result := CoWordApplication.Create;
      _Word := Result;
      _Word.Caption := _Title;
      s := _Word.Version;
      _WordVersion := StrToFloatDef(_Word.Version, 0.0);
    except
      on E: Exception do
         s := E.Message;
    end;
  LogLine('Version: '+s);
  LogLine('Caption: '+_Title);

  _DocWindowHandle := WordFindDocumentWindow;
  LogEnterExit('WordStart', true);
end;

function WordCreateDocument:WordDocument;
var
  DocType: OleVariant;
begin
  LogEnterExit('WordCreateDocument');
  VariantInit(DocType);
  try
    DocType := wdNewBlankDocument;
    Result := _Word.Documents.Add(_NullStr, _FalseVar, DocType, _FalseVar);
    Result.Activate;
    _Doc := Result;
  finally
    VarClear(DocType);
  end;
  LogEnterExit('WordCreateDocument', true);
end;

procedure WordCreateDialogs;
begin
  LogEnterExit('WordCreateDialogs');
  VariantInit(_Dialog);
  _Dialog := _Word.Dialogs.Item(wdDialogToolsOptionsSpellingAndGrammar);
  VariantInit(_DocDlg);
  _DocDlg := _Word.ActiveDocument;
  LogEnterExit('WordCreateDialogs', true);
end;

function WordDoCheck(var aText:String; aSpell:Boolean=True):Boolean;
begin
  LogEnterExit('WordDoCheck');
  LogLine(#13#10+aText);

  WordSpellCheckInit;
  if _Word <> nil then
    begin
      _Doc.Content.InsertAfter(aText); //_Doc.Content.Text := aText;
      _Doc.Content.SpellingChecked := False;
      _Doc.Content.GrammarChecked := False;

      SetForegroundWindow(_DocWindowHandle);
      SetFocus(_DocWindowHandle);

      if aSpell then
      begin
        _DocDlg.Content.CheckSpelling;
        Result := _Doc.Content.SpellingChecked;
        aText := _Doc.Content.Text;
      end
      else
      begin
        _Doc.Content.CheckGrammar;
        Result := _Doc.Content.GrammarChecked;
        aText := _Doc.Content.Text;
      end;
    end
  else
    Result := False;
  LogEnterExit('WordDoCheck', true);
  SendMessage(Application.MainForm.Handle,UM_SPELLCHECK_OVER,0,0);
end;

procedure WordExit;
var
  Save: OleVariant;
  Doc: OleVariant;

begin
  LogEnterExit('WordExit');
  if _Word = nil then
     ShowMessage('No WORD Application available')
  else
    begin
      VarClear(_Dialog);
      VarClear(_DocDlg);
      VariantInit(Save);
      VariantInit(Doc);
      try
        Save := wdDoNotSaveChanges;
        Doc := wdWordDocument;
        _Word.Quit(Save, Doc, _FalseVar);
        _Word := nil;
      finally
        VarClear(Save);
        VarClear(Doc);
      end;
    end;
  LogEnterExit('WordExit', true);
end;

procedure WordConfigDoc;
begin
  LogEnterExit('WordConfigDoc');
  _Doc.TrackRevisions        := False;
  _Doc.ShowRevisions         := False;
  if (_WordVersion < 13) then            // altered for Word 2010
    _Doc.ShowSummary         := False;
  _Word.Height               := 1000;
  _Word.Width                := 1000;
  _Word.Top                  := -2000;
  _Word.Left                 := -2000;
  LogEnterExit('WordConfigDoc', true);
end;

procedure WordConfigWord;
begin
  LogEnterExit('WordConfigWord');
// save all old values to FWord, restore when done.
  _Word.Options.AutoFormatAsYouTypeApplyBorders             := False;
  _Word.Options.AutoFormatAsYouTypeApplyBulletedLists       := False;
  _Word.Options.AutoFormatAsYouTypeApplyFirstIndents        := False;
  _Word.Options.AutoFormatAsYouTypeApplyHeadings            := False;
  _Word.Options.AutoFormatAsYouTypeApplyNumberedLists       := False;
  _Word.Options.AutoFormatAsYouTypeApplyTables              := False;
  _Word.Options.AutoFormatAsYouTypeAutoLetterWizard         := False;
  _Word.Options.AutoFormatAsYouTypeDefineStyles             := False;
  _Word.Options.AutoFormatAsYouTypeFormatListItemBeginning  := False;
  _Word.Options.AutoFormatAsYouTypeInsertClosings           := False;
  _Word.Options.AutoFormatAsYouTypeReplaceQuotes            := False;
  _Word.Options.AutoFormatAsYouTypeReplaceFractions         := False;
  _Word.Options.AutoFormatAsYouTypeReplaceHyperlinks        := False;
  _Word.Options.AutoFormatAsYouTypeReplaceOrdinals          := False;
  _Word.Options.AutoFormatAsYouTypeReplacePlainTextEmphasis := False;
  _Word.Options.AutoFormatAsYouTypeReplaceSymbols           := False;
  _Word.Options.AutoFormatReplaceQuotes                     := False;
  _Word.Options.TabIndentKey                                := False;
  _Word.WindowState                                         := wdWindowStateNormal;
  _Word.Options.SaveInterval                                := 0;
  _Word.ResetIgnoreAll;

  LogEnterExit('WordConfigWord', true);
end;

procedure WordSettingsSave;

  procedure Save(Value, Index: integer);
  begin
    while _WordSettings.Count <= Index do
      _WordSettings.Add(nil);
    _WordSettings[Index] := Pointer(Value);
  end;

begin
  LogEnterExit('WordSettingsSave');

  Save(Ord(_Word.Options.AutoFormatAsYouTypeApplyBorders)             , wsAFAYTApplyBorders);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeApplyBulletedLists)       , wsAFAYTApplyBulletedLists);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeApplyFirstIndents)        , wsAFAYTApplyFirstIndents);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeApplyHeadings)            , wsAFAYTApplyHeadings);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeApplyNumberedLists)       , wsAFAYTApplyNumberedLists);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeApplyTables)              , wsAFAYTApplyTables);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeAutoLetterWizard)         , wsAFAYTAutoLetterWizard);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeDefineStyles)             , wsAFAYTDefineStyles);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeFormatListItemBeginning)  , wsAFAYTFormatListItemBeginning);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeInsertClosings)           , wsAFAYTInsertClosings);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeReplaceQuotes)            , wsAFAYTReplaceQuotes);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeReplaceFractions)         , wsAFAYTReplaceFractions);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeReplaceHyperlinks)        , wsAFAYTReplaceHyperlinks);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeReplaceOrdinals)          , wsAFAYTReplaceOrdinals);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeReplacePlainTextEmphasis) , wsAFAYTReplacePlainTextEmphasis);
  Save(Ord(_Word.Options.AutoFormatAsYouTypeReplaceSymbols)           , wsAFAYTReplaceSymbols);
  Save(Ord(_Word.Options.AutoFormatReplaceQuotes)                     , wsAutoFormatReplaceQuotes);
  Save(Ord(_Word.Options.TabIndentKey)                                , wsTabIndentKey);
  Save(Ord(_Word.WindowState)                                         , wsWindowState);
  Save(Ord(_Word.Options.SaveInterval)                                , wsSaveInterval);
  Save(Ord(_Doc.TrackRevisions)                                       , wsTrackRevisions);
  Save(Ord(_Doc.ShowRevisions)                                        , wsShowRevisions);
  if (_WordVersion < 13) then                                         // altered for Word 2010
    Save(Ord(_Doc.ShowSummary)                                        , wsShowSummary);

  LogEnterExit('WordSettingsSave', true);
end;

procedure WordGetDialogs;
begin
  LogEnterExit('WordGetDialogs');
    VariantInit(_Dialog);
    _Dialog := _Word.Dialogs.Item(wdDialogToolsOptionsSpellingAndGrammar);
    VariantInit(_DocDlg);
    _DocDlg := _Word.ActiveDocument;
  LogEnterExit('WordGetDialogs', true);
end;

procedure WordUserSettingsLoad;

  function UserSetting(Index: integer): boolean;
  begin
    if SpellCheckerSettings = '' then
    begin
      case Index of
        usCheckSpellingAsYouType:         Result := True;
        usCheckGrammarAsYouType:          Result := False;
        usIgnoreInternetAndFileAddresses: Result := True;
        usIgnoreMixedDigits:              Result := True;
        usIgnoreUppercase:                Result := True;
        usCheckGrammarWithSpelling:       Result := False;
        usShowReadabilityStatistics:      Result := False;
        usSuggestFromMainDictionaryOnly:  Result := False;
        usSuggestSpellingCorrections:     Result := True;
        usHideSpellingErrors:             Result := False;
        usHideGrammarErrors:              Result := True;
        else                              Result := False;
      end;
    end
    else
      Result := copy(SpellCheckerSettings,Index,1) = sTrueCode;
  end;

begin
  LogEnterExit('WordUserSettingsLoad');
  try
    _Dialog.AutomaticSpellChecking   := UserSetting(usCheckSpellingAsYouType);
    _Dialog.AutomaticGrammarChecking := UserSetting(usCheckGrammarAsYouType);
    _Dialog.FilenamesEmailAliases    := UserSetting(usIgnoreInternetAndFileAddresses);
    _Dialog.IgnoreMixedDigits        := UserSetting(usIgnoreMixedDigits);
    _Dialog.ForegroundGrammar        := UserSetting(usCheckGrammarWithSpelling);
    _Dialog.ShowStatistics           := UserSetting(usShowReadabilityStatistics);
    _Dialog.SuggestFromMainDictOnly  := UserSetting(usSuggestFromMainDictionaryOnly);
    _Dialog.IgnoreAllCaps            := UserSetting(usIgnoreUppercase);
    _Dialog.AlwaysSuggest            := UserSetting(usSuggestSpellingCorrections);
    _Dialog.HideSpellingErrors       := UserSetting(usHideSpellingErrors);
    _Dialog.HideGrammarErrors        := UserSetting(usHideGrammarErrors);
    _Dialog.Execute;
  finally
  end;
  LogEnterExit('WordUserSettingsLoad', true);
end;

procedure WordUserSettingsSave;

  procedure SaveSetting(Value: boolean; Index: integer);
  begin
    while length(SpellCheckerSettings) < Index do
      SpellCheckerSettings := SpellCheckerSettings + ' ';
    if Value then
      SpellCheckerSettings[Index] := sTrueCode
    else
      SpellCheckerSettings[Index] := sFalseCode;
  end;

begin
  LogEnterExit('WordUserSettingsSave');

  try
    SpellCheckerSettings := '';
    _Dialog.Update;
    SaveSetting(_Dialog.AutomaticSpellChecking,    usCheckSpellingAsYouType);
    SaveSetting(_Dialog.AutomaticGrammarChecking,  usCheckGrammarAsYouType);
    SaveSetting(_Dialog.FilenamesEmailAliases,     usIgnoreInternetAndFileAddresses);
    SaveSetting(_Dialog.IgnoreMixedDigits,         usIgnoreMixedDigits);
    SaveSetting(_Dialog.IgnoreAllCaps,             usIgnoreUppercase);
    SaveSetting(_Dialog.ForegroundGrammar,         usCheckGrammarWithSpelling);
    SaveSetting(_Dialog.ShowStatistics,            usShowReadabilityStatistics);
    SaveSetting(_Dialog.SuggestFromMainDictOnly,   usSuggestFromMainDictionaryOnly);
    SaveSetting(_Dialog.AlwaysSuggest,             usSuggestSpellingCorrections);
    SaveSetting(_Dialog.HideSpellingErrors,        usHideSpellingErrors);
    SaveSetting(_Dialog.HideGrammarErrors,         usHideGrammarErrors);
  finally
  end;

  LogEnterExit('WordUserSettingsSave',true);
end;

procedure WordSettingsRestore;

  function Load(Index: integer): integer;
  begin
    if _WordSettings.Count > Index then
      Result := Integer(_WordSettings[Index])
    else
      Result := 0
  end;

begin
  LogEnterExit('WordSettingsRestore');

  _Word.Options.AutoFormatAsYouTypeApplyBorders             := boolean(Load(wsAFAYTApplyBorders));
  _Word.Options.AutoFormatAsYouTypeApplyBulletedLists       := boolean(Load(wsAFAYTApplyBulletedLists));
  _Word.Options.AutoFormatAsYouTypeApplyFirstIndents        := boolean(Load(wsAFAYTApplyFirstIndents));
  _Word.Options.AutoFormatAsYouTypeApplyHeadings            := boolean(Load(wsAFAYTApplyHeadings));
  _Word.Options.AutoFormatAsYouTypeApplyNumberedLists       := boolean(Load(wsAFAYTApplyNumberedLists));
  _Word.Options.AutoFormatAsYouTypeApplyTables              := boolean(Load(wsAFAYTApplyTables));
  _Word.Options.AutoFormatAsYouTypeAutoLetterWizard         := boolean(Load(wsAFAYTAutoLetterWizard));
  _Word.Options.AutoFormatAsYouTypeDefineStyles             := boolean(Load(wsAFAYTDefineStyles));
  _Word.Options.AutoFormatAsYouTypeFormatListItemBeginning  := boolean(Load(wsAFAYTFormatListItemBeginning));
  _Word.Options.AutoFormatAsYouTypeInsertClosings           := boolean(Load(wsAFAYTInsertClosings));
  _Word.Options.AutoFormatAsYouTypeReplaceQuotes            := boolean(Load(wsAFAYTReplaceQuotes));
  _Word.Options.AutoFormatAsYouTypeReplaceFractions         := boolean(Load(wsAFAYTReplaceFractions));
  _Word.Options.AutoFormatAsYouTypeReplaceHyperlinks        := boolean(Load(wsAFAYTReplaceHyperlinks));
  _Word.Options.AutoFormatAsYouTypeReplaceOrdinals          := boolean(Load(wsAFAYTReplaceOrdinals));
  _Word.Options.AutoFormatAsYouTypeReplacePlainTextEmphasis := boolean(Load(wsAFAYTReplacePlainTextEmphasis));
  _Word.Options.AutoFormatAsYouTypeReplaceSymbols           := boolean(Load(wsAFAYTReplaceSymbols));
  _Word.Options.AutoFormatReplaceQuotes                     := boolean(Load(wsAutoFormatReplaceQuotes));
  _Word.Options.TabIndentKey                                := boolean(Load(wsTabIndentKey));
  _Word.WindowState                                         := Load(wsWindowState);
  _Word.Options.SaveInterval                                := Load(wsSaveInterval);
  _Doc.TrackRevisions                                       := boolean(Load(wsTrackRevisions));
  _Doc.ShowRevisions                                        := boolean(Load(wsShowRevisions));
  if (_WordVersion < 13) then                               // altered for Word 2010
    _Doc.ShowSummary                                        := boolean(Load(wsShowSummary));

  LogEnterExit('WordSettingsRestore', true);
end;

procedure WordDocumentNil;
begin
  LogEnterExit('WordDocumentToNil');
  if _Doc <> nil then
    _Doc := nil;
  LogEnterExit('WordDocumentToNil',true);
end;

procedure WordToNil;
begin
  LogEnterExit('WordToNil');
  _Word := nil;
  LogEnterExit('WordToNil', true);
end;


var
  WindowTitle: string;
  WindowHandle: HWnd;

function WordDocTitle: string;
var
  Guid: TGUID;
begin
  if WinApi.ActiveX.Succeeded(CreateGUID(Guid)) then
    Result := '/' + GUIDToString(Guid)
  else
    Result := '';
  Result := TX_WINDOW_TITLE + IntToStr(Application.Handle) + Result;
end;

function FindDocWindow(Handle: HWnd; Info: Pointer): BOOL; stdcall;
var
  title: string;

  function GetWindowTitle(Handle: HWnd): String;
  begin
    SetLength(result, 240);
    SetLength(result, GetWindowText(Handle, PChar(result), Length(result)));
  end;

begin
  title := GetWindowTitle(Handle);
  if title = WindowTitle then
  begin
    WindowHandle := Handle;
    result := FALSE;
  end
  else
    result := TRUE;
end;

function WordFindDocumentWindow: HWnd;
begin
  LogEnterExit('WordFindDocumentWindow');
  WindowTitle := _Title;
  WindowHandle := 0;
  EnumWindows(@FindDocWindow, 0);
  Result := WindowHandle;
  LogLine('DocWindowHandle: '+ IntToStr(Result));
  LogEnterExit('WordFindDocumentWindow', true);
//  _DocWindowHandle := Result;
end;

////////////////////////////////////////////////////////////////////////////////
function WordSpellCheck(var aText:String;aSpell:Boolean = True): Boolean;
begin
  LogEnterExit('WordSpellCheck');
//  LogLine('Before: '+#13#10#13#10+aText);
  Result := WordDoCheck(aText,aSpell);
//  LogLine('After: '+#13#10#13#10+aText);
  LogEnterExit('WordSpellCheck', true);
end;


function ListHasText(aList:TStrings):boolean;
var
  s: String;
begin
  Result := false;
  if not assigned(aList) then
    exit
  else
    for s in aList do
      begin
      result := result or (trim(s) <> '');
        if Result then
          System.Break;
      end;
end;

procedure SpellCheckMemo(aMemo: TCustomMemo;aGrammar:Boolean = false);
var
  s: String;
begin
  LogEnterExit('SpellCheckMemo');
  if Assigned(aMemo) then
    begin
      if ListHasText(aMemo.Lines) then
        begin
          s := aMemo.Lines.Text;
          if WordSpellCheck(s,aGrammar) then
            aMemo.Lines.Text := s
          else
            LogLine('WordSpellCheck canceled');
        end
      else
        LogLine('No text to check');
    end
  else
    begin
      LogLine('Control not assigned');
    end;
  LogEnterExit('SpellCheckMemo',true);
end;

procedure SpellCheckForControl(aMemo: TCustomMemo);
begin
  LogEnterExit('SpellCheckforControl');
  SpellCheckMemo(aMemo,true);
  LogEnterExit('SpellCheckforControl',true);
end;

procedure GrammarCheckForControl(aMemo: TCustomMemo);
begin
  LogEnterExit('GrammarCheckforControl');
  SpellCheckMemo(aMemo,false);
  LogEnterExit('GrammarforControl',true);
end;

////////////////////////////////////////////////////////////////////////////////
initialization
  _NullStr := '';
  _FalseVar := 0;
  _Title := WordDocTitle;
  _WordSettings := TList.Create;

finalization
  WordSpellCheckClose;
  _WordSettings.Free;

end.
