unit uSimilarNames;

{ ------------------------------------------------------------------------------
  Extention of ORCtrl.TORComboBox.
  Verifies if there are names "similar" to the selected one.
  If so - requests confirmation of the selection.
  NSR#20110606 (Similar Provider/Cosigner names)
  ---------------------------------------------------------------------------- }
interface

uses
  ORCtrls, System.Classes, WinApi.Windows;

type
  TSelector = (APt, APr, ACo, Unknown);

  TORComboBox = class(ORCtrls.TORComboBox)
  private
    fSelectorType: TSelector;
    fFilter: String;
    fNeedsValidation: Boolean;
    procedure OnComponentChange(Sender: TObject);
    procedure OnComponentKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure OnComponentDblClick(Sender: TObject);
    procedure setSelectorType(aType: TSelector);
  public
    ItemName:String;
    Exceptions: TStrings;
    ValidationDate: Double;
    TitleIEN: Integer;
    msgNoSelection: String;
    property NeedsValidation: Boolean read fNeedsValidation
      write fNeedsValidation;
    property SelectorType: TSelector read fSelectorType write setSelectorType;
    constructor Create(AOwner: TComponent); override;
    function isValidSelection(Sender: TObject): Boolean;
    function ValidationReport: String;
  end;

procedure setUpSimilarNameSelector(aSelector: TORComboBox; aType: TSelector;
  aFilter: String);

function getItemIDFromList(aList: TStrings; aType: TSelector = APt;
  anExceptions: TStrings = nil): Int64;
function getPersonCheckedForSimilarNames(aDUZ: Int64; aKey: String = '';
  anExceptions: TStrings = nil): Int64;
function getProviderIdCheckedForSimilarName(aDUZ: Int64; aKey: String = '';
  anExceptions: TStrings = nil): Int64;
function getCosignerCheckedForSimilarNames(aDUZ: Int64; CSPDate: Double;
  TitleIEN: Integer): Int64;

implementation

uses
  rCore, uCore, System.UITypes, Vcl.Dialogs, System.SysUtils, ORFn, fDupPts;

const
  fmtInvalidItemSelected =
    'The name selected is not a CPRS user name allowable for entry in this %s field.';
  fmtMultipleItemNames =
    'The %s name selected is not unique. The name confirmation is required.';
{$IFDEF DEBUG_AA}
  fmtInvalidDUZ =
    'The provided DUZ (%d) does not match a CPRS user name allowable for entry in this field.'
    + #13#10 + 'Please select another name.';
{$ENDIF}

function getItemIDFromList(aList: TStrings; aType: TSelector = APt;
  anExceptions: TStrings = nil): Int64;
var
  frmDupPts: TfrmDupPts;
begin
  Result := -1;
  if assigned(aList) then
  begin
    frmDupPts := TfrmDupPts.CreateSelector(aType, aList, anExceptions);
    try
      if frmDupPts.ShowModal = mrOK then
        Result := frmDupPts.lboSelPt.ItemID;
    finally
      frmDupPts.Release;
    end;
  end
end;

function isSingleProviderDUZ(aDUZ: Int64; aKey: String; var msg: String;
  ItemName: String = 'Provider'): Boolean;
var
  SL: TStrings;
begin
  Result := False;
  SL := SubsetOfActiveAndInactivePersonsWithSimilarNames(aDUZ, aKey);
  try
    case SL.Count of
      0:
        msg := Format(fmtInvalidItemSelected, [ItemName]);
      1:
        Result := True;
    else
      msg := Format(fmtMultipleItemNames, [ItemName]);
    end;
  finally
    SL.Free;
  end;
end;

function isSingleCosignerDUZ(aDUZ: Int64; CSPDate: Double; TitleIEN: Integer;
  var msg: String; ItemName: String = 'Cosigner'): Boolean;
var
  SL: TStrings;
begin
  Result := False;
  SL := nil;
  try
    SL := SubsetOfCosignersWithSimilarNames(aDUZ, CSPDate, TitleIEN);
    if not assigned(SL) or (SL.Count = 0) then
      msg := Format(fmtInvalidItemSelected, [ItemName])
    else if SL.Count > 1 then
      msg := Format(fmtMultipleItemNames, [ItemName])
    else
      Result := True;
  finally
    if assigned(SL) then
      SL.Free;
  end;
end;

function getPersonCheckedForSimilarNames(aDUZ: Int64; aKey: String = '';
  anExceptions: TStrings = nil): Int64;
var
  SL: TStrings;
  sDUZ: String;

  function countExceptions: Integer;
  var
    sDUZ: String;
    i, j: Integer;
  begin
    Result := 0;
    if not assigned(anExceptions) then
      exit;
    i := 0;
    while i < SL.Count do
    begin
      sDUZ := Piece(SL[i], U, 1);
      for j := 0 to anExceptions.Count - 1 do
        if Piece(anExceptions[j], U, 1) = sDUZ then
        begin
          Inc(Result);
          break;
        end;
      Inc(i);
    end;
  end;

begin
  Result := -1;
  SL := SubsetOfActiveAndInactivePersonsWithSimilarNames(aDUZ, aKey);
  try
    case SL.Count of
{$IFDEF DEBUG_AA}
      0:
        ShowMessage('getPersonCheckedForSimilarNames' + CRLF +
          Format(fmtInvalidDUZ, [aDUZ]));
{$ELSE}
      0:
        ShowMessage(Format(fmtInvalidItemSelected, ['']) +
          ' Please Select another name');
{$ENDIF}
      1:
        begin
          sDUZ := Piece(SL[0], U, 1);
          if sDUZ = IntToStr(aDUZ) then
            Result := aDUZ
          else
            MessageDLG('getPersonCheckedForSimilarNames:' + #13#10#13#10 +
              'Search for DUZ=' + IntToStr(aDUZ) +
              ' returns one record with DUZ=' + sDUZ, mtError, [mbOK], 0);
        end
    else
      begin
        if SL.Count = countExceptions then
          MessageDLG
            ('The selected person and all the persons with similar names were already added'
            + CRLF + 'Please select another person', mtInformation, [mbOK], 0)
        else
          Result := getItemIDFromList(SL, APr, anExceptions);
      end;
    end;
  finally
    SL.Free;
  end;
end;

function getProviderIdCheckedForSimilarName(aDUZ: Int64; aKey: String = '';
  anExceptions: TStrings = nil): Int64;
begin
  // "Similar Provider" window should not be shown if the logged on user is the selected provider
  if aDUZ = User.DUZ then
    Result := aDUZ
  else
    Result := getPersonCheckedForSimilarNames(aDUZ, aKey, anExceptions);
end;

function getCosignerCheckedForSimilarNames(aDUZ: Int64; CSPDate: Double;
  TitleIEN: Integer): Int64;
var
  sDUZ: String;
  SL: TStrings;
begin
  Result := -1;
  SL := SubsetOfCosignersWithSimilarNames(aDUZ, CSPDate, TitleIEN);
  case SL.Count of
{$IFDEF DEBUG_AA}
    0:
      ShowMessage(Format('getCosignerCheckedForSimilarNames' + #13#10 +
        fmtInvalidDUZ, [aDUZ]));
{$ELSE}
    0:
      ShowMessage(Format(fmtInvalidItemSelected, ['']) +
        ' Please Select another name');
{$ENDIF}
    1:
      begin
        sDUZ := Piece(SL[0], U, 1);
        if sDUZ = IntToStr(aDUZ) then
          Result := aDUZ
        else
          MessageDLG('getCosignerCheckedForSimilarNames:' + #13#10#13#10 +
            'Search for DUZ=' + IntToStr(aDUZ) + ' returns one record with DUZ='
            + sDUZ, mtError, [mbOK], 0);
      end
  else
    Result := getItemIDFromList(SL, ACo);
  end;
  SL.Free;
end;

function getCosignerIdCheckedForSimilarNames(aDUZ: Int64; CSPDate: Double;
  TitleIEN: Integer): Int64;
begin
  // "Similar Provider" window should not be shown if the logged on user is the selected provider
  if aDUZ = User.DUZ then
    Result := aDUZ
  else
    Result := getCosignerCheckedForSimilarNames(aDUZ, CSPDate, TitleIEN);
end;

procedure setUpSimilarNameSelector(aSelector: TORComboBox; aType: TSelector;
  aFilter: String);
begin
  if aSelector = nil then
    exit;
  with aSelector do
  begin
    SelectorType := aType;
    fFilter := aFilter;
  end;

end;

/// /////////////////////////////////////////////////////////////////////////////

constructor TORComboBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  OnChange := OnComponentChange;
  OnKeyDown := OnComponentKeyDown;
  OnDblClick := OnComponentDblClick;
  NeedsValidation := True;
  SelectorType := Unknown;
end;

function TORComboBox.isValidSelection(Sender: TObject): Boolean;
var
  ID: Int64;  // Issue Tracker #41
begin
  ID := -1;
  Result := False;
  if ItemIEN = 0 then // V32 Defect #111 4/17/2017 SDS
    exit;
  case SelectorType of
    APt:
      ;
    APr:
      ID := getProviderIdCheckedForSimilarName(ItemIEN, fFilter);
    ACo:
      ID := getCosignerIdCheckedForSimilarNames(ItemIEN, ValidationDate,
        TitleIEN);
    Unknown:
      ;
  end;
 // Issue Tracker #41 --------------------------------------------- begin
  if ID >= 0 then  
    begin
      SelectByIEN(ID);
    end;
  Result := ID >= 0;
  NeedsValidation := not Result;  
// Issue Tracker #41 ----------------------------------------------- end
end;

procedure TORComboBox.OnComponentChange(Sender: TObject);
begin
  inherited;
  fNeedsValidation := True;
end;

procedure TORComboBox.OnComponentDblClick(Sender: TObject);
begin
  inherited;
  if NeedsValidation then
    if isValidSelection(Sender) then
      fNeedsValidation := False;
end;

procedure TORComboBox.OnComponentKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  inherited;
  if Key = VK_Return then
  begin
    if NeedsValidation then
    begin
      if isValidSelection(Sender) then
        fNeedsValidation := False;
      Key := 0; // to prevent the default button to close the paren dialog
    end;
  end;
end;

function TORComboBox.ValidationReport: String;
var
  errMsg: String;
begin
  case fSelectorType of
    Unknown:
      Result := 'Unknown type of validation';
    APt:
      Result := 'Patient Validation';
    APr:
      begin
        if ItemIEN = 0 then
          Result := msgNoSelection
        else if (not NeedsValidation) or isSingleProviderDUZ(ItemIEN, fFilter,
          errMsg, ItemName) then
          Result := ''
        else
          Result := errMsg;
      end;
    ACo:
      begin
        if ItemIEN = 0 then
          Result := msgNoSelection
        else if (not NeedsValidation) or isSingleCosignerDUZ(ItemIEN,
          ValidationDate, TitleIEN, errMsg, ItemName) then
          Result := ''
        else
          Result := errMsg;
      end;
  end;
  if Result <> '' then
    try
      Result := CRLF + Result;
//      SetFocus; - RTC Defect 636270
//                  setting focus inside this procedure ignites an infinite loop
    except
      on E: Exception do
        ShowMessage('ERROR (TORComboBox.ValidationReport):' + CRLF + E.Message);
    end;
end;

procedure TORComboBox.setSelectorType(aType: TSelector);
begin
  fSelectorType := aType;
end;

end.
