/**
 * The package gov.va.med.mbac.vlh.impl contains the VistaLinkHelper Class,
 * which implements the IVistaLinkHelper interface.
 * 
 * Including the unit tests in the same package as the classes under test allows test cases access to
 * friendly and protected members, without breaking encapsulation.
 * 
 */
package gov.va.med.mbac.vlh.impl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.stub;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.powermock.api.mockito.PowerMockito.mockStatic;
import static org.powermock.api.mockito.PowerMockito.verifyNew;
import static org.powermock.api.mockito.PowerMockito.verifyStatic;
import static org.powermock.api.mockito.PowerMockito.whenNew;
import gov.va.med.mbac.vlh.common.IVistaLinkHelper;
import gov.va.med.vistalink.adapter.cci.VistaLinkAppProxyConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnection;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionFactory;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkDuzConnectionSpec;
import gov.va.med.vistalink.institution.InstitutionMapNotInitializedException;
import gov.va.med.vistalink.institution.InstitutionMappingDelegate;
import gov.va.med.vistalink.institution.InstitutionMappingNotFoundException;
import gov.va.med.vistalink.rpc.RpcRequest;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.resource.ResourceException;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;
import org.springframework.beans.BeansException;
import org.springframework.context.support.ClassPathXmlApplicationContext;

/**
 * This is the VistaLinkHelperTest class, which implements the unit tests
 * for the VistaLinkHelper class.
 * 
 * @author PII
 * @version 1.0.0
 */

@RunWith(PowerMockRunner.class)
@PrepareForTest({ InstitutionMappingDelegate.class, VistaLinkDuzConnectionSpec.class })
public class VistaLinkHelperTest {

	private static final String SITE_ID = "500";
	private static final String RPC_CONTEXT = "ORWS WEB SERVICES";
	private static final String APP_PROXY_NAME = "CHAMPVAPAY,APPLICATION PROXY";

	private ClassPathXmlApplicationContext context = null;

	@Before
	public void setUp() {
		this.context = new ClassPathXmlApplicationContext("/vlhTestBeans.xml");
	}

	@After
	public void tearDown() {
		this.context.close();
	}

	
	/**
	 * *** Tests for Spring DI ***
	 */

	/**
	 * Test DI of vistaLinkHelper
	 */
	@Test
	public final void testVistaLinkHelperInjection() {
		IVistaLinkHelper vistaLinkHelper = (VistaLinkHelper) context.getBean("vistaLinkHelper");
		assertNotNull("vistaLinkHelper failed to instantiate.", vistaLinkHelper);
		assertTrue("vistaLinkHelper instance of VistaLinkHelper.", (vistaLinkHelper instanceof VistaLinkHelper));
		RpcRequest rpcRequest = vistaLinkHelper.getRpcRequest();
		assertNotNull("rpcRequest failed to instantiate.", rpcRequest);
		assertTrue("rpcRequest instance of RpcRequest.", (rpcRequest instanceof RpcRequest));
		assertEquals(RPC_CONTEXT, rpcRequest.getRpcContext());
		assertFalse(rpcRequest.isXmlResponse());
	}

	/**
	 * Test DI of rpcRequest
	 */
	@Test
	public final void testRpcRequestInjection() {
		IVistaLinkHelper vistaLinkHelper = (VistaLinkHelper) context.getBean("vistaLinkHelper");
		RpcRequest rpcRequest = vistaLinkHelper.getRpcRequest();
		assertNotNull("rpcRequest failed to instantiate.", rpcRequest);
		assertTrue("rpcRequest instance of RpcRequest.", (rpcRequest instanceof RpcRequest));
		assertEquals(RPC_CONTEXT, rpcRequest.getRpcContext());
		assertFalse(rpcRequest.isXmlResponse());
	}
	
	/**
	 * Test that the appContext is returning a new RpcRequest bean with each call
	 * i.e., NOT a Spring singleton.
	 */
	@Test
	public final void testThatGetRpcRequestReturnsNewInstanceEachCall() {
		IVistaLinkHelper vistaLinkHelper = (VistaLinkHelper) context.getBean("vistaLinkHelper");
		RpcRequest rpcRequest0 = vistaLinkHelper.getRpcRequest();
		assertNotNull("rpcRequest failed to instantiate.", rpcRequest0);
		RpcRequest rpcRequest1 = vistaLinkHelper.getRpcRequest();
		assertNotNull("rpcRequest failed to instantiate.", rpcRequest1);

		assertFalse(rpcRequest0 == rpcRequest1);
	}

	/**
	 * Test that getRpcRequest rethrows BeansException when context.getBean fails
	 */
	@Test
	public final void testGetRpcRequestThrowsBeansException() {
		ClassPathXmlApplicationContext localContext = new ClassPathXmlApplicationContext("/vlhTestBeansNoReq.xml");
		try {
			IVistaLinkHelper vistaLinkHelper = (VistaLinkHelper) localContext.getBean("vistaLinkHelper");
			RpcRequest rpcRequest = vistaLinkHelper.getRpcRequest();
			assertNotNull("rpcRequest failed to instantiate.", rpcRequest);
		} catch (BeansException be) {
			assertEquals("No bean named 'rpcRequest' is defined", be.getMessage());
		} finally {
			localContext.close();
		}
	}

	/**
	 * Verify that getConnection returns a valid VistaLink Connection (happy path)
	 * 
	 */
	@Test
	public void verifyThatGetConnectionReturnsVistaLinkConnection() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);

			assertNotNull("connection not obtained.", vlc);
			assertEquals(mockedConnection, vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Test that getConnection throws ResourceException when siteId == null
	 */
	@Test
	public void testThatGetConnectionThrowsResourceExceptionWhenSiteIdEqualsNull() {
		IVistaLinkHelper vlh;
		try {
			vlh = (VistaLinkHelper) context.getBean("vistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);
			vlh.getConnection(null, "777");
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("VistA Site not identified", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Test that getConnection throws ResourceException when siteId == ""
	 */
	@Test
	public void testThatGetConnectionThrowsResourceExceptionWhenSiteIdEqualsEmptyString() {
		IVistaLinkHelper vlh;
		try {
			vlh = (VistaLinkHelper) context.getBean("vistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);
			vlh.getConnection("", "777");
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("VistA Site not identified", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Test that getConnection throws ResourceException when duz == null
	 */
	@Test
	public void testThatGetConnectionThrowsResourceExceptionWhenDuzEqualsNull() {
		IVistaLinkHelper vlh;
		try {
			vlh = (VistaLinkHelper) context.getBean("vistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);
			vlh.getConnection(SITE_ID, null);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("User not identified", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Test that getConnection throws ResourceException when duz == ""
	 */
	@Test
	public void testThatGetConnectionThrowsResourceExceptionWhenDuzEqualsEmptyString() {
		IVistaLinkHelper vlh;
		try {
			vlh = (VistaLinkHelper) context.getBean("vistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);
			vlh.getConnection(SITE_ID, "");
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("User not identified", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that new VistaLinkConnectionSpec(siteId, duz) is called from
	 * getConnection(siteId, duz)
	 */
	@Test
	public void verifyThatNewVistaLinkConnectionSpecIsCalledFromGetConnection() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			
			VistaLinkDuzConnectionSpec mockedConnSpec = mock(VistaLinkDuzConnectionSpec.class);

			whenNew(VistaLinkDuzConnectionSpec.class)
					.withArguments(siteId, duz)
					.thenReturn(mockedConnSpec);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenReturn("vlj/mydev500");

			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);
			assertNotNull("connection not obtained.", vlc);
			
			verifyNew(VistaLinkDuzConnectionSpec.class, times(1));
			
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that
	 * InstitutionMappingDelegate.getJndiConnectorNameForInstitution(siteId) is
	 * called from getConnection(siteId, duz)
	 */
	@Test
	public void verifyThatInstitutionMappingDelegateGetJndiConnectorNameForInstitutionIsCalledFromGetConnection() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);

			verifyStatic(times(1));
			InstitutionMappingDelegate
					.getJndiConnectorNameForInstitution(siteId);
			assertNotNull("connection not obtained.", vlc);
			assertEquals(mockedConnection, vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that InitialContext.lookup(jndiName) is called from
	 * getConnection(siteId, duz) once
	 */
	@Test
	public void verifyThatInitialContextLookupIsCalledFromGetConnection() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			stub(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.toReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);

			verify(mockedIc, times(1)).lookup("vlj/mydev500");
			assertNotNull("connection not obtained.", vlc);
			assertEquals(mockedConnection, vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that VistaLinkConnectionFactory.getConnection(connSpec) is called
	 * from getConnection(siteId, duz) once
	 */
	@Test
	public void verifyThatVistaLinkConnectionFactoryGetConnectionIsCalledFromGetConnection() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);

			verify(mockedVLcf, times(1)).getConnection(
					any(VistaLinkConnectionSpec.class));
			assertNotNull("connection not obtained.", vlc);
			assertEquals(mockedConnection, vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that getConnection(siteId, duz) rethrows InstitutionMapNotInitialized
	 * when InstitutionMappingDelegate.getJndiConnectorNameForInstitution(siteId)
	 * throws InstitutionMapNotInitialized
	 */
	@Test
	public void verifyThatGetConnectionRethrowsInstitutionMapNotInitialized() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenThrow(new InstitutionMapNotInitializedException("NO MAP, GOOGLE!"));

			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);

			assertNotNull("connection not obtained.", vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			assertEquals("NO MAP, GOOGLE!", e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that getConnection(siteId, duz) rethrows InstitutionMappingNotFoundException
	 * when InstitutionMappingDelegate.getJndiConnectorNameForInstitution(siteId)
	 * throws InstitutionMappingNotFoundException
	 */
	@Test
	public void verifyThatGetConnectionRethrowsInstitutionMappingNotFound() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenThrow(new InstitutionMappingNotFoundException("WHERE'S THE MAP, GOOGLE!"));

			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);

			assertNotNull("connection not obtained.", vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			assertEquals("WHERE'S THE MAP, GOOGLE!", e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that getConnection(siteId, duz) rethrows NamingException
	 * when InitialContext.lookup(jndiName) throws NamingException.
	 *
	 */
	@Test
	public void verifyThatGetConnectionRethrowsNamingException() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenThrow(new NamingException("Rational Jazz is NOT!"));
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);

			assertNotNull("connection not obtained.", vlc);
		} catch (NamingException e) {
			assertEquals("Rational Jazz is NOT!", e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that getConnection(siteId, duz) rethrows ResourceException
	 * when connectionFactory.getConnection(connSpec) throws ResourceException.
	 *
	 */
	@Test
	public void verifyThatGetConnectionRethrowsResourceException() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		String duz = "983";
		try {
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenThrow(new ResourceException("Team Concert != Agile"));
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getConnection(siteId, duz);

			assertNotNull("connection not obtained.", vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("Team Concert != Agile", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	
	/**
	 * --- Tests for getAppProxyConnection(siteId, appProxyName) ---
	 */
	
	
	/**
	 * Verify that getAppProxyConnection returns a valid VistaLink Connection (happy path)
	 * 
	 */
	@Test
	public void verifyThatGetAppProxyConnectionReturnsVistaLinkConnection() {
		IVistaLinkHelper vlh;
		String siteId = SITE_ID;
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(siteId))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getAppProxyConnection(siteId, APP_PROXY_NAME);

			assertNotNull("connection not obtained.", vlc);
			assertEquals(mockedConnection, vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Test that getAppProxyConnection throws ResourceException when siteId == null
	 */
	@Test
	public void testThatGetAppProxyConnectionThrowsResourceExceptionWhenSiteIdEqualsNull() {
		IVistaLinkHelper vlh;
		try {
			vlh = (VistaLinkHelper) context.getBean("vistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);
			vlh.getAppProxyConnection(null, APP_PROXY_NAME);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("VistA Site not identified", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Test that getAppProxyConnection throws ResourceException when siteId == ""
	 */
	@Test
	public void testThatGetAppProxyConnectionThrowsResourceExceptionWhenSiteIdEqualsEmptyString() {
		IVistaLinkHelper vlh;
		try {
			vlh = (VistaLinkHelper) context.getBean("vistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);
			vlh.getAppProxyConnection("", APP_PROXY_NAME);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("VistA Site not identified", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Test that getAppProxyConnection throws ResourceException when appProxyName == null
	 */
	@Test
	public void testThatGetAppProxyConnectionThrowsResourceExceptionWhenAppProxyNameEqualsNull() {
		IVistaLinkHelper vlh;
		try {
			vlh = (VistaLinkHelper) context.getBean("vistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);
			vlh.getAppProxyConnection(SITE_ID, null);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("Application Proxy not identified", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Test that getAppProxyConnection throws ResourceException when appProxyName == ""
	 */
	@Test
	public void testThatGetAppProxyConnectionThrowsResourceExceptionWhenAppProxyNameEqualsEmptyString() {
		IVistaLinkHelper vlh;
		try {
			vlh = (VistaLinkHelper) context.getBean("vistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);
			vlh.getAppProxyConnection(SITE_ID, "");
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("Application Proxy not identified", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that new VistaLinkAppProxyConnectionSpec(siteId, appProxyName) is called from
	 * getAppProxyConnection(siteId, appProxyName)
	 */
	@Test
	public void verifyThatNewVistaLinkConnectionSpecIsCalledFromGetAppProxyConnection() {
		IVistaLinkHelper vlh;
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			
			VistaLinkAppProxyConnectionSpec mockedConnSpec = mock(VistaLinkAppProxyConnectionSpec.class);

			whenNew(VistaLinkAppProxyConnectionSpec.class)
					.withArguments(SITE_ID, APP_PROXY_NAME)
					.thenReturn(mockedConnSpec);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(SITE_ID))
					.thenReturn("vlj/mydev500");

			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			VistaLinkConnection vlc = vlh.getAppProxyConnection(SITE_ID, APP_PROXY_NAME);
			assertNotNull("connection not obtained.", vlc);
			
			verifyNew(VistaLinkAppProxyConnectionSpec.class, times(1));
			
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that
	 * InstitutionMappingDelegate.getJndiConnectorNameForInstitution(siteId) is
	 * called from getAppProxyConnection(siteId, appProxyName)
	 */
	@Test
	public void verifyThatInstitutionMappingDelegateGetJndiConnectorNameForInstitutionIsCalledFromGetAppProxyConnection() {
		IVistaLinkHelper vlh;
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(SITE_ID))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getAppProxyConnection(SITE_ID, APP_PROXY_NAME);

			verifyStatic(times(1));
			InstitutionMappingDelegate
					.getJndiConnectorNameForInstitution(SITE_ID);
			assertNotNull("connection not obtained.", vlc);
			assertEquals(mockedConnection, vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that InitialContext.lookup(jndiName) is called from
	 * getAppProxyConnection(siteId, appProxyName) once
	 */
	@Test
	public void verifyThatInitialContextLookupIsCalledFromGetAppProxyConnection() {
		IVistaLinkHelper vlh;
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			stub(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.toReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(SITE_ID))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getAppProxyConnection(SITE_ID, APP_PROXY_NAME);

			verify(mockedIc, times(1)).lookup("vlj/mydev500");
			assertNotNull("connection not obtained.", vlc);
			assertEquals(mockedConnection, vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that VistaLinkConnectionFactory.getAppProxyConnection(connSpec) is called
	 * from getAppProxyConnection(siteId, appProxyName) once
	 */
	@Test
	public void verifyThatVistaLinkConnectionFactoryGetConnectionIsCalledFromGetAppProxyConnection() {
		IVistaLinkHelper vlh;
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkAppProxyConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(SITE_ID))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getConnection(SITE_ID, APP_PROXY_NAME);

			verify(mockedVLcf, times(1)).getConnection(
					any(VistaLinkConnectionSpec.class));
			assertNotNull("connection not obtained.", vlc);
			assertEquals(mockedConnection, vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that getAppProxyConnection(siteId, appProxyName) rethrows InstitutionMapNotInitialized
	 * when InstitutionMappingDelegate.getJndiConnectorNameForInstitution(siteId)
	 * throws InstitutionMapNotInitialized
	 */
	@Test
	public void verifyThatGetAppProxyConnectionRethrowsInstitutionMapNotInitialized() {
		IVistaLinkHelper vlh;
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(SITE_ID))
					.thenThrow(new InstitutionMapNotInitializedException("NO MAP, GOOGLE!"));

			VistaLinkConnection vlc = vlh.getAppProxyConnection(SITE_ID, APP_PROXY_NAME);

			assertNotNull("connection not obtained.", vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			assertEquals("NO MAP, GOOGLE!", e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that getAppProxyConnection(siteId, appProxyName) rethrows InstitutionMappingNotFoundException
	 * when InstitutionMappingDelegate.getJndiConnectorNameForInstitution(siteId)
	 * throws InstitutionMappingNotFoundException
	 */
	@Test
	public void verifyThatGetAppProxyConnectionRethrowsInstitutionMappingNotFound() {
		IVistaLinkHelper vlh;
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkAppProxyConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(SITE_ID))
					.thenThrow(new InstitutionMappingNotFoundException("WHERE'S THE MAP, GOOGLE!"));

			VistaLinkConnection vlc = vlh.getAppProxyConnection(SITE_ID, APP_PROXY_NAME);

			assertNotNull("connection not obtained.", vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			assertEquals("WHERE'S THE MAP, GOOGLE!", e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that getAppProxyConnection(siteId, appProxyName) rethrows NamingException
	 * when InitialContext.lookup(jndiName) throws NamingException.
	 *
	 */
	@Test
	public void verifyThatGetAppProxyConnectionRethrowsNamingException() {
		IVistaLinkHelper vlh;
		try {
			VistaLinkConnection mockedConnection = mock(VistaLinkConnection.class);
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkAppProxyConnectionSpec.class)))
					.thenReturn(mockedConnection);
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenThrow(new NamingException("Rational Jazz is NOT!"));
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(SITE_ID))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getAppProxyConnection(SITE_ID, APP_PROXY_NAME);

			assertNotNull("connection not obtained.", vlc);
		} catch (NamingException e) {
			assertEquals("Rational Jazz is NOT!", e.getMessage());
		} catch (ResourceException e) {
			fail(e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}

	/**
	 * Verify that getAppProxyConnection(siteId, appProxyName) rethrows ResourceException
	 * when connectionFactory.getConnection(connSpec) throws ResourceException.
	 *
	 */
	@Test
	public void verifyThatGetAppProxyConnectionRethrowsResourceException() {
		IVistaLinkHelper vlh;
		try {
			VistaLinkConnectionFactory mockedVLcf = mock(VistaLinkConnectionFactory.class);
			when(mockedVLcf.getConnection(any(VistaLinkAppProxyConnectionSpec.class)))
					.thenThrow(new ResourceException("Team Concert != Agile"));
			InitialContext mockedIc = (InitialContext) context.getBean("mockedInitialContext");
			when(mockedIc.lookup("vlj/mydev500")).thenReturn(mockedVLcf);
			whenNew(InitialContext.class).withNoArguments()
					.thenReturn(mockedIc);
			vlh = (VistaLinkHelper) context.getBean("testVistaLinkHelper");
			assertNotNull("vlh failed to instantiate.", vlh);

			mockStatic(InstitutionMappingDelegate.class);
			when(
					InstitutionMappingDelegate
							.getJndiConnectorNameForInstitution(SITE_ID))
					.thenReturn("vlj/mydev500");

			VistaLinkConnection vlc = vlh.getAppProxyConnection(SITE_ID, APP_PROXY_NAME);

			assertNotNull("connection not obtained.", vlc);
		} catch (NamingException e) {
			fail(e.getMessage());
		} catch (ResourceException e) {
			assertEquals("Team Concert != Agile", e.getMessage());
		} catch (InstitutionMapNotInitializedException e) {
			fail(e.getMessage());
		} catch (InstitutionMappingNotFoundException e) {
			fail(e.getMessage());
		} catch (Exception e) {
			fail(e.getMessage());
		}
	}
	
}
