/**
 * The package gov.va.med.mbac.vlh.impl contains the VistaLinkHelper Class,
 * which implements the IVistaLinkHelper interface.
 * 
 */
package gov.va.med.mbac.vlh.impl;

import gov.va.med.mbac.vlh.common.IVistaLinkHelper;
import gov.va.med.vistalink.adapter.cci.VistaLinkAppProxyConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnection;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionFactory;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnectionSpec;
import gov.va.med.vistalink.adapter.cci.VistaLinkDuzConnectionSpec;
import gov.va.med.vistalink.institution.InstitutionMapNotInitializedException;
import gov.va.med.vistalink.institution.InstitutionMappingDelegate;
import gov.va.med.vistalink.institution.InstitutionMappingNotFoundException;
import gov.va.med.vistalink.rpc.RpcRequest;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.resource.ResourceException;

import org.apache.log4j.Logger;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

/**
 * This is the VistaLinkHelper class, which helps preserve coherence
 * and single-responsibility for our DAO classes by handling interactions
 * with VistaLink on their behalf.
 * 
 * @author PII
 * @version 1.0.0
 */
public class VistaLinkHelper implements IVistaLinkHelper, ApplicationContextAware {

	private static final Logger LOGGER = Logger.getLogger(VistaLinkHelper.class);

	private ApplicationContext appContext;
	private InitialContext initialContext;

	/**
	 * VistaLinkHelper is lazily instantiated by Spring DI
	 * 
	 * @param InitialContext
	 * 
	 */
	public VistaLinkHelper(InitialContext initialContext) {
		super();
		this.initialContext = initialContext;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gov.va.med.mbac.vpls.dao.common.IVistaLinkHelper#getRpcRequest()
	 */
	@Override
	public RpcRequest getRpcRequest() throws BeansException {
		RpcRequest rpcRequest;
		
		try {
			rpcRequest = (RpcRequest) appContext.getBean("rpcRequest");
			return rpcRequest;			
		} catch (BeansException be) {
			LOGGER.fatal(be.getMessage());
			throw be;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * gov.va.med.mbac.vpls.dao.common.IVistaLinkHelper#getConnection(java.lang.String
	 * , java.lang.String)
	 */
	@Override
	public VistaLinkConnection getConnection(String siteId, String duz)
			throws InstitutionMapNotInitializedException,
			InstitutionMappingNotFoundException, NamingException,
			ResourceException {

		VistaLinkConnection connection = null;

		if ((siteId == null) || siteId.equals("")) {
			throw new ResourceException("VistA Site not identified");
		}

		if ((duz == null) || duz.equals("")) {
			throw new ResourceException("User not identified");
		}

		VistaLinkConnectionSpec connSpec = new VistaLinkDuzConnectionSpec(
				siteId, duz);
		String jndiName;

		try {
			jndiName = InstitutionMappingDelegate
					.getJndiConnectorNameForInstitution(siteId);

			VistaLinkConnectionFactory connectionFactory = (VistaLinkConnectionFactory) initialContext
					.lookup(jndiName);
			connection = (VistaLinkConnection) connectionFactory.getConnection(connSpec);
			/**
			 * Use default timeout for now, consider implementing default in
			 * properties file if needed.
			 * 
			 * connection.setTimeOut(Integer.parseInt(CRProServerProps.
			 * getProperty("vistalink_socket_timeout")));
			 */

			return connection;

		} catch (InstitutionMapNotInitializedException imnie) {
			LOGGER.fatal(imnie.getMessage());
			throw imnie;
		} catch (InstitutionMappingNotFoundException imnfe) {
			LOGGER.fatal(imnfe.getMessage());
			throw imnfe;
		} catch (NamingException ne) {
			LOGGER.fatal(ne.getMessage());
			throw ne;
		} catch (ResourceException re) {
			LOGGER.fatal(re.getMessage());
			throw re;
		}

	}

	/* (non-Javadoc)
	 * @see gov.va.med.mbac.vlh.common.IVistaLinkHelper#getAppProxyConnection(java.lang.String, java.lang.String)
	 */
	@Override
	public VistaLinkConnection getAppProxyConnection(String siteId, String appProxyName)
		throws InstitutionMapNotInitializedException, InstitutionMappingNotFoundException, NamingException,
		ResourceException {

		VistaLinkConnection connection = null;

		if ((siteId == null) || siteId.equals("")) {
			throw new ResourceException("VistA Site not identified");
		}

		if ((appProxyName == null) || appProxyName.equals("")) {
			throw new ResourceException("Application Proxy not identified");
		}

		VistaLinkConnectionSpec connSpec = new VistaLinkAppProxyConnectionSpec(
				siteId, appProxyName);

		String jndiName;

		try {
			jndiName = InstitutionMappingDelegate
					.getJndiConnectorNameForInstitution(siteId);

			VistaLinkConnectionFactory connectionFactory = (VistaLinkConnectionFactory) initialContext
					.lookup(jndiName);
			connection = (VistaLinkConnection) connectionFactory.getConnection(connSpec);
			/**
			 * Use default timeout for now, consider implementing default in
			 * properties file if needed.
			 * 
			 * connection.setTimeOut(Integer.parseInt(CRProServerProps.
			 * getProperty("vistalink_socket_timeout")));
			 */

			return connection;

		} catch (InstitutionMapNotInitializedException imnie) {
			LOGGER.fatal(imnie.getMessage());
			throw imnie;
		} catch (InstitutionMappingNotFoundException imnfe) {
			LOGGER.fatal(imnfe.getMessage());
			throw imnfe;
		} catch (NamingException ne) {
			LOGGER.fatal(ne.getMessage());
			throw ne;
		} catch (ResourceException re) {
			LOGGER.fatal(re.getMessage());
			throw re;
		}

	}

	/* (non-Javadoc)
	 * @see org.springframework.context.ApplicationContextAware#setApplicationContext(org.springframework.context.ApplicationContext)
	 */
	@Override
	public void setApplicationContext(ApplicationContext ac)
			throws BeansException {
		this.appContext = ac;
		
	}

}
