/**
 * The package gov.va.med.mbcp.champva.rest contains the RESTful Web Application & Resource classes for
 * retrieval of Code Results from the target VistA instance.
 * 
 */
package gov.va.med.mbcp.champva.rest;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.powermock.api.mockito.PowerMockito.mockStatic;
import static org.powermock.api.mockito.PowerMockito.verifyStatic;

import javax.naming.NamingException;
import javax.resource.ResourceException;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.ResponseBuilder;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;

import com.fasterxml.jackson.core.JsonGenerationException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import gov.va.med.exception.FoundationsException;
import gov.va.med.mbcp.champva.dao.common.IChampVAPayDAO;
import gov.va.med.mbcp.champva.dao.impl.ChampVAPayDAO;
import gov.va.med.mbcp.champva.domain.common.ICodeResult;
import gov.va.med.mbcp.champva.domain.impl.ChampVAException;
import gov.va.med.mbcp.champva.domain.impl.ChampVAFoundationsException;
import gov.va.med.mbcp.champva.domain.impl.ChampVANamingException;
import gov.va.med.mbcp.champva.domain.impl.ChampVAResourceException;
import gov.va.med.mbcp.champva.domain.impl.ChampVAVLFaultException;
import gov.va.med.mbcp.champva.domain.impl.CodeResult;
import gov.va.med.vistalink.adapter.record.VistaLinkFaultException;

/**
 * Test class for ChampVAResource Class.
 * 
 * @author PII
 * @version 1.0.0
 */
@RunWith(PowerMockRunner.class)
@PrepareForTest({ Response.class })
public class ChampVAResourceTest {

    private static final String NDC_RESULT = "No";

    private static final String CPT_RESULT = "Yes";

    private static final String ICD_RESULT = "Yes^I can't love ya 'cause your feet's too BIG!";

    private static final String ICD9 = "253.8"; // conditional

    private static final String NDC = "55700-0373-30"; // "No"

    private static final String CPT = "98941"; // "Yes"

    private static final Object EXPECTED_JSON = "{\"result\":\"No\",\"message\":null}";

    private ChampVAResource cr;

    /**
     * The setUp method, to be executed before each test and initialize the
     * ChampVAResource under test.
     * 
     * @throws java.lang.Exception - an Exception propagated during set up.
     */
    @Before
    public void setUp() throws Exception {
	this.cr = new ChampVAResource();
    }

    /**
     * The tearDown method, to be executed after each test and dispose of the
     * ChampVAResource under test.
     * 
     * @throws java.lang.Exception - an Exception propagated during tear down.
     */
    @After
    public void tearDown() throws Exception {
	this.cr = null;
    }

    /**
     * Test that getCodeResult(code) calls
     * champVAPayDAO.getCodeResult(codeSystem, code)
     */
    @Test
    public void testThatGetCodeResultCallsChampVAPayDAOGetCodeResultWithCodeSystemAndCode() {
	ICodeResult ecr = new CodeResult(ICD_RESULT);
	this.cr.codeSystem = "ICD9";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, ICD9)).thenReturn(ecr);

	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);

	    Response resp = cr.getCodeResult(ICD9);

	    assertNotNull("resp is null.", resp);
	    verify(mockedDAO, times(1)).getCodeResult(this.cr.codeSystem, ICD9);

	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	}
    }

    /**
     * Test that getCodeResult(code) calls
     * objectMapper.writeValueAsString(codeResult)
     */
    @Test
    public void testThatGetCodeResultCallsObjectMapperWriteValueAsStringWithCode() {
	ICodeResult ecr = new CodeResult(CPT_RESULT);
	this.cr.codeSystem = "CPT";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, CPT)).thenReturn(ecr);

	    ObjectMapper mockedMapper = mock(ObjectMapper.class);
	    when(mockedMapper.writeValueAsString(ecr)).thenReturn("[]");

	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);
	    cr.setMapper(mockedMapper);

	    Response resp = cr.getCodeResult(CPT);

	    assertNotNull("resp is null.", resp);
	    verify(mockedMapper, times(1)).writeValueAsString(ecr);

	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	} catch (JsonProcessingException e) {
	    fail("failed with JsonProcessingException: " + e.getMessage());
	}
    }

    /**
     * Test that getCodeResult(code) calls Response.status(Response.Status.OK),
     * responseBuilder.entity(json), and responseBuilder.build() once.
     * 
     */
    @Test
    public void testThatGetCodeResultCallsResponseStatusOnce() {
	ICodeResult ecr = new CodeResult(NDC_RESULT);
	this.cr.codeSystem = "NDC";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, NDC)).thenReturn(ecr);

	    ResponseBuilder mockedBuilder = mock(ResponseBuilder.class);
	    when(mockedBuilder.entity(EXPECTED_JSON)).thenReturn(mockedBuilder);
	    Response mockedResponse = mock(Response.class);
	    when(mockedBuilder.build()).thenReturn(mockedResponse);
	    mockStatic(Response.class);
	    when(Response.status(Response.Status.OK)).thenReturn(mockedBuilder);

	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);

	    Response resp = cr.getCodeResult(NDC);

	    assertNotNull("resp is null.", resp);

	    verifyStatic(times(1));
	    Response.status(Response.Status.OK);

	    verify(mockedBuilder, times(1)).entity(EXPECTED_JSON);
	    verify(mockedBuilder, times(1)).build();

	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	}
    }

    /**
     * Test method for
     * {@link gov.va.med.mbcp.champva.rest.ChampVAResource#getCodeResult(java.lang.String)}.
     */
    @Test
    public void testGetCodeResult() {
	ICodeResult ecr = new CodeResult(NDC_RESULT);
	this.cr.codeSystem = "NDC";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, NDC)).thenReturn(ecr);

	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);

	    Response resp = cr.getCodeResult(NDC);

	    assertNotNull("resp is null.", resp);
	    assertTrue(resp.getEntity().equals(EXPECTED_JSON));
	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	}
    }

    /**
     * Test that getCodeResult(code) returns Internal Server Error when
     * champVAPayDAO.getCodeResult(codeSystem, code) raises a
     * ChampVAVLFaultVAException
     */
    @Test
    public void testThatGetCodeResultReturnsInternalServerErrorOnVistaLinkFaultException() {
	VistaLinkFaultException vlfe = new VistaLinkFaultException("BOOM!");
	this.cr.codeSystem = "NDC";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, NDC)).thenThrow(new ChampVAVLFaultException(vlfe));

	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);

	    Response resp = cr.getCodeResult(NDC);

	    assertNotNull("resp is null.", resp);
	    assertEquals(Response.Status.INTERNAL_SERVER_ERROR.getStatusCode(), resp.getStatus());
	    assertEquals(ChampVAException.VISTALINK_FAULT_EXCEPTION, resp.getHeaderString("Exception"));
	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	}
    }

    /**
     * Test that getCodeResult(code) returns Internal Server Error when
     * champVAPayDAO.getCodeResult(codeSystem, code) raises a
     * ChampVAFoundationsException
     */
    @Test
    public void testThatGetCodeResultReturnsInternalServerErrorOnFoundationsException() {
	FoundationsException fe = new FoundationsException("BOOM!");
	this.cr.codeSystem = "NDC";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, NDC)).thenThrow(new ChampVAFoundationsException(fe));

	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);

	    Response resp = cr.getCodeResult(NDC);

	    assertNotNull("resp is null.", resp);
	    assertEquals(Response.Status.INTERNAL_SERVER_ERROR.getStatusCode(), resp.getStatus());
	    assertEquals(ChampVAException.FOUNDATIONS_EXCEPTION, resp.getHeaderString("Exception"));
	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	}
    }

    /**
     * Test that getCodeResult(code) returns Internal Server Error when
     * champVAPayDAO.getCodeResult(codeSystem, code) raises a
     * ChampVANamingException
     */
    @Test
    public void testThatGetCodeResultReturnsInternalServerErrorOnNamingException() {
	NamingException ne = new NamingException("BOOM!");
	this.cr.codeSystem = "NDC";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, NDC)).thenThrow(new ChampVANamingException(ne));

	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);

	    Response resp = cr.getCodeResult(NDC);

	    assertNotNull("resp is null.", resp);
	    assertEquals(Response.Status.INTERNAL_SERVER_ERROR.getStatusCode(), resp.getStatus());
	    assertEquals(ChampVAException.NAMING_EXCEPTION, resp.getHeaderString("Exception"));
	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	}
    }

    /**
     * Test that getCodeResult(code) returns Internal Server Error when
     * champVAPayDAO.getCodeResult(codeSystem, code) raises a
     * ChampVAResourceException
     */
    @Test
    public void testThatGetCodeResultReturnsInternalServerErrorOnResourceException() {
	ResourceException re = new ResourceException("BOOM!");
	this.cr.codeSystem = "NDC";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, NDC)).thenThrow(new ChampVAResourceException(re));

	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);

	    Response resp = cr.getCodeResult(NDC);

	    assertNotNull("resp is null.", resp);
	    assertEquals(Response.Status.INTERNAL_SERVER_ERROR.getStatusCode(), resp.getStatus());
	    assertEquals(ChampVAException.RESOURCE_EXCEPTION, resp.getHeaderString("Exception"));
	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	}
    }

    /**
     * Test that getCodeResult(code) returns Internal Server Error when
     * champVAPayDAO.getCodeResult(codeSystem, code) raises a
     * ChampVAJsonProcessingException
     */
    @Test
    public void testThatGetCodeResultReturnsInternalServerErrorOnJsonProcessingException() {
	ICodeResult ecr = new CodeResult(NDC_RESULT);
	JsonProcessingException jpe = new JsonGenerationException("BOOM!");
	this.cr.codeSystem = "NDC";
	try {
	    IChampVAPayDAO mockedDAO = mock(ChampVAPayDAO.class);
	    when(mockedDAO.getCodeResult(this.cr.codeSystem, NDC)).thenReturn(ecr);
	    ObjectMapper mockedMapper = mock(ObjectMapper.class);
	    when(mockedMapper.writeValueAsString(ecr)).thenThrow(jpe);
	    assertNotNull("cr failed to initialize.", this.cr);

	    cr.setChampVAPayDAO(mockedDAO);
	    cr.setMapper(mockedMapper);

	    Response resp = cr.getCodeResult(NDC);

	    assertNotNull("resp is null.", resp);
	    assertEquals(Response.Status.INTERNAL_SERVER_ERROR.getStatusCode(), resp.getStatus());
	    assertEquals(ChampVAException.JSON_PROCESSING_EXCEPTION, resp.getHeaderString("Exception"));
	} catch (ChampVAException e) {
	    fail("failed with ChampVAException: " + e.getMessage());
	} catch (JsonProcessingException e) {
	    fail("failed with JsonProcessingException: " + e.getMessage());
	}
    }

}
