/**
 * The package gov.va.med.mbcp.champva.dao.impl contains classes implementing the Interfaces for retrieval
 * of CHAMPVA Pay Results in the target VistA instance.
 * 
 * Including the unit tests in the same package as the classes under test allows test cases access to
 * friendly and protected members, without breaking encapsulation.
 * 
 */
package gov.va.med.mbcp.champva.dao.impl;

import static gov.va.med.mbac.vlh.impl.TestVistaLinkHelper.INST_MAP_NOT_FOUND_EXCEPTION;
import static gov.va.med.mbac.vlh.impl.TestVistaLinkHelper.INST_MAP_NOT_INIT_EXCEPTION;
import static gov.va.med.mbac.vlh.impl.TestVistaLinkHelper.NAMING_EXCEPTION;
import static gov.va.med.mbac.vlh.impl.TestVistaLinkHelper.RESOURCE_EXCEPTION;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import javax.resource.ResourceException;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.BeansException;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import gov.va.med.exception.FoundationsException;
import gov.va.med.mbac.vlh.common.IVistaLinkHelper;
import gov.va.med.mbac.vlh.impl.TestVistaLinkHelper;
import gov.va.med.mbcp.champva.dao.common.IChampVAPayDAO;
import gov.va.med.mbcp.champva.domain.common.ICodeResult;
import gov.va.med.mbcp.champva.domain.impl.ChampVAException;
import gov.va.med.mbcp.champva.domain.impl.CodeResult;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnection;
import gov.va.med.vistalink.adapter.record.VistaLinkFaultException;
import gov.va.med.vistalink.rpc.RpcRequest;
import gov.va.med.vistalink.rpc.RpcResponse;

/**
 * Test class for ChampVAPayDAO.
 * 
 * @author PII
 * @version 1.0.0
 */
public class ChampVAPayDAOTest {

    private static final String RPC_NAME = "CHSRVLKP VALIDCD";

    private static final String APP_PROXY_NAME = "CHAMPVAPAY,APPLICATION PROXY";

    private static final String SITE_ID = "500";

    private static final String NO_RESULT = "No";

    private static final String YES_RESULT = "Yes";

    private static final String CONDITIONAL_RESULT = "Yes^I can't love ya 'cause your feet's too BIG!";

    private static final String ICD9 = "253.8"; // conditional

    private static final String NDC = "55700-0373-30"; // "No"

    private static final String CPT = "98941"; // "Yes"

    private ClassPathXmlApplicationContext context = null;

    @Before
    public void setUp() throws Exception {
	this.context = new ClassPathXmlApplicationContext("/champvaTestBeans.xml");
    }

    @After
    public void tearDown() throws Exception {
	this.context.close();
	this.context = null;
    }

    /**
     * Tests for Spring DI
     */

    /**
     * Test lazy instantiation of vistaLinkHelper
     */
    @Test
    public final void testVistaLinkHelperInjection() {
	IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");
	assertNotNull("vistaLinkHelper failed to instantiate.", vistaLinkHelper);
	assertTrue("vistaLinkHelper instance of TestVistaLinkHelper.",
		(vistaLinkHelper instanceof TestVistaLinkHelper));
    }

    /**
     * Test lazy instantiation of champVAPayDAO, as well as getAppProxyName()
     * and getSiteId() methods
     */
    @Test
    public final void testChampVAPayDAOInjection() {
	IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	assertNotNull("champVAPayDAO failed to instantiate.", champVAPayDAO);
	assertEquals(APP_PROXY_NAME, ((ChampVAPayDAO) champVAPayDAO).getAppProxyName());
	assertEquals(SITE_ID, ((ChampVAPayDAO) champVAPayDAO).getSiteId());
    }

    /**
     * *** Tests for getCodeResult method ***
     */

    /**
     * Verify that getCodeResult() calls
     * vistaLinkConnection.executeRPC(rpcRequest) once
     * rpcRequest.setRpcName("CHSRVLKP VALIDCD") once rpcRequest.clearParams()
     * once rpcResponse.getResults() once
     */
    @Test
    public void verifyGetCodeResultsCalls() {
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");

	    RpcRequest mockedReq = ((TestVistaLinkHelper) vistaLinkHelper).getMockedReq();
	    VistaLinkConnection mockedVlc = ((TestVistaLinkHelper) vistaLinkHelper).getMockedVlc();
	    RpcResponse mockedResp = ((TestVistaLinkHelper) vistaLinkHelper).getMockedResp();
	    when(mockedResp.getResults()).thenReturn(CONDITIONAL_RESULT);
	    when(mockedVlc.executeRPC(mockedReq)).thenReturn(mockedResp);

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("ICD-9", ICD9);
	    assertNotNull("champVAPayDAO.getCodeResult(\"ICD9\", \"" + ICD9 + "\") returned void", codeResult);

	    verify(mockedVlc, times(1)).executeRPC(mockedReq);
	    verify(mockedReq, times(1)).setRpcName(RPC_NAME);
	    verify(mockedReq, times(1)).clearParams();
	    verify(mockedResp, times(1)).getResults();

	} catch (FoundationsException e) {
	    fail(e.getMessage());
	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    fail(e.getMessage());
	}
    }

    /**
     * Verify that getCodeResult() returns valid CodeResult when RPC call result
     * is valid.
     * 
     */
    @Test
    public void verifyThatGetCodeResultReturnsValidCodeResult() {
	ICodeResult expRes = new CodeResult(NO_RESULT);
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");

	    RpcRequest mockedReq = ((TestVistaLinkHelper) vistaLinkHelper).getMockedReq();
	    VistaLinkConnection mockedVlc = ((TestVistaLinkHelper) vistaLinkHelper).getMockedVlc();
	    RpcResponse mockedResp = ((TestVistaLinkHelper) vistaLinkHelper).getMockedResp();
	    when(mockedResp.getResults()).thenReturn(NO_RESULT);
	    when(mockedVlc.executeRPC(mockedReq)).thenReturn(mockedResp);

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("CPT", CPT);
	    assertNotNull("champVAPayDAO.getCodeResult(\"CPT\", \"" + CPT + "\") returned void", codeResult);

	    assertEquals(expRes, codeResult);
	} catch (FoundationsException e) {
	    fail(e.getMessage());
	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    fail(e.getMessage());
	}
    }

    /**
     * Verify that getCodeResult() returns null CodeResult when getConnection()
     * returns null connection.
     * 
     */
    @Test
    public void verifyThatGetCodeResultReturnsNullCallResultWhenGetConnectionFails() {
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");
	    ((TestVistaLinkHelper) vistaLinkHelper).setMockedVlc(null);

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("NDC", NDC);
	    assertNull("champVAPayDAO.getCodeResult(\"NDC\", \"" + NDC + "\") returned void", codeResult);

	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    fail(e.getMessage());
	}
    }

    /**
     * Verify that getCodeResult() throws ChampVAResourceException when RPC call
     * returns empty String
     * 
     */
    @Test
    public void verifyThatGetCodeResultThrowsChampVAResourceExceptionWhenRpcReturnsEmptyString() {

	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");

	    RpcRequest mockedReq = ((TestVistaLinkHelper) vistaLinkHelper).getMockedReq();
	    VistaLinkConnection mockedVlc = ((TestVistaLinkHelper) vistaLinkHelper).getMockedVlc();
	    RpcResponse mockedResp = ((TestVistaLinkHelper) vistaLinkHelper).getMockedResp();
	    when(mockedResp.getResults()).thenReturn("");
	    when(mockedVlc.executeRPC(mockedReq)).thenReturn(mockedResp);

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("ICD-9", ICD9);
	    assertNotNull("champVAPayDAO.getCodeResult(\"ICD9\", \"253.8\") returned void", codeResult);

	} catch (FoundationsException e) {
	    fail(e.getMessage());
	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    assertEquals(ChampVAException.RESOURCE_EXCEPTION, e.getErrorCode());
	}
    }

    /**
     * Verify that getCodeResult() throws ChampVAFoundationsException when
     * getConnection throws InstitutionMapNotInitializedException
     */
    @Test
    public void verifyThatGetCodeResultRethrowsFoundationsExceptionWhenInstMapNotInit() {
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");
	    ((TestVistaLinkHelper) vistaLinkHelper).setConnectionFailCondition(INST_MAP_NOT_INIT_EXCEPTION);
	    ((TestVistaLinkHelper) vistaLinkHelper).setMockedVlc(null);

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("ICD-9", ICD9);
	    assertNotNull("champVAPayDAO.getCodeResult(\"ICD9\", \"253.8\") returned void", codeResult);

	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    assertEquals(ChampVAException.FOUNDATIONS_EXCEPTION, e.getErrorCode());
	}
    }

    /**
     * Verify that getCodeResult() throws ChampVAException when getConnection
     * throws InstitutionMappingNotFoundException
     */
    @Test
    public void verifyThatGetCodeResultThrowsChampVAExceptionWhenInstMapNotFound() {
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");
	    ((TestVistaLinkHelper) vistaLinkHelper).setConnectionFailCondition(INST_MAP_NOT_FOUND_EXCEPTION);
	    ((TestVistaLinkHelper) vistaLinkHelper).setMockedVlc(null);

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("ICD-9", ICD9);
	    assertNotNull("champVAPayDAO.getCodeResult(\"ICD9\", \"253.8\") returned void", codeResult);

	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    assertEquals(ChampVAException.FOUNDATIONS_EXCEPTION, e.getErrorCode());
	}
    }

    /**
     * Verify that getCodeResult() throws ChampVAException when getConnection
     * throws NamingException
     */
    @Test
    public void verifyThatGetCodeResultThrowsChampVAExceptionWhenNamingException() {
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");
	    ((TestVistaLinkHelper) vistaLinkHelper).setConnectionFailCondition(NAMING_EXCEPTION);
	    ((TestVistaLinkHelper) vistaLinkHelper).setMockedVlc(null);

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("ICD-9", ICD9);
	    assertNotNull("champVAPayDAO.getCodeResult(\"ICD9\", \"253.8\") returned void", codeResult);

	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    assertEquals(ChampVAException.NAMING_EXCEPTION, e.getErrorCode());
	}
    }

    /**
     * Verify that getCodeResult() throws ChampVAException when getConnection
     * throws ResourceException
     */
    @Test
    public void verifyThatGetCodeResultThrowsChampVAExceptionWhenResourceException() {
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");
	    ((TestVistaLinkHelper) vistaLinkHelper).setConnectionFailCondition(RESOURCE_EXCEPTION);
	    ((TestVistaLinkHelper) vistaLinkHelper).setMockedVlc(null);

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("ICD-9", ICD9);
	    assertNotNull("champVAPayDAO.getCodeResult(\"ICD9\", \"253.8\") returned void", codeResult);

	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    assertEquals(ChampVAException.RESOURCE_EXCEPTION, e.getErrorCode());
	}
    }

    /**
     * Verify that getCodeResult() throws ChampVAException when executeRPC
     * throws VistaLinkFaultException
     */
    @Test
    public void verifyThatGetCodeResultThrowsChampVAExceptionWhenVistaLinkFaultException() {
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");

	    RpcRequest mockedReq = ((TestVistaLinkHelper) vistaLinkHelper).getMockedReq();
	    VistaLinkConnection mockedVlc = ((TestVistaLinkHelper) vistaLinkHelper).getMockedVlc();
	    RpcResponse mockedResp = ((TestVistaLinkHelper) vistaLinkHelper).getMockedResp();
	    when(mockedResp.getResults()).thenReturn(YES_RESULT);
	    when(mockedVlc.executeRPC(mockedReq)).thenThrow(new VistaLinkFaultException("BAZINGA!"));

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("ICD-9", ICD9);
	    assertNotNull("champVAPayDAO.getCodeResult(\"ICD9\", \"253.8\") returned void", codeResult);

	} catch (VistaLinkFaultException e) {
	    fail(e.getMessage());
	} catch (FoundationsException e) {
	    fail(e.getMessage());
	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    assertEquals(ChampVAException.VISTALINK_FAULT_EXCEPTION, e.getErrorCode());
	}
    }

    /**
     * Verify that getCodeResult() throws ChampVAException when connection.close
     * throws ResourceException
     */
    @Test
    public void verifyGetCodeResultThrowsResourceExceptionWhenConnectionCloseFails() {
	try {

	    IChampVAPayDAO champVAPayDAO = (ChampVAPayDAO) context.getBean("champVAPayDAO");
	    IVistaLinkHelper vistaLinkHelper = (TestVistaLinkHelper) context.getBean("vistaLinkHelper");

	    RpcRequest mockedReq = ((TestVistaLinkHelper) vistaLinkHelper).getMockedReq();
	    VistaLinkConnection mockedVlc = ((TestVistaLinkHelper) vistaLinkHelper).getMockedVlc();
	    RpcResponse mockedResp = ((TestVistaLinkHelper) vistaLinkHelper).getMockedResp();
	    when(mockedResp.getResults()).thenReturn(NO_RESULT);
	    when(mockedVlc.executeRPC(mockedReq)).thenReturn(mockedResp);
	    doThrow(new ResourceException("URP!")).when(mockedVlc).close();

	    ICodeResult codeResult = champVAPayDAO.getCodeResult("ICD-9", ICD9);
	    assertNotNull("champVAPayDAO.getCodeResult(\"ICD9\", \"253.8\") returned void", codeResult);

	} catch (FoundationsException e) {
	    fail(e.getMessage());
	} catch (BeansException e) {
	    fail(e.getMessage());
	} catch (ResourceException e) {
	    fail(e.getMessage());
	} catch (ChampVAException e) {
	    assertEquals(ChampVAException.RESOURCE_EXCEPTION, e.getErrorCode());
	}
    }

}
