/**
 * The package gov.va.med.mbcp.champva.rest contains the RESTful Web Application & Resource classes for
 * retrieval of Code Results from the target VistA instance.
 * 
 */
package gov.va.med.mbcp.champva.rest;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.apache.log4j.Logger;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import gov.va.med.mbcp.champva.dao.common.IChampVAPayDAO;
import gov.va.med.mbcp.champva.dao.impl.ChampVAPayDAO;
import gov.va.med.mbcp.champva.domain.common.ICodeResult;
import gov.va.med.mbcp.champva.domain.impl.ChampVAException;
import gov.va.med.mbcp.champva.domain.impl.ChampVAJsonException;

/**
 * The ChampVAResource Class implements the resource endpoint for retrieval of
 * the Code Result for a given Diagnosis, Procedure, or National Drug Code from
 * the VistA system at the Health Administration Center (HAC) in Denver
 * 
 * @author PII
 * @version 1.0.0
 */
@Path("/code-systems/{code-system}")
public class ChampVAResource {

    /** A reference to the HTTP request sent from the client application */
    @Context
    HttpServletRequest request;

    /** The Code System for the Request. */
    @PathParam("code-system")
    String codeSystem;

    /**
     * An instance of the PersonDAO class which communicates with VistA through
     * VistALink
     */
    private IChampVAPayDAO champVAPayDAO;

    /** An instance of the Jackson parser */
    private ObjectMapper mapper;

    /** An instance of Logger utility */
    private final static Logger LOGGER = Logger.getLogger(ChampVAResource.class);

    private ClassPathXmlApplicationContext appContext;

    /**
     * Default Constructor for the ChampVAResource class. Construction is
     * handled by the JAX-RS container. The constructor will then create a
     * Spring Application Context which will manage the lazy instantiation of
     * the DAO and its dependent classes as defined in the
     * src/main/resources/champvaBeans.xml file.
     */
    public ChampVAResource() {
	super();

	this.appContext = new ClassPathXmlApplicationContext("/champvaBeans.xml");
	this.champVAPayDAO = (ChampVAPayDAO) appContext.getBean("champVAPayDAO");
	this.mapper = (ObjectMapper) appContext.getBean("objectMapper");
	this.appContext.close();
    }

    /**
     * This is the HTTP GET method for the Web Service. The fully-specified URI
     * is: GET /champva/v1/code-systems/{code-system}/codes/{code}
     * 
     * @param code - String the code to be evaluated.
     * @return an HTTP Response
     */
    @GET
    @Path("/codes/{code}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getCodeResult(@PathParam("code") String code) {

	ICodeResult codeResult;
	String json;

	LOGGER.debug("entered getCodeResult(" + this.codeSystem + ", " + code + ")");

	try {

	    codeResult = champVAPayDAO.getCodeResult(this.codeSystem, code);

	    // Format the response and send to the resource server
	    json = mapper.writeValueAsString(codeResult);

	    LOGGER.debug("getUserProperties() response: " + json);

	    return Response.status(Response.Status.OK).entity(json).build();

	} catch (JsonProcessingException jpe) {
	    ChampVAException ce = new ChampVAJsonException(jpe);
	    return Response.status(Response.Status.INTERNAL_SERVER_ERROR).header("Exception", ce.getErrorCode())
		    .build();
	} catch (ChampVAException ce) {
	    return Response.status(Response.Status.INTERNAL_SERVER_ERROR).header("Exception", ce.getErrorCode())
		    .build();
	}

    }

    /**
     * Setter for the champVAPayDAO field.
     * 
     * @param champVAPayDAO
     *            the champVAPayDAO to set
     */
    void setChampVAPayDAO(IChampVAPayDAO champVAPayDAO) {
	this.champVAPayDAO = champVAPayDAO;
    }

    /**
     * Setter for the mapper field.
     * 
     * @param mapper
     *            the mapper to set
     */
    void setMapper(ObjectMapper mapper) {
	this.mapper = mapper;
    }

}
