/**
 * The package gov.va.med.mbcp.champva.dao.impl contains the public Implementation for 
 * retrieval of CHAMPVA CodeResults from the target VistA instance.
 * 
 */
package gov.va.med.mbcp.champva.dao.impl;

import javax.naming.NamingException;
import javax.resource.ResourceException;

import gov.va.med.exception.FoundationsException;
import gov.va.med.mbac.vlh.common.IVistaLinkHelper;
import gov.va.med.mbcp.champva.dao.common.IChampVAPayDAO;
import gov.va.med.mbcp.champva.domain.common.ICodeResult;
import gov.va.med.mbcp.champva.domain.impl.ChampVAException;
import gov.va.med.mbcp.champva.domain.impl.ChampVAFoundationsException;
import gov.va.med.mbcp.champva.domain.impl.ChampVANamingException;
import gov.va.med.mbcp.champva.domain.impl.ChampVAResourceException;
import gov.va.med.mbcp.champva.domain.impl.ChampVAVLFaultException;
import gov.va.med.mbcp.champva.domain.impl.CodeResult;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnection;
import gov.va.med.vistalink.adapter.record.VistaLinkFaultException;
import gov.va.med.vistalink.rpc.RpcRequest;
import gov.va.med.vistalink.rpc.RpcResponse;

/**
 * The ChampVAPayDAO Class implements the Data Access Object for the CHAMPVA Pay Service.
 * 
 * @author PII
 * @version 1.0.0
 */
public class ChampVAPayDAO implements IChampVAPayDAO {

    private static final String RPC_NAME = "CHSRVLKP VALIDCD";
    private IVistaLinkHelper vlHelper;
    private RpcRequest vReq;
    private String siteId;
    private String appProxyName;

    /**
     * This class is lazily constructed and managed as a singleton by Spring DI. The 
     * VistaLinkHelper is injected into the constructor, while the siteId and 
     * appProxyName are injected into their respective setters.
     * 
     * @param vistaLinkHelper - the VistaLinkHelper to be injected
     */
    public ChampVAPayDAO(IVistaLinkHelper vistaLinkHelper) {
	super();

	this.vlHelper = vistaLinkHelper;
	this.vReq = vistaLinkHelper.getRpcRequest();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * gov.va.med.mbcp.champva.dao.common.IChampVAPayDAO#getCodeResult(java.lang
     * .String, java.lang.String)
     */
    @Override
    public ICodeResult getCodeResult(String codeSystem, String code) throws ChampVAException {

	VistaLinkConnection connection = null;
	ICodeResult codeResult = null;
	String crString;

	try {

	    // set-up the RPC Request
	    vReq.setRpcName(RPC_NAME);
	    vReq.clearParams();

	    // get the VL connection
	    connection = vlHelper.getAppProxyConnection(siteId, appProxyName);

	    if (connection != null) {

		// issue the RPC
		RpcResponse vResp = connection.executeRPC(vReq);

		crString = vResp.getResults();

		if (crString.equals("")) {
		    throw new ResourceException("Invalid RPC Call Result.");
		}

		codeResult = new CodeResult(crString);

	    }

	    return codeResult;

	} catch (VistaLinkFaultException vlfe) {
	    throw new ChampVAVLFaultException(vlfe);
	} catch (FoundationsException fe) {
	    throw new ChampVAFoundationsException(fe);
	} catch (NamingException ne) {
	    throw new ChampVANamingException(ne);
	} catch (ResourceException re) {
	    throw new ChampVAResourceException(re);
	} finally {
	    if (connection != null) {
		try {
		    connection.close();
		} catch (ResourceException re) {
		    throw new ChampVAResourceException(re);
		}
	    }
	}
    }

    /**
     * @return the siteId
     */
    public String getSiteId() {
        return siteId;
    }

    /**
     * @param siteId the siteId to set
     */
    public void setSiteId(String siteId) {
        this.siteId = siteId;
    }

    /**
     * @return the appProxyName
     */
    public String getAppProxyName() {
        return appProxyName;
    }

    /**
     * @param appProxyName the appProxyName to set
     */
    public void setAppProxyName(String appProxyName) {
        this.appProxyName = appProxyName;
    }

}
