﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Threading;
using System.IO;
using System.Configuration;
using System.Net;
using System.IO.Compression;
using System.DirectoryServices.AccountManagement;

namespace AHOBPR_PDF_Utility
{
    public partial class PDFUtility : Form
    {
        private int _workerCount = 0;
        private Logger _logger = new Logger();
        private string _path = string.Empty;
        private string _csvFileName = string.Empty;
        private string _htmlFileName = string.Empty;
        private string _csvEdipiFileName = string.Empty;
        private string _htmlEdipiFileName = string.Empty;
        private string _renameBatFileName = string.Empty;
        private string _renameShFileName = string.Empty;
        private List<string> _userIds;

        public PDFUtility()
        {
            InitializeComponent();

            ServicePointManager.Expect100Continue = true;
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls11 | SecurityProtocolType.Tls12;

            backgroundWorker.WorkerReportsProgress = true;
            backgroundWorker.WorkerSupportsCancellation = true;
            backgroundWorker.DoWork += new DoWorkEventHandler(backgroundWorker_DoWork);
            backgroundWorker.ProgressChanged += new ProgressChangedEventHandler(backgroundWorker_ProgressChanged);
        }

        private void PDFUtility_Shown(object sender, System.EventArgs e)
        {
            if (IsUserAuthenticated())
            {
                _logger.StartLog();
                _logger.LogMessageToFile("AHOBPR PDF Utility started...");

                DateTime today = DateTime.Today;
                DateTime firstDayThisMonth = new DateTime(today.Year, today.Month, 1);
                dateTimePickerFromDate.Text = firstDayThisMonth.AddMonths(-1).ToShortDateString();
                dateTimePickerToDate.Text = firstDayThisMonth.AddDays(-1).ToShortDateString();

                string url = ConfigurationManager.AppSettings["url"] + "/lastpdfrundate";
                string lastRunDate = GetDataFromRestService(url);
                lastRunDate = lastRunDate.Replace("LastPdfRunDateResult", string.Empty).Replace("\"", string.Empty);
                lastRunDate = lastRunDate.Replace("{", string.Empty).Replace("}", string.Empty);
                if (!string.IsNullOrEmpty(lastRunDate))
                {
                    labelLastRun.Text = "Last Time PDFs Compiled:  " + lastRunDate.Substring(1, lastRunDate.Length - 1).Replace("\\", string.Empty);
                }
            }
            else
            {
                this.groupBoxDateRange.Visible = false;
                this.groupBoxFolder.Visible = false;
                this.progressBar.Visible = false;
                this.buttonBrowse.Visible = false;
                this.buttonGeneratePDF.Visible = false;
                this.labelStatus.Visible = false;
                this.labelLastRun.Text = " You are not authorized to use this utility.  \n\r You need to have \"" + ConfigurationManager.AppSettings["userRole"] + "\" role in AHOBPR Clinical Portal. \n\r Please contact  your system administator.";
                this.buttonClose.Top = buttonClose.Top - 110;
                this.buttonClose.Left = buttonClose.Left - 200;
                this.Height = this.Height - 180;
                this.Width = this.Width + 120;
            }

        }


        private void backgroundWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            _workerCount++;

            if (_workerCount == 1)
            {
                int total = _userIds.Count;
                int countProcessed = 0;
                foreach (string userId in _userIds)
                {
                    countProcessed = countProcessed + 1;
                    if (string.IsNullOrEmpty(userId))
                    {
                        backgroundWorker.ProgressChanged += new ProgressChangedEventHandler(backgroundWorker_ProgressChanged);
                        backgroundWorker.ReportProgress(countProcessed / total * 100);
                    }
                    else
                    {
                        byte[] pdfData = GetPdf(userId);
                        string metaData = GetMetaData(userId);
                        metaData = metaData.Replace("{\"PdfMetaDataResult\":\"", "");
                        metaData = metaData.Substring(0, metaData.Length - 2);

                        ////TLB: Old logic is a problem when the user/registrant has parenthesis in their name
                        //string pdfNewName = metaData.Substring(metaData.IndexOf("(") + 3, metaData.IndexOf(")") - metaData.IndexOf("(") - 5);
                        //string[] fileNameParts = pdfNewName.Split('_');
                        //string pdfFileName = fileNameParts[2] + "_" + fileNameParts[4] + "_" + fileNameParts[5];
                        
                        string[] metaDataParts = metaData.Split(',');

                        string pdfNewName = metaDataParts[0] + "_" + metaDataParts[1] + "_" + metaDataParts[2] + "_" + metaDataParts[3] + "_" + metaDataParts[4] + "_" + metaDataParts[5] + ".pdf";
                        string pdfFileName = metaDataParts[2] + "_" + metaDataParts[4] + "_" + metaDataParts[5] + ".pdf";

                        bool pdfCreated = CreatePdfFile(_path + "\\" + pdfFileName, pdfData);

                        //remove name and ssn for edipicsvfile
                        string edipiMetaData = metaDataParts[0] + "," + metaDataParts[1] + "," + metaDataParts[2] + "," + metaDataParts[3] + "," +
                                               metaDataParts[4] + "," + metaDataParts[5] + "," + pdfFileName + "," + metaDataParts[7] + "," +
                                               metaDataParts[8] + "," + metaDataParts[9] + "," + metaDataParts[10];

                        if (pdfCreated)
                        {
                            _logger.LogMessageToFile("File Created: " + pdfFileName);
                            AppendToFile(_csvFileName, metaData.Replace("\\", ""));
                            AppendToFile(_renameBatFileName, "move " + pdfFileName + " " + pdfNewName);
                            AppendToFile(_renameShFileName, "mv " + pdfFileName + " " + pdfNewName);
                            AppendNewLineToHtmlFile(_htmlFileName, metaData, pdfNewName);
                            //append to Edipi csv/html file. Name and SSN are not a part of the file name.
                            AppendToFile(_csvEdipiFileName, edipiMetaData);
                            AppendNewLineToHtmlFile(_htmlEdipiFileName, metaData, pdfFileName);

                            //Save interface log
                            string url = ConfigurationManager.AppSettings["url"] + "/saveinterfacelog/" + userId;
                            string filenamejson = "{\"value\":\"" + pdfFileName + "\"}";
                            SaveInterfaceLog(url, userId, filenamejson);

                            backgroundWorker.ReportProgress(countProcessed * 99 / total);
                        }
                    }
                }

                AddClosingTagsForHtmlFile();

                //Create zip file
                DirectoryInfo directorySelected = new DirectoryInfo(_path);
                _logger.LogMessageToFile("Creating zip file...");
                CreateZipFile();
                CleanupDirectory();
                _logger.LogMessageToFile("Successfully completed.");
                backgroundWorker.ReportProgress(100);
            }

        }

        private bool IsUserAuthenticated()
        {
            bool authenticated = false;

            // find currently logged in user
            UserPrincipal user = UserPrincipal.Current;
            string username = user.SamAccountName;

            string url = ConfigurationManager.AppSettings["url"]
                        + "/userrole/" + username;

            string userRole = GetDataFromRestService(url);
            if (!string.IsNullOrEmpty(userRole) && !string.IsNullOrEmpty(ConfigurationManager.AppSettings["userRole"]))
            {
                authenticated = userRole.Contains(ConfigurationManager.AppSettings["userRole"]);
            }

            return authenticated;
        }

        public bool CreatePdfFile(string fileName, byte[] _bites)
        {
            try
            {
                //don't create empty PDF
                if  (_bites.Length > 0)
                { 
                // Open file for reading
                FileStream _FileStream =
                   new System.IO.FileStream(fileName, System.IO.FileMode.Create,
                                            System.IO.FileAccess.Write);

                // Writes a block of bytes to this stream using data from
                // a byte array.
                _FileStream.Write(_bites, 0, _bites.Length);

                // close file stream
                _FileStream.Close();

                return true;
                }
                else
                {
                    return false;
                }
            }
            catch (Exception ex)
            {
                // Error
                _logger.LogMessageToFile("AHOBPR PDF Utility: Error: " + ex.Message);
            }

            // error occured, return false
            return false;
        }

        private void backgroundWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            // Change the value of the ProgressBar to the BackgroundWorker progress.
            progressBar.Value = e.ProgressPercentage;
            // Set the text.
            labelStatus.Text = "Progress: " + e.ProgressPercentage.ToString() + "% has completed.";
        }


        private void buttonBrowse_Click(object sender, EventArgs e)
        {
            if (folderBrowserDialog.ShowDialog() == DialogResult.OK)
            {
                textBoxPath.Text = folderBrowserDialog.SelectedPath;
            }

        }

        private void buttonGeneratePDF_Click(object sender, EventArgs e)
        {
            labelStatus.Text = "";
            if (string.IsNullOrEmpty(dateTimePickerFromDate.Text))
            {
                labelStatus.Text = "Please select a From Date";
                dateTimePickerFromDate.Focus();
            }
            else if (string.IsNullOrEmpty(dateTimePickerToDate.Text))
            {
                labelStatus.Text = "Please select a To Date";
                dateTimePickerToDate.Focus();
            }
            else if (string.IsNullOrEmpty(textBoxPath.Text))
            {
                labelStatus.Text = "Please select a file path";
                textBoxPath.Focus();
            }
            else if (Convert.ToDateTime(dateTimePickerToDate.Text) < Convert.ToDateTime(dateTimePickerFromDate.Text))
            {
                labelStatus.Text = "The To Date is less than the From Date. Please try again.";
                dateTimePickerToDate.Focus();
            }
            else if (!Directory.Exists(textBoxPath.Text))
            {
                labelStatus.Text = "Invalid path. Please try again.";
                textBoxPath.Focus();
            }
            else
            {
                // Create a sub directory. Delete it if already exists
                if (CreateSubDirectory())
                {
                    // Start the BackgroundWorker.
                    _workerCount = 0;
                    labelStatus.Text = "Get the list of registrants...";
                    _logger.LogMessageToFile(labelStatus.Text);
                    _userIds = GetUserIds();

                    DialogResult dialogResult = MessageBox.Show("There are " + _userIds.Count.ToString() + " PDFs found within the date range.  Do you want to compile PDFs?", "Total PDFs", MessageBoxButtons.YesNo, MessageBoxIcon.Information);

                    if (dialogResult == DialogResult.Yes)
                    {
                        if (backgroundWorker.IsBusy != true)
                        {
                            backgroundWorker.RunWorkerAsync();
                        }
                    }
                }
                else
                {
                    labelStatus.Text = "Error: Please close all the PDF related open files and try again.";
                    _logger.LogMessageToFile(labelStatus.Text);
                }
            }
        }

        private void CreateZipFile()
        {
            string dateRange = FormatDate(dateTimePickerFromDate.Text) + "-" + FormatDate(dateTimePickerToDate.Text);
            string zipFileName = textBoxPath.Text + @"\index_" + dateRange + ".zip";
            ZipFile.CreateFromDirectory(_path, zipFileName);
        }

                /// <summary>
        /// Create a sub directory. example:  AHOBPR_PDF_20160301-20160331
        /// </summary>
        private void CleanupDirectory()
        {

            string dateRange = FormatDate(dateTimePickerFromDate.Text) + "-" + FormatDate(dateTimePickerToDate.Text);
            string subDirectory = "AHOBPR_PDF_" + dateRange;
            _path = Path.Combine(textBoxPath.Text, subDirectory);
            if (Directory.Exists(_path))
            {
                try
                {
                    Directory.Delete(_path, true);
                }
                catch (Exception ex)
                {
                    _logger.LogErrorMessage(ex);
                }
            }
        }

        /// <summary>
        /// Create a sub directory. example:  AHOBPR_PDF_20160301-20160331
        /// </summary>
        private bool CreateSubDirectory()
        {
            bool result = true;

            string dateRange = FormatDate(dateTimePickerFromDate.Text) + "-" + FormatDate(dateTimePickerToDate.Text);
            string subDirectory = "AHOBPR_PDF_" + dateRange;
            _path = Path.Combine(textBoxPath.Text, subDirectory);
            string zipFile = textBoxPath.Text + @"\index_" + dateRange + ".zip";
            if (Directory.Exists(_path))
            {
                try
                {
                    Directory.Delete(_path, true);
                }
                catch (Exception ex)
                {
                    _logger.LogErrorMessage(ex);
                    return false;
                }
            }

            if (File.Exists(zipFile))
            {
                File.Delete(zipFile);
            }

            Directory.CreateDirectory(_path);

            _csvFileName = _path + "\\" + "index_" + dateRange + ".csv";
            _htmlFileName = _path + "\\" + "index_" + dateRange + ".html";
            _csvEdipiFileName = _path + "\\" + "index_Edipi_" + dateRange + ".csv";
            _htmlEdipiFileName = _path + "\\" + "index_Edipi_" + dateRange + ".html";
            _renameBatFileName =  _path + "\\" + "rename.bat";
            _renameShFileName = _path + "\\" + "rename.sh";

            if (!File.Exists(_csvFileName))
            {
                FileStream fileCvs = File.Create(_csvFileName);
                fileCvs.Close();
                string headerInfo = "Last,First,EDIPI,SSN,OBM Form,Participant Date,File Name,Earliest Deployment Start,Latest Deployment End,Deployment Countries,Service Status";
                AppendToFile(_csvFileName, headerInfo);
            }

            if (!File.Exists(_csvEdipiFileName))
            {
                FileStream fileEdipiCvs = File.Create(_csvEdipiFileName);
                fileEdipiCvs.Close();
                string headerInfoEdipi = "Last,First,EDIPI,SSN,OBM Form,Participant Date,File Name,Earliest Deployment Start,Latest Deployment End,Deployment Countries,Service Status";
                AppendToFile(_csvEdipiFileName, headerInfoEdipi);
            }

            if (!File.Exists(_htmlFileName))
            {
                FileStream fileHtml = File.Create(_htmlFileName);
                fileHtml.Close();
                AppendToFile(_htmlFileName, @"<!DOCTYPE HTML PUBLIC '-//W3C//DTD HTML 4.01 Transitional//EN'>");
                AppendToFile(_htmlFileName, @"<html>");
                AppendToFile(_htmlFileName, @"<head>");
                AppendToFile(_htmlFileName, @"<title>AHOBPR Participants " + dateRange + "</title>");
                AppendToFile(_htmlFileName, @"</head>");
                AppendToFile(_htmlFileName, @"<body>");
                AppendToFile(_htmlFileName, @"<h1>AHOBPR Participants " + dateRange + "</h1>");
                AppendToFile(_htmlFileName, @"<table border='1'>");
                AppendToFile(_htmlFileName, @"<tr><th scope='col'>Last</th><th scope='col'>First</th><th scope='col'>EDIPI</th><th scope='col'>SSN</th><th scope='col'>OMB Form</th><th scope='col'>Participant Date</th><th scope='col'>File Name</th><th scope='col'>Earliest Deployment Start</th><th scope='col'>Latest Deployment End</th><th scope='col'>Deployment Countries</th><th scope='col'>Service Status</th></tr>");
            }

            if (!File.Exists(_htmlEdipiFileName))
            {
                FileStream fileEdipiHtml = File.Create(_htmlEdipiFileName);
                fileEdipiHtml.Close();
                AppendToFile(_htmlEdipiFileName, @"<!DOCTYPE HTML PUBLIC '-//W3C//DTD HTML 4.01 Transitional//EN'>");
                AppendToFile(_htmlEdipiFileName, @"<html>");
                AppendToFile(_htmlEdipiFileName, @"<head>");
                AppendToFile(_htmlEdipiFileName, @"<title>AHOBPR Participants " + dateRange + "</title>");
                AppendToFile(_htmlEdipiFileName, @"</head>");
                AppendToFile(_htmlEdipiFileName, @"<body>");
                AppendToFile(_htmlEdipiFileName, @"<h1>AHOBPR Participants " + dateRange + "</h1>");
                AppendToFile(_htmlEdipiFileName, @"<table border='1'>");
                AppendToFile(_htmlEdipiFileName, @"<tr><th scope='col'>Last</th><th scope='col'>First</th><th scope='col'>EDIPI</th><th scope='col'>SSN</th><th scope='col'>OMB Form</th><th scope='col'>Participant Date</th><th scope='col'>File Name</th><th scope='col'>Earliest Deployment Start</th><th scope='col'>Latest Deployment End</th><th scope='col'>Deployment Countries</th><th scope='col'>Service Status</th></tr>");
            }

            if (!File.Exists(_renameBatFileName))
            {
                FileStream fileBatch = File.Create(_renameBatFileName);
                fileBatch.Close();
            }


            if (!File.Exists(_renameShFileName))
            {
                FileStream fileShell = File.Create(_renameShFileName);
                fileShell.Close();
            }

            return result;
        }

        private void AddClosingTagsForHtmlFile()
        {
            AppendToFile(_htmlFileName, @"</table>");
            AppendToFile(_htmlFileName, @"</body>");
            AppendToFile(_htmlFileName, @"</html>");

            AppendToFile(_htmlEdipiFileName, @"</table>");
            AppendToFile(_htmlEdipiFileName, @"</body>");
            AppendToFile(_htmlEdipiFileName, @"</html>");
        }

        private void AppendToFile(string fileName, string line)
        {
            using (StreamWriter writer = File.AppendText(fileName))
            {
                writer.WriteLine(line);
            }
        }

        private void AppendNewLineToHtmlFile(string fileName, string metaData, string pdfFileName)
        {
            string line = "<tr>";
            List<string> dataList = metaData.Split(',').ToList();
            foreach (string data in dataList)
            {
                if (data.Contains("HYPERLINK"))
                {
                    line = line + "<td><a target='_blank' href='./" + pdfFileName + "'>" + pdfFileName + "</a></td>";
                }
                else
                {
                    line = line + "<td>" + data + "</td>";
                }
            }
            line = line + "</tr>";

            AppendToFile(fileName, line);
        }

        /// <summary>
        /// Get the userIDs based on the data range
        /// </summary>
        /// <returns></returns>
        private List<string> GetUserIds()
        {
            List<string> result = new List<string>();

            string url = ConfigurationManager.AppSettings["url"]
                        + "/pdfregistrantlist/" + FormatDate(dateTimePickerFromDate.Text)
                        + "/" + FormatDate(dateTimePickerToDate.Text);

            _logger.LogMessageToFile("url: " + url);

            string serviceResponse = GetDataFromRestService(url);
            serviceResponse = serviceResponse.Replace("PdfRegistrantListResult", string.Empty).Replace("\"", string.Empty);
            serviceResponse = serviceResponse.Replace("[", string.Empty).Replace("]", string.Empty).Replace(":", string.Empty);
            serviceResponse = serviceResponse.Replace("{", string.Empty).Replace("}", string.Empty).Replace(" ", string.Empty);
            string[] userIds = serviceResponse.Split(',');
            result = userIds.ToList();

            return result;
        }


        private string GetMetaData(string userId)
        {
            string metaData = string.Empty;

            string url = ConfigurationManager.AppSettings["url"]
                        + "/pdfmetadata/" + userId;

            metaData = GetDataFromRestService(url);

            return metaData;
        }

        /// <summary>
        /// Get PDF by user ID
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        private byte[] GetPdf(string userId)
        {
            _logger.LogMessageToFile("AHOBPR PDF Utility getting PDF for: " + userId);
            byte[] bites = null;
            try
            {
                String secretKey = System.Configuration.ConfigurationManager.AppSettings["secretKey"];
                HttpWebRequest getRequest = (HttpWebRequest)WebRequest.Create(ConfigurationManager.AppSettings["url"] + "/download/responsePdf/" + userId);
                getRequest.ContentType = "application/json";
                getRequest.Accept = "application/json";
                getRequest.Method = "GET";
                getRequest.Headers.Add("Authorization", "Basic " + secretKey);

                HttpWebResponse getResponse = (HttpWebResponse)getRequest.GetResponse();
                String result = new StreamReader(getResponse.GetResponseStream()).ReadToEnd();
                int startIndex = result.IndexOf(":[");
                if (0 < startIndex)
                {
                    result = result.Substring(startIndex + 2, result.Length - 2 - (startIndex + 2));
                    String[] tokens = result.Split(new char[] { ',' });
                    bites = new byte[tokens.Length];
                    for (int i = 0; i < tokens.Length; i++)
                    {
                        bites[i] = byte.Parse(tokens[i]);
                    }
                }
            }
            catch (Exception ex)
            {
                _logger.LogMessageToFile("AHOBPR PDF Utility: Unable to get PDF for user ID = " + userId + ". Error: " + ex.Message);
                bites = null;
            }
            return bites;
        }


        /// <summary>
        /// Format date to YYYYMMDD
        /// </summary>
        /// <param name="dateText"></param>
        /// <returns></returns>
        private string FormatDate(string dateText)
        {
            string result = string.Empty;

            try
            {
                DateTime dateTime = Convert.ToDateTime(dateText);
                result = dateTime.Year.ToString() + GetTwoDigits(dateTime.Month.ToString()) + GetTwoDigits(dateTime.Day.ToString());
            }
            catch
            {
                labelStatus.Text = "Invalid Date.";
            }
            return result;
        }

        private string GetTwoDigits(string text)
        {
            return text.Length == 1 ? "0" + text : text;
        }

        private string GetDataFromRestService(string url)
        {
            string jsonResponse = string.Empty;
            WebRequest request = WebRequest.Create(url);

            int AhobprWebServiceTimeout;
            bool result = int.TryParse(ConfigurationManager.AppSettings["AhobprWebServiceTimeout"], out AhobprWebServiceTimeout);
            if (result)
            {
                request.Timeout = AhobprWebServiceTimeout;
            }
            else
            {
                request.Timeout = 350000; // <-- Default Value just in case there is something wron with the WebConfig file value.
            }
            request.Method = "GET";
            request.ContentType = @"application/json; charset=utf-8";
            String secretKey = ConfigurationManager.AppSettings["secretKey"];
            request.Headers.Add("Authorization", "Basic " + secretKey);

            HttpWebResponse response = (HttpWebResponse)request.GetResponse();
            using (StreamReader sr = new StreamReader(response.GetResponseStream()))
            {
                jsonResponse = sr.ReadToEnd();
            }

            return jsonResponse;
        }

        private string SaveInterfaceLog(string url, string userId, string pdfFileName)
        {
            string result = string.Empty;

            var httpWebRequest = (HttpWebRequest)WebRequest.Create(url);
            httpWebRequest.ContentType = "application/json";
            httpWebRequest.Method = "POST";
            String secretKey = ConfigurationManager.AppSettings["secretKey"];
            httpWebRequest.Headers.Add("Authorization", "Basic " + secretKey);

            using (var streamWriter = new StreamWriter(httpWebRequest.GetRequestStream()))
            {
                streamWriter.Write(pdfFileName);
                streamWriter.Flush();
                streamWriter.Close();
            }

            var httpResponse = (HttpWebResponse)httpWebRequest.GetResponse();
            using (var streamReader = new StreamReader(httpResponse.GetResponseStream()))
            {
                result = streamReader.ReadToEnd();
            }
            return result;
        }

        private void buttonClose_Click(object sender, EventArgs e)
        {
            this.Close();
        }

    }
}
