
/* Create ToProcess table
Author: Rolli
Date: 2018-03-21
*/
If not exists (select 1 from sys.objects where ([name] = 'EDIPIsToProcess'))
Begin
	CREATE TABLE [AHOBPR].[EDIPIsToProcess](
		[EDIPI] [varchar](50) NOT NULL,
		[SENT] [BIT] NULL default(0),
		[CREATED] [datetime] NOT NULL,
		[CREATEDBY] [varchar](30) NOT NULL default(getdate()),
		[UPDATED] [datetime] NOT NULL,
		[UPDATEDBY] [varchar](30) NOT NULL,
		CONSTRAINT [PK_AHOBPR_EDIPIsToProcess] PRIMARY KEY CLUSTERED 
	(
		[EDIPI] ASC
	)WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON, FILLFACTOR = 90) ON [AHOBPR]
	) ON [AHOBPR]
End
GO
if exists (select 1 from sys.objects where ([type] = 'P' and [name] = 'INS_EDIPIsToProcess'))
DROP PROCEDURE [AHOBPR].[INS_EDIPIsToProcess]
GO
Create Procedure [AHOBPR].[INS_EDIPIsToProcess]
(@BatchSize int,
@ReloadTable bit = 0) 
as 
Begin
/*
Author: Rolli
Comments: 
This procedure is designed to add historical AHOBPR patients to the BENS processing queue so that their information can be updated by the BENS Service. 
This procedure is designed to be executed by a SQL Job on a daily basis at 2:45am.  
At 3am numerous other processing patterns begin their schedules, and this procedure should be executed just prior to those other processes beginning.

The following procedure...
1. If it doesn't exist, the procedure will create the AHOBPR.EDIPIsToProcess table
2. Populates the AHOBPR.EDIPIsToProcess table with unique EDIPIs from the AHOBPR.Registrant table, that do not exist in AHOBPR.UPDATES_BENS_NOTIFICATION, and assumingly have not been processed with BENS yet.
	** The @ReloadTable variable provides the option to essentially begin this entire process over.   Reloading the EDIPIsToProcess table works identically to the initial execution, but will truncate the records existing in EDIPIsToProcess, and therefore we'd lose the [SENT] flag field information, but, history is retained in AHOBPR.UPDATES_BENS_NOTIFICATION table for those records that have been added.
3. Populate the AHOBPR.UPDATES_BENS_NOTIFICATION table with a specific number (Adjustable Batch Size (@BatchSize)) of EDIPIs identified in #2 for processing.
4. Update the [SENT] flag field within the AHOBPR.EDIPIsToProcess which identifies which records have already been sent into the processing queue. 

Variables Usage
	@BatchSize INT - determines the number of EDIPIs to load per each execution
	@ReloadTable BIT -  Allows the ability to force a reload the EDIPIsToProcess table by setting the variable to 1.   0 is the default setting, and should only be changed for administrative purposes.

Example Usage:  Exec [AHOBPR].[INS_EDIPIsToProcess] @BatchSize=8000, @ReloadTable=0
Example Results: Command(s) Completed Successfully
	-- Upon the successful execution of the initial run, the procedure would have created the new AHOBPR.EDIPIsToProcess table, populated it with approx 179k EDIPIs, and will insert 8000 EDIPIs from AHOBPR.EDIPIsToProcess into AHOBPR.UPDATES_BENS_NOTIFICATION, and mark the Records in AHOBPR.EDIPIstoProcess as SENT = 1
	-- Upon successful execution of subsequent runs, the procedure would have added the next 15000 EDIPIs indentified as Unsent (in in the AHOBPR.EDIPIsToProcess table as [SENT] = 0), and mark all EDIPIs in AHOBPR.EDIPIsToProcess (found in AHOBPR.UPDATES_BENS_NOTIFICATION) as [SENT] = 1
	-- W/ the batchsize set to 8000, it will take 23 executions to complete all 179k patient
*/

SET NOCOUNT on
Declare 
@1stRun bit = 0,
@StdRegistryID INT

	/* variable validation */
	If (@BatchSize < 0 or @BatchSize > 250000)
	Begin
		Print ('Error: Batch Size out of range.')
		Return
	End

	/* Create ToProcess table*/
	If not exists (select 1 from sys.objects where ([name] = 'EDIPIsToProcess'))
	Begin
		set @1stRun = 1

		CREATE TABLE [AHOBPR].[EDIPIsToProcess](
			[EDIPI] [varchar](50) NOT NULL,
			[SENT] [BIT] NULL default(0),
			[CREATED] [datetime] NOT NULL,
			[CREATEDBY] [varchar](30) NOT NULL default(getdate()),
			[UPDATED] [datetime] NOT NULL,
			[UPDATEDBY] [varchar](30) NOT NULL,
		 CONSTRAINT [PK_AHOBPR_EDIPIsToProcess] PRIMARY KEY CLUSTERED 
		(
			[EDIPI] ASC
		)WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON, FILLFACTOR = 90) ON [AHOBPR]
		) ON [AHOBPR]
	End

	if (@ReloadTable = 1)
	Begin
		Truncate table [AHOBPR].[EDIPIsToProcess]
	End

	if (@1stRun = 1 or @ReloadTable = 1)
	Begin
		/* Load ToProcess Table - previous failures (19) -- Not sure if these should be included..but there are only 19, so I included them*/
		Insert into [AHOBPR].[EDIPIsToProcess]
		([EDIPI], [Created], [CreatedBy])
		Select distinct(r.EDIPI), Getdate(), 'AHOBPR.INS_EDIPIsToProcess'
		from [AHOBPR].[Registrant] r
			join [AHOBPR].[UPDATES_PROCESSED_NOTIFICATION_STATUS] s on r.[EDIPI] = s.[EDIPI]
		where (s.[process_status]='failure')
			and (r.[EDIPI] not in (select distinct([EDIPI]) from [AHOBPR].[EDIPIsToProcess]))
				
		/* Load ToProcess Table - Non-existing Historicals (178201) -- the mass of registrants*/
		Insert into [AHOBPR].[EDIPIsToProcess]
		([EDIPI], [Created], [CreatedBy])
		Select distinct(r.EDIPI), Getdate(), 'AHOBPR.INS_EDIPIsToProcess'
		from [AHOBPR].[Registrant] r
			left join [AHOBPR].[UPDATES_BENS_NOTIFICATION] s on r.[EDIPI] = s.[EDIPI]
		where (s.[EDIPI] is null)
			and (r.[EDIPI] not in (select distinct([EDIPI]) from [AHOBPR].[EDIPIsToProcess]))
	End

	/* Add batch to processing table */
	Declare
	@DynSQL varchar(max)

	set @DynSQL = 
	'INSERT INTO [AHOBPR].[UPDATES_BENS_NOTIFICATION]
	([EDIPI], [CREATED], [CREATEDBY])
	Select top '+cast(@BatchSize as varchar)+' e.[EDIPI], getdate(), ''AHOBPR.INS_EDIPIsToProcess''
	from [AHOBPR].[EDIPIsToProcess] e
	where (e.[SENT] = 0)
		and (e.[EDIPI] not in (select distinct([EDIPI]) from [AHOBPR].[UPDATES_BENS_NOTIFICATION]))
	group by e.[EDIPI]'


	Exec (@DynSQL)
		if (@@Error<>0)
		Begin
			select @StdRegistryID = [ID] from [DBO].[STD_REGISTRY] where ([CODE] = 'AHOBPR')

			Exec [dbo].[CRS_SP_INS_CRSDBLOG] 
				@StdRegistryID,
				1,
				'INS_EDIPIsToProcess',
				'Error executing the Batch statement',
				'Records are not being written to the UPDATES_BENS_NOTIFICATION table',
				'ETLProcess'
		End

	/* update sent flag */
	Update t
		set t.[SENT] = 1,
			t.[Updated] = getdate(),
			t.[updatedby] = 'AHOBPR.INS_EDIPIsToProcess'
	From [AHOBPR].[EDIPIsToProcess] t
		join [AHOBPR].[UPDATES_BENS_NOTIFICATION] n on t.[EDIPI] = n.[EDIPI]
	where (t.[Sent] = 0)
	
		if (@@Error<>0)
		Begin
			select @StdRegistryID = [ID] from [DBO].[STD_REGISTRY] where ([CODE] = 'AHOBPR')

			Exec [dbo].[CRS_SP_INS_CRSDBLOG] 
				@StdRegistryID,
				1,
				'INS_EDIPIsToProcess',
				'Error updating the [SENT] flag in [EDIPIsToProcess]',
				'Failure to resolve this issue will result in re-processing the same records each day',
				'ETLProcess'
		End
End
GO

-- Exec [AHOBPR].[INS_EDIPIsToProcess] @BatchSize=1, @ReloadTable=0


if exists (Select 1 from sys.objects where [name] = 'VIEW_BENS_HistProcessCompletion')
Begin
	DROP VIEW [AHOBPR].[VIEW_BENS_HistProcessCompletion]
End

GO
CREATE VIEW [AHOBPR].[VIEW_BENS_HistProcessCompletion]
AS 
	/*
	This view provides insight into how many historical records have been processed by the [AHOBPR].[INS_EDIPIsToProcess] procedure
	*/

	Select 
		Count([EDIPI]) as [RecordsSent], 
		(Select Count(distinct([EDIPI])) from [AHOBPR].[EDIPIsToProcess]) as [TotalRecords],
		((cast(Count([EDIPI]) as decimal(4,2)) / 
			(Select Count(distinct([EDIPI])) from [AHOBPR].[EDIPIsToProcess]))*100
		) as [PercentComplete]
	From [AHOBPR].[EDIPIsToProcess]
	where ([Sent] = 1)
	group by [SENT]

GO