/* 
Author: L. Mansfield
Date: 9-14-2017
Description: Create new tables: STD_DEPLOYMENT_STAGE, STD_RETIREMENT_TERMINATION_REASON, STD_RETIREMENT_TYPE, REGISTRANT_RETIREMENT,
			and USER_STATES_FILTER.
			Alter tables: REGISTRANT and REGISTRANT_DEPLOYMENT
*/
:setvar DatabaseName "REGISTRYHRE"
:setvar MajorReleaseNumber "4" 
:setvar MinorReleaseNumber "0" 
:setvar PointReleaseNumber "8"
:setvar ScriptName "1_INSTALL.sql"

PRINT ' 
------------------------------------------------------------------------------------------------
$(ScriptName)

Script in support of AHOBPR Release $(MajorReleaseNumber).$(MinorReleaseNumber).$(PointReleaseNumber)

------------------------------------------------------------------------------------------------
 ';
PRINT 'Started ' + CAST(GETDATE() as varchar(20)) + '.';
PRINT ' 
*********************************************************
NOTE: This script must be run with SQLCMD Mode turned on.
*********************************************************

';
GO
:on error exit
GO
:setvar __IsSqlCmdEnabled "True"
/*
Detect SQLCMD mode and disable script execution if SQLCMD mode is not supported.
To re-enable the script after enabling SQLCMD mode, execute the following two lines:
*/
SET NOEXEC OFF; 
SET NOCOUNT OFF;
GO
IF N'$(__IsSqlCmdEnabled)' NOT LIKE N'True'
    BEGIN
        PRINT N'SQLCMD mode must be enabled to successfully execute this script.';
        SET NOCOUNT ON;
        SET NOEXEC ON; --When SET NOEXEC is ON, SQL Server compiles each batch of Transact-SQL statements but does not execute them.
    END

GO
USE $(DatabaseName)
GO

IF EXISTS ( select * from [dbo].[DatabaseChangeLog]
            where [MajorReleaseNumber]	= '$(MajorReleaseNumber)'
              and [MinorReleaseNumber]	= '$(MinorReleaseNumber)'
              and [PointReleaseNumber]	= '$(PointReleaseNumber)'
			  and [TargetRegistry]		= 'AHOBPR'
              and [ScriptName]			= '$(ScriptName)'
          )
BEGIN
   RAISERROR('!!!!The deployment script may not be rerun without first backing out the changes, including the DatabaseChangeLog record.!!!!'
              ,11
              ,1);
END
ELSE
  INSERT 
    INTO [dbo].[DatabaseChangeLog]
             ([MajorReleaseNumber]
             ,[MinorReleaseNumber]
             ,[PointReleaseNumber]
             ,[Hotfix]
             ,[ScriptName]
             ,[ScriptDescription]
             ,[TargetRegistry]
             ,[DateApplied])
       VALUES
             ('$(MajorReleaseNumber)'
             ,'$(MinorReleaseNumber)'
             ,'$(PointReleaseNumber)'
             ,0
             ,'$(ScriptName)'
		         ,'Script Started...'
             ,'AHOBPR'
             ,GETDATE()
            );

GO
---------------------------------------------------------------------------------------------------------
PRINT '$(ScriptName) script started...';
PRINT GETDATE();
GO
---------------------------------------------------------------------------------------------------------

-- Add Columns to [AHOBPR].[REGISTRANT] if they do not exist
IF NOT EXISTS (
  SELECT * 
  FROM   sys.columns 
  WHERE  object_id = OBJECT_ID(N'[AHOBPR].[REGISTRANT]') 
         AND name = 'PRE_911_DEPLOYMENT')
BEGIN
	ALTER TABLE [AHOBPR].[REGISTRANT]
		ADD	[PRE_911_DEPLOYMENT] [varchar](3) NULL
END
GO
IF NOT EXISTS (
  SELECT * 
  FROM   sys.columns 
  WHERE  object_id = OBJECT_ID(N'[AHOBPR].[REGISTRANT]') 
         AND name = 'POST_911_DEPLOYMENT')
BEGIN
	ALTER TABLE [AHOBPR].[REGISTRANT]
		 ADD	[POST_911_DEPLOYMENT] [varchar](3) NULL
END
GO	
-- Create the STD_DEPLOYMENT_STAGE table
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[AHOBPR].[STD_DEPLOYMENT_STAGE]') AND type in (N'U'))
BEGIN
	CREATE TABLE [AHOBPR].[STD_DEPLOYMENT_STAGE](
		[STD_DEPLOYMENT_STAGE_ID] 	[int] IDENTITY(1,1) NOT NULL,
		[DEPLOYMENT_STAGE] 			[varchar](100)		NULL,
		[CREATEDBY] 				[varchar](30) 		NOT NULL,
		[UPDATEDBY] 				[varchar](30) 		NOT NULL,
		[CREATED] 					[datetime2](0) 		NOT NULL,
		[UPDATED] 					[datetime2](0) 		NOT NULL,
		[ROW_VERSION] 				[timestamp] 		NOT NULL,
		CONSTRAINT [STD_DEPLOYMENT_STAGE_PK] PRIMARY KEY CLUSTERED 
		([STD_DEPLOYMENT_STAGE_ID] ASC)
		WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON) ON [AHOBPR]
	) ON [AHOBPR]
END
GO

-- Clean out STD_DEPLOYMENT_STAGE
SET IDENTITY_INSERT [AHOBPR].[STD_DEPLOYMENT_STAGE] ON 
;WITH  MasterData
AS    ( select * 
		from (VALUES
				(1, N'New', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE()),
				(2, N'Not Eligible', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE()),
				(3, N'Verified', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE()),
				(4, N'Refuted', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE()),
				(5, N'Accepted', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE()),
				(6, N'Rejected', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())
			) AS MasterRecord(STD_DEPLOYMENT_STAGE_ID, DEPLOYMENT_STAGE,CREATEDBY, UPDATEDBY, CREATED, UPDATED)
)
MERGE INTO [AHOBPR].[STD_DEPLOYMENT_STAGE] as Tgt
USING MasterData as Src
ON Tgt.[STD_DEPLOYMENT_STAGE_ID] = Src.STD_DEPLOYMENT_STAGE_ID
WHEN NOT MATCHED
THEN INSERT
-- INSERT STD_DEPLOYMENT_STAGE values
 ([STD_DEPLOYMENT_STAGE_ID], [DEPLOYMENT_STAGE], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) 
  VALUES
  (Src.STD_DEPLOYMENT_STAGE_ID, Src.DEPLOYMENT_STAGE, Src.CREATEDBY, Src.UPDATEDBY, Src.CREATED, Src.UPDATED)
WHEN MATCHED
THEN UPDATE SET 
				Tgt. [DEPLOYMENT_STAGE] = Src.DEPLOYMENT_STAGE, 
				Tgt.[CREATEDBY] = Src.CREATEDBY, 
				Tgt.[UPDATEDBY] = Src.UPDATEDBY, 
				Tgt.[CREATED] = Src.CREATED,
				Tgt.[UPDATED] = Src.UPDATED;

SET IDENTITY_INSERT [AHOBPR].[STD_DEPLOYMENT_STAGE] OFF
GO
-- Create the STD_RETIREMENT_TERMINATION_REASON Table
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[AHOBPR].[STD_RETIREMENT_TERMINATION_REASON]') AND type in (N'U'))
BEGIN
	CREATE TABLE [AHOBPR].[STD_RETIREMENT_TERMINATION_REASON](
		[STD_RETIREMENT_TERMINATION_REASON_ID] 		[int] IDENTITY(1,1) NOT NULL,
		[RETIREMENT_TERMINATION_REASON_CODE] 		[varchar](100) 		NOT NULL,
		[RETIREMENT_TERMINATION_REASON_DESCRIPTION] [varchar](1000) 	NOT NULL,
		[CREATEDBY]	 								[varchar](30) 		NOT NULL,
		[UPDATEDBY] 								[varchar](30) 		NOT NULL,
		[CREATED] 									[datetime2](0) 		NOT NULL,
		[UPDATED] 									[datetime2](0) 		NOT NULL,
		[ROW_VERSION] 								[timestamp] 		NOT NULL,
	CONSTRAINT [STD_RETIREMENT_TERMINATION_REASON_PK] PRIMARY KEY CLUSTERED 
	([STD_RETIREMENT_TERMINATION_REASON_ID] ASC)
		WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON) ON [AHOBPR]
	) ON [AHOBPR]
END
GO
-- Create the STD_RETIREMENT_TYPE Table
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[AHOBPR].[STD_RETIREMENT_TYPE]') AND type in (N'U'))
BEGIN
	CREATE TABLE [AHOBPR].[STD_RETIREMENT_TYPE](
		[STD_RETIREMENT_TYPE_ID] 		[int] 			IDENTITY(1,1) NOT NULL,
		[RETIREMENT_TYPE_CODE] 			[varchar](100) 	NOT NULL,
		[RETIREMENT_TYPE_DESCRIPTION] 	[varchar](1000) NOT NULL,
		[CREATEDBY] 					[varchar](30) 	NOT NULL,
		[UPDATEDBY] 					[varchar](30) 	NOT NULL,
		[CREATED] 						[datetime2](0) 	NOT NULL,
		[UPDATED] 						[datetime2](0) 	NOT NULL,
		[ROW_VERSION]					[timestamp] 	NOT NULL,
	CONSTRAINT [STD_RETIREMENT_TYPE_PK] PRIMARY KEY CLUSTERED 
	([STD_RETIREMENT_TYPE_ID] ASC)
		WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON) ON [AHOBPR]
	) ON [AHOBPR]
END
GO
--Create the REGISTRANT_RETIREMENT Table
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[AHOBPR].[REGISTRANT_RETIREMENT]') AND type in (N'U'))
BEGIN
	CREATE TABLE [AHOBPR].[REGISTRANT_RETIREMENT](
		[REGISTRANT_RETIREMENT_ID] 				[int] 	IDENTITY(1,1) NOT NULL,
		[REGISTRANT_ID] 						[int] NOT NULL,
		[STD_RETIREMENT_TYPE_ID] 				[int] NULL,
		[RETIREMENT_SERVICE_CODE] 				[varchar](256) NULL,
		[RETIREMENT_BEGIN_DATE] 				[datetime2](0) NOT NULL,
		[RETIREMENT_TERMINATION_DATE] 			[datetime2](0) NULL,
		[STD_RETIREMENT_TERMINATION_REASON_ID] 	[int] NULL,
		[CREATEDBY] 							[varchar](30) NOT NULL,
		[UPDATEDBY] 							[varchar](30) NOT NULL,
		[CREATED] 								[datetime2](0) NOT NULL,
		[UPDATED] 								[datetime2](0) NOT NULL,
		[ROW_VERSION] 							[timestamp] NOT NULL,
		CONSTRAINT [REGISTRANT_RETIREMENT_PK] PRIMARY KEY CLUSTERED 
		([REGISTRANT_RETIREMENT_ID] ASC)
		WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON) ON [AHOBPR]
	) ON [AHOBPR]

	SET ANSI_PADDING OFF

	ALTER TABLE [AHOBPR].[REGISTRANT_RETIREMENT]  WITH CHECK ADD  CONSTRAINT [FK_REGISTRANT_RETIREMENT_REGISTRANT] FOREIGN KEY([REGISTRANT_ID])
	REFERENCES [AHOBPR].[REGISTRANT] ([REGISTRANT_ID])

	ALTER TABLE [AHOBPR].[REGISTRANT_RETIREMENT] CHECK CONSTRAINT [FK_REGISTRANT_RETIREMENT_REGISTRANT]

	ALTER TABLE [AHOBPR].[REGISTRANT_RETIREMENT]  WITH CHECK ADD  CONSTRAINT [FK_REGISTRANT_RETIREMENT_STD_RETIREMENT_TERMINATION_REASON] FOREIGN KEY([STD_RETIREMENT_TERMINATION_REASON_ID])
	REFERENCES [AHOBPR].[STD_RETIREMENT_TERMINATION_REASON] ([STD_RETIREMENT_TERMINATION_REASON_ID])

	ALTER TABLE [AHOBPR].[REGISTRANT_RETIREMENT] CHECK CONSTRAINT [FK_REGISTRANT_RETIREMENT_STD_RETIREMENT_TERMINATION_REASON]

	ALTER TABLE [AHOBPR].[REGISTRANT_RETIREMENT]  WITH CHECK ADD  CONSTRAINT [FK_REGISTRANT_RETIREMENT_STD_RETIREMENT_TYPE] FOREIGN KEY([STD_RETIREMENT_TYPE_ID])
	REFERENCES [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID])

	ALTER TABLE [AHOBPR].[REGISTRANT_RETIREMENT] CHECK CONSTRAINT [FK_REGISTRANT_RETIREMENT_STD_RETIREMENT_TYPE]
END
GO
--End Create Table REGISTRANT_RETIRMENT

-- Create Table USER_STATES_FILTER
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[AHOBPR].[USER_STATES_FILTER]') AND type in (N'U'))
BEGIN
CREATE TABLE [AHOBPR].[USER_STATES_FILTER](
	[USER_STATE_FILTER_ID] 	[int] 			IDENTITY(1,1) NOT NULL,
	[USER_ID] 				[int] 			NOT NULL,
	[STATES] 				[nvarchar](30) 	NOT NULL,
	[CREATEDBY] 			[varchar](30) 	NOT NULL,
	[CREATED] 				[datetime] 		NOT NULL,
	[UPDATEDBY] 			[varchar](30) 	NOT NULL,
	[UPDATED] 				[datetime] 		NOT NULL,
	[ROW_VERSION] 			[timestamp] 	NOT NULL,
 	CONSTRAINT [STATE_FILTER_PK] PRIMARY KEY CLUSTERED 
	([USER_STATE_FILTER_ID] ASC)
	WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON) ON [AHOBPR],
	UNIQUE NONCLUSTERED 
	([USER_ID] ASC)
	WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON) ON [AHOBPR]
) ON [AHOBPR]

	ALTER TABLE [AHOBPR].[USER_STATES_FILTER]  WITH CHECK ADD CONSTRAINT [STATE_FILTER_USERS_FK] FOREIGN KEY([USER_ID])
	REFERENCES [dbo].[USERS] ([USER_ID])

	ALTER TABLE [AHOBPR].[USER_STATES_FILTER] CHECK CONSTRAINT [STATE_FILTER_USERS_FK]
END
GO
-- End Create Table USER_STATES_FILTER
--Update the REGISTRANT_DEPLOYMENT table
IF NOT EXISTS (
  SELECT * 
  FROM   sys.columns 
  WHERE  object_id = OBJECT_ID(N'[AHOBPR].[REGISTRANT_DEPLOYMENT]') 
         AND name = 'STD_DEPLOYMENT_STAGE_ID')
BEGIN
	ALTER TABLE [AHOBPR].[REGISTRANT_DEPLOYMENT] 
	ADD STD_DEPLOYMENT_STAGE_ID int NOT NULL 
	CONSTRAINT [DF_REGISTRANT_DEPLOYMENT_STD_DEPLOYMENT_STAGE] DEFAULT 1		
END
-- Add the FK_REGISTRANT_DEPLOYMENT_STD_DEPLOYMENT_STAGE foreign key constraint if it does not exist
GO

IF (OBJECT_ID('ahobpr.[FK_REGISTRANT_DEPLOYMENT_STD_DEPLOYMENT_STAGE]', 'F') IS NULL)
BEGIN
	Print N'Trying to add foreign key FK_REGISTRANT_DEPLOYMENT_STD_DEPLOYMENT_STAGE'
	ALTER 
		TABLE [AHOBPR].[REGISTRANT_DEPLOYMENT]  
		ADD  CONSTRAINT [FK_REGISTRANT_DEPLOYMENT_STD_DEPLOYMENT_STAGE] 
		FOREIGN KEY([STD_DEPLOYMENT_STAGE_ID])
		REFERENCES [AHOBPR].[STD_DEPLOYMENT_STAGE] ([STD_DEPLOYMENT_STAGE_ID]);
END
GO
/* 
Author: L. Mansfield
Date: 9-14-2017
Description: Insert Data into STD_GUI_CONTROLS, STD_DEPLOYMENT_STAGE, STD_RETIREMENT_TYPE, and STD_COUNTRY tables.
*/

DECLARE  @RegistryId int

BEGIN

	SELECT @RegistryId = ID FROM [dbo].[STD_REGISTRY] where [CODE] = N'AHOBPR'

	-- Delete the previously inserted AHOBPR GUI control entries...
	DELETE FROM [dbo].[STD_GUI_CONTROLS] 
	WHERE [STD_REGISTRY_ID] = @RegistryId 
		and CATEGORY = 'AHOBPR_REGISTRANT_INFORMATION_USER_CONTROL' 
		AND [USER_CONTROL_ID] IN ('BaseControlLabelDeathDate', 'BaseControlLabelRetirementType', 'BaseControlLabelRetirementDate')

	INSERT INTO [dbo].[STD_GUI_CONTROLS] ([STD_REGISTRY_ID], [CATEGORY], [NAME], [BASE_CONTROL_TYPE], [USER_CONTROL_ID], [VALIDATION_ERROR_MESSAGE], [VALIDATION_REGULAR_EXPRESSION], [TOOL_TIP], [SORT_ORDER], [CREATED], [UPDATED], [CREATEDBY], [UPDATEDBY], [LOOKUP_LIST_CATEGORY], [BASE_CONTROL_WIDTH], [DATA_ELEMENT_WIDTH], [REQUIRED], [INCLUDE_LABEL], [LABEL_TEXT], [LOOKUP_LIST_CATEGORY2], [LOOKUP_LIST_CATEGORY3], [VALIDATION_GROUP], [DATA_TYPE_MAX_LENGTH], [PANEL_NAME]) VALUES ( @RegistryId, N'AHOBPR_REGISTRANT_INFORMATION_USER_CONTROL', N'Death Date', 1, N'BaseControlLabelDeathDate', NULL, NULL, NULL, 0, GETDATE(), GETDATE(), N'AHOBPR DBA', N'AHOBPR DBA', NULL, NULL, NULL, 0, 1, N'Death Date:&nbsp&nbsp', NULL, NULL, NULL, NULL, N'PanelRegistrantDemographics')
	INSERT INTO [dbo].[STD_GUI_CONTROLS] ([STD_REGISTRY_ID], [CATEGORY], [NAME], [BASE_CONTROL_TYPE], [USER_CONTROL_ID], [VALIDATION_ERROR_MESSAGE], [VALIDATION_REGULAR_EXPRESSION], [TOOL_TIP], [SORT_ORDER], [CREATED], [UPDATED], [CREATEDBY], [UPDATEDBY], [LOOKUP_LIST_CATEGORY], [BASE_CONTROL_WIDTH], [DATA_ELEMENT_WIDTH], [REQUIRED], [INCLUDE_LABEL], [LABEL_TEXT], [LOOKUP_LIST_CATEGORY2], [LOOKUP_LIST_CATEGORY3], [VALIDATION_GROUP], [DATA_TYPE_MAX_LENGTH], [PANEL_NAME]) VALUES ( @RegistryId, N'AHOBPR_REGISTRANT_INFORMATION_USER_CONTROL', N'Retirement Type', 1, N'BaseControlLabelRetirementType', NULL, NULL, NULL, 0, GETDATE(), GETDATE(), N'AHOBPR DBA', N'AHOBPR DBA', NULL, NULL, NULL, 0, 1, N'Retirement Type:&nbsp&nbsp', NULL, NULL, NULL, NULL, N'PanelSeparationInfo')
	INSERT INTO [dbo].[STD_GUI_CONTROLS] ([STD_REGISTRY_ID], [CATEGORY], [NAME], [BASE_CONTROL_TYPE], [USER_CONTROL_ID], [VALIDATION_ERROR_MESSAGE], [VALIDATION_REGULAR_EXPRESSION], [TOOL_TIP], [SORT_ORDER], [CREATED], [UPDATED], [CREATEDBY], [UPDATEDBY], [LOOKUP_LIST_CATEGORY], [BASE_CONTROL_WIDTH], [DATA_ELEMENT_WIDTH], [REQUIRED], [INCLUDE_LABEL], [LABEL_TEXT], [LOOKUP_LIST_CATEGORY2], [LOOKUP_LIST_CATEGORY3], [VALIDATION_GROUP], [DATA_TYPE_MAX_LENGTH], [PANEL_NAME]) VALUES ( @RegistryId, N'AHOBPR_REGISTRANT_INFORMATION_USER_CONTROL', N'Retirement Date', 1, N'BaseControlLabelRetirementDate', NULL, NULL, NULL, 0, GETDATE(), GETDATE(), N'AHOBPR DBA', N'AHOBPR DBA', NULL, NULL, NULL, 0, 1, N'Retirement Date:&nbsp&nbsp', NULL, NULL, NULL, NULL, N'PanelRegistrantDemographics')
	INSERT INTO [dbo].[STD_GUI_CONTROLS] ([STD_REGISTRY_ID], [CATEGORY], [NAME], [BASE_CONTROL_TYPE], [USER_CONTROL_ID], [VALIDATION_ERROR_MESSAGE], [VALIDATION_REGULAR_EXPRESSION], [TOOL_TIP], [SORT_ORDER], [CREATED], [UPDATED], [CREATEDBY], [UPDATEDBY], [LOOKUP_LIST_CATEGORY], [BASE_CONTROL_WIDTH], [DATA_ELEMENT_WIDTH], [REQUIRED], [INCLUDE_LABEL], [LABEL_TEXT], [LOOKUP_LIST_CATEGORY2], [LOOKUP_LIST_CATEGORY3], [VALIDATION_GROUP], [DATA_TYPE_MAX_LENGTH], [PANEL_NAME]) VALUES ( @RegistryId, N'AHOBPR_CHANGE_STATUS_REGISTRANT_SEARCH', N'State Filter Parameters', '2', N'BaseControlTextStatesFilter', NULL, NULL, N'The selection criteria used to filter with on or more states.', N'0', GETDATE(), GETDATE(), N'AHOBPR DBA', N'AHOBPR DBA', N'STD_STATE', N'50', N'50', N'False', 'True', N'State:<br />',  NULL, NULL, NULL, NULL, 'PanelSearchCriteria')
	INSERT INTO [dbo].[STD_GUI_CONTROLS] ([STD_REGISTRY_ID], [CATEGORY], [NAME], [BASE_CONTROL_TYPE], [USER_CONTROL_ID], [VALIDATION_ERROR_MESSAGE], [VALIDATION_REGULAR_EXPRESSION], [TOOL_TIP], [SORT_ORDER], [CREATED], [UPDATED], [CREATEDBY], [UPDATEDBY], [LOOKUP_LIST_CATEGORY], [BASE_CONTROL_WIDTH], [DATA_ELEMENT_WIDTH], [REQUIRED], [INCLUDE_LABEL], [LABEL_TEXT], [LOOKUP_LIST_CATEGORY2], [LOOKUP_LIST_CATEGORY3], [VALIDATION_GROUP], [DATA_TYPE_MAX_LENGTH], [PANEL_NAME]) VALUES ((SELECT ID FROM [dbo].[STD_REGISTRY] where [CODE] = 'AHOBPR'), N'AHOBPR_MYACCOUNT_ACCOUNTINFORMATION', 'My Account State Parameters', N'5', N'UserControlListBoxState', NULL, NULL, N'State is the state from the current addresses a Registrant has in AHOBPR.  The Selection Criteria allows a user to filter with one or more states.', N'0', GETDATE(),GETDATE(), N'AHOBPR DBA', N'AHOBPR DBA', N'STD_STATE', N'170', N'170', N'True', N'True', N'Change Status States:</br>', NULL, NULL, NULL, NULL, 'PanelUserInformation')

END
GO
-- Insert new values for STD_RETIREMENT_TYPE
DELETE from  [AHOBPR].[STD_RETIREMENT_TYPE]
SET IDENTITY_INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ON 
INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID], [RETIREMENT_TYPE_CODE], [RETIREMENT_TYPE_DESCRIPTION], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) VALUES (1, N'A', N'Mandatory', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())
INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID], [RETIREMENT_TYPE_CODE], [RETIREMENT_TYPE_DESCRIPTION], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) VALUES (2, N'B', N'Voluntary', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())
INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID], [RETIREMENT_TYPE_CODE], [RETIREMENT_TYPE_DESCRIPTION], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) VALUES (3, N'C', N'Fleet Reserve', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())
INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID], [RETIREMENT_TYPE_CODE], [RETIREMENT_TYPE_DESCRIPTION], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) VALUES (4, N'D', N'Temporary Disability Retirement List', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())
INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID], [RETIREMENT_TYPE_CODE], [RETIREMENT_TYPE_DESCRIPTION], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) VALUES (5, N'E', N'Permanent Disability Retirement List', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())
INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID], [RETIREMENT_TYPE_CODE], [RETIREMENT_TYPE_DESCRIPTION], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) VALUES (6, N'F', N'Title III', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())
INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID], [RETIREMENT_TYPE_CODE], [RETIREMENT_TYPE_DESCRIPTION], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) VALUES (7, N'G', N'Special Act', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())
INSERT [AHOBPR].[STD_RETIREMENT_TYPE] ([STD_RETIREMENT_TYPE_ID], [RETIREMENT_TYPE_CODE], [RETIREMENT_TYPE_DESCRIPTION], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED]) VALUES (8, N'H', N'Philippine Scouts', N'AHOBPR DBA', N'AHOBPR DBA', GETDATE(), GETDATE())

SET IDENTITY_INSERT [AHOBPR].[STD_RETIREMENT_TYPE] OFF
GO
-- Update Major Bodies of Water
;WITH  MasterData
AS    ( 
        SELECT [CODE],  [DESCRIPTION]
        FROM    (VALUES             
( '1A', N'North Atlantic Ocean'),
( '1B', N'Bay of Biscay'),
( '1C', N'Bristol Channel'),
( '1D', N'Denmark Strait'),
( '1E', N'English Channel'),
( '1F', N'Bay of Fundy'),
( '1G', N'Gulf of Guinea'),
( '1H', N'Hudson Bay'),
( '1J', N'Panama Canal'),
( '1K', N'Inner Seas (west coast of Scotland)'),
( '1L', N'Labrador Sea'),
( '1M', N'Gulf of Mexico'),
( '1N', N'North Sea'),
( '1P', N'Baffin Bay'),
( '1Q', N'St. Georges Channel'),
( '1R', N'Irish Sea'),
( '1S', N'Skagerrak'),
( '1T', N'Gulf of St. Lawrence'),
( '1U', N'Hudson Strait'),
( '1V', N'Davis Strait'),
( '1X', N'Caribbean Sea'),
( '2A', N'South Atlantic Ocean'),
( '2D', N'Drake Passage'),
( '2J', N'Gulf of San Jorge'),
( '2M', N'Gulf of San Matias'),
( '2R', N'Rio de la Plata'),
( '2S', N'Scotia Sea'),
( '2W', N'Weddell Sea'),
( '3A', N'North Pacific Ocean'),
( '3B', N'Gulf of Sakhalin'),
( '3C', N'Celebes Sea'),
( '3D', N'Gulf of Tatary'),
( '3E', N'East China Sea'),
( '3F', N'Formosa Strait'),
( '3G', N'Gulf of Tonkin'),
( '3H', N'Halmahera Sea'),
( '3I', N'Luzon Strait'),
( '3J', N'Sea of Japan'),
( '3K', N'Shelekova Gulf'),
( '3L', N'Gulf of California'),
( '3M', N'Molucca Sea'),
( '3N', N'Inland Sea'),
( '3P', N'Philippine Sea'),
( '3Q', N'Sea of Okhotsk'),
( '3R', N'Korea Bay'),
( '3S', N'Sulu Sea'),
( '3T', N'Gulf of Siam'),
( '3U', N'South China Sea'),
( '3V', N'Gulf of Tomini'),
( '3X', N'Gulf of Chihli'),
( '3Y', N'Yellow Sea'),
( '3Z', N'Singapore Strait'),
( '4A', N'South Pacific Ocean'),
( '4B', N'Banda Sea'),
( '4C', N'Coral Sea'),
( '4D', N'Amundsen Sea'),
( '4E', N'Gulf of Bone'),
( '4F', N'Flores Sea'),
( '4G', N'Bellingshausen Sea'),
( '4J', N'Java Sea'),
( '4K', N'Bismarck Sea'),
( '4L', N'Bali Sea'),
( '4M', N'Makassar Strait'),
( '4P', N'Gulf of Carpentaria'),
( '4Q', N'Ceram Sea'),
( '4R', N'Ross Sea'),
( '4S', N'Solomon Sea'),
( '4T', N'Tasman Sea'),
( '4U', N'Arafura Sea'),
( '5A', N'Arctic Ocean'),
( '5B', N'Barents Sea'),
( '5C', N'Chukchi Sea'),
( '5D', N'Bering Sea'),
( '5E', N'Coastal waters of SE Alaska and British Columbia'),
( '5F', N'Gulf of Alaska'),
( '5G', N'Greenland Sea'),
( '5K', N'Kara Sea'),
( '5L', N'Lincoln Sea'),
( '5N', N'Norwegian Sea'),
( '5P', N'Laptev Sea'),
( '5R', N'Bering Strait'),
( '5S', N'East Siberian Sea'),
( '5T', N'Northwest Passage'),
( '5U', N'Beaufort Sea'),
( '5W', N'White Sea'),
( '5Y', N'Gulf of Anadyr'),
( '6A', N'Indian Ocean'),
( '6B', N'Bay of Bengal'),
( '6C', N'Malacca Strait'),
( '6D', N'Gulf of Aden'),
( '6E', N'Red Sea'),
( '6F', N'Bass Strait'),
( '6G', N'Great Australian Bight'),
( '6L', N'Laccadive Sea'),
( '6M', N'Gulf of Oman'),
( '6N', N'Andaman Sea'),
( '6P', N'Persian Gulf'),
( '6Q', N'Gulf of Aqaba'),
( '6R', N'Arabian Sea'),
( '6S', N'Savu Sea'),
( '6T', N'Timor Sea'),
( '6U', N'Suez Canal'),
( '6W', N'Gulf of Suez'),
( '6Z', N'Mozambique Channel'),
( '7B', N'Baltic Sea'),
( '7F', N'Gulf of Finland'),
( '7H', N'Gulf of Riga'),
( '7K', N'Kattegat'),
( '7T', N'Gulf of Bothnia'),
( '8B', N'Black Sea'),
( '8C', N'Caspian Sea'),
( '8D', N'Adriatic Sea'),
( '8E', N'Eastern Mediterranean Sea'),
( '8G', N'Aegean Sea'),
( '8J', N'Balearic Sea'),
( '8L', N'Ligurian Sea'),
( '8M', N'Sea of Marmara'),
( '8N', N'Ionian Sea'),
( '8P', N'Bosporus'),
( '8R', N'Aral Sea'),
( '8S', N'Strait of Gibraltar'),
( '8T', N'Tyrrhenian Sea'),
( '8U', N'Dardanelles'),
( '8W', N'Western Mediterranean Sea'),
( '8X', N'Gulf of Lion'),
( '8Y', N'Alboran Sea'),
( '8Z', N'Sea of Azov'),
( '9E', N'Lake Erie'),
( '9H', N'Lake Huron'),
( '9L', N'St. Lawrence Seaway'),
( '9M', N'Lake Michigan'),
( '9N', N'Lake Ontario'),
( '9S', N'Lake Superior'))

       AS MasterRecord(CODE, DESCRIPTION)
       )
MERGE
  INTO [AHOBPR].[STD_COUNTRY] AS Tgt
  USING MasterData AS Src
  ON    Tgt.[CODE] = Src.CODE
 WHEN  NOT MATCHED
 THEN  INSERT
        ([CODE],  [DESCRIPTION]) 
        VALUES
        (Src.CODE, Src.DESCRIPTION)
 WHEN MATCHED 
 THEN UPDATE SET Tgt.[CODE] = Src.CODE,
                 Tgt.DESCRIPTION = Src.DESCRIPTION;

GO
/* 
Author: L. Mansfield
Date: 9-14-2017
Description: Add correct values to STD_APPAGE and STD_ROLE_PERMISSIONS based on Roles
*/

DECLARE @PageId INT, @AdminRoleId INT, @AdvancedUserRoleId INT, @CareTeamRoleId INT, @RegistryId INT, @HelpDeskRoleId INT;
DECLARE @UserName VARCHAR(10) = 'AHOBPR DBA';
DECLARE @Today DateTime2 = GETDATE();

SET @RegistryId  = (SELECT ID FROM [dbo].[STD_REGISTRY] where [CODE] = N'AHOBPR');

-- Load STD_ROLE
WITH  MasterData
AS    ( 
        SELECT *
        FROM    ( VALUES  
               (N'AHOBPR_REGISTRY_MANAGER', N'AHOBPR Registry Manager', N'AHOBPR Registry Manager who is a super user of the registry', 1, 1),
               (N'AHOBPR_ADVANCED_USER', N'AHOBPR Advanced User', N'AHOBPR Advanced User', 2, 0),
               (N'AHOBPR_CARE_TEAM',    N'AHOBPR Care Team',    N'AHOBPR Care Team', 3, 0),
			   (N'AHOBPR_HELP_DESK',    N'AHOBPR Help Desk',    N'AHOBPR Help Desk', 4, 0))
       AS MasterRecord(CODE, NAME, DESCRIPTION_TEXT, SORT_ORDER, SUPER_USER_FLAG)
       )
MERGE
  INTO  [dbo].[STD_ROLE] AS Tgt
  USING MasterData AS Src
  ON    Tgt.[CODE] = Src.CODE
 WHEN  NOT MATCHED
 THEN  INSERT
        ([CODE], [NAME], [DESCRIPTION_TEXT], [SORT_ORDER], [INACTIVE_FLAG], [INACTIVE_DATE], [CREATEDBY], [UPDATEDBY], [CREATED], [UPDATED], [PARENT_ROLE_ID], [STD_REGISTRY_ID], [SUPER_USER_FLAG]) 
        VALUES
        (Src.CODE, Src.NAME, Src.DESCRIPTION_TEXT, Src.SORT_ORDER, 0, NULL,  @UserName, @UserName, @Today,@Today, NULL, @RegistryId, Src.SUPER_USER_FLAG)
 WHEN MATCHED 
 THEN UPDATE SET Tgt.[NAME] = Src.Name,
                 Tgt.DESCRIPTION_TEXT = Src.DESCRIPTION_TEXT,
                 Tgt.SORT_ORDER = Src.SORT_ORDER,
                 Tgt.SUPER_USER_FLAG = Src.SUPER_USER_FLAG,
                 Tgt.STD_REGISTRY_ID = @RegistryId;

-- Load STD_APPPAGE
WITH  MasterData
AS    ( 
        SELECT *
        FROM   ( VALUES  
				('AHOBPR ADMIN EDIT USERS','AHOBPR ADMIN EDIT USERS','AHOBPR Admin Edit Users',1),
				('AHOBPR ADMIN LIST USERS','AHOBPR ADMIN LIST USERS','AHOBPR Admin List Users',1),
				('AHOBPR ADMIN TRACKING','AHOBPR ADMIN TRACKING','AHOBPR ADMIN TRACKING',1),
				('AHOBPR ADMIN USER ROLE','AHOBPR ADMIN USER ROLE','AHOBPR Admin User Role',1),
				('AHOBPR CHANGE STATUS','AHOBPR CHANGE STATUS','AHOBPR CHANGE STATUS',1),
				('AHOBPR QUESTIONNAIRE FULL','AHOBPR QUESTIONNAIRE FULL','AHOBPR QUESTIONNAIRE FULL',1),
				('AHOBPR QUESTIONNAIRE SECTION','AHOBPR QUESTIONNAIRE SECTION','AHOBPR QUESTIONNAIRE SECTION',1),
				('AHOBPR QUESTIONNAIRE SUMMARY','AHOBPR QUESTIONNAIRE SUMMARY','AHOBPR QUESTIONNAIRE SUMMARY',1),
				('AHOBPR REGISTRANT SEARCH','AHOBPR REGISTRANT SEARCH','AHOBPR REGISTRANT SEARCH',1),
				('AHOBPR REGISTRANTS','AHOBPR REGISTRANTS','AHOBPR REGISTRANTS',1),
				('AHOBPR REGISTRY FLAG','AHOBPR REGISTRY FLAG','AHOBPR Registry Flag will be used to flag registrants',1),
				('AHOBPR Report Ad Hoc','AHOBPR Report Ad Hoc','AHOBPR Report Ad Hoc',1),
				('AHOBPR REPORT AGING','AHOBPR REPORT AGING','AHOBPR REPORT AGING',1),
				('AHOBPR Report Selected Questionnaire','AHOBPR Report Selected Questionnaire','AHOBPR Report Selected Questionnaire',1),
				('AHOBPR Report Status by Location','AHOBPR Report Status by Location','AHOBPR Report Status by Location',1),
				('AHOBPR SUBPOPULATION FLAG','AHOBPR SUBPOPULATION FLAG','AHOBPR SUBPOPULATION FLAG',1),
				('AHOBPR STATUS HISTORY','AHOBPR STATUS HISTORY', 'AHOBPR STATUS HISTORY',1),
				('AHOBPR REPORT PARTICIPATION','AHOBPR REPORT PARTICIPATION','AHOBPR REPORT PARTICIPATION',1),
				('AHOBPR REPORT CONTACT INFORMATION','AHOBPR REPORT CONTACT INFORMATION','AHOBPR REPORT CONTACT INFORMATION',1),
				('AHOBPR REPORT AD HOC BY LOCATION','AHOBPR REPORT AD HOC BY LOCATION','AHOBPR REPORT AD HOC BY LOCATION',1),
				('AHOBPR REPORT ELIGIBILITY REVIEW','AHOBPR REPORT ELIGIBILITY REVIEW','AHOBPR REPORT ELIGIBILITY REVIEW',1),
				('AHOBPR REGISTRANT INFORMATION','AHOBPR REGISTRANT INFORMATION','AHOBPR REGISTRANT INFORMATION',1),
				('AHOBPR REGISTRY MESSAGES','AHOBPR REGISTRY MESSAGES','AHOBPR REGISTRY MESSAGES',1),
				('AHOBPR ADMIN EDIT TEMPLATE','AHOBPR ADMIN EDIT TEMPLATE','AHOBPR ADMIN EDIT TEMPLATE',1),
				('AHOBPR SEND MESSAGE','AHOBPR SEND MESSAGE','AHOBPR SEND MESSAGE',1),
				('AHOBPR TOOL','AHOBPR TOOL','AHOBPR TOOL',1),
				('AHOBPR ADMIN ADD FLAGS','AHOBPR ADMIN ADD FLAGS','AHOBPR ADMIN ADD FLAGS',1),
				('AHOBPR MY TEMPLATES','AHOBPR MY TEMPLATES','AHOBPR MY TEMPLATES',1),
				('AHOBPR REPORT FOLLOW-UP QUESTIONS','AHOBPR REPORT FOLLOW-UP QUESTIONS','AHOBPR REPORT FOLLOW-UP QUESTIONS',1),
				('AHOBPR MY ACCOUNT','AHOBPR MY ACCOUNT','AHOBPR MY ACCOUNT',1),
				('AHOBPR NAME UPDATE','AHOBPR NAME UPDATE','AHOBPR NAME UPDATE',1),
                ('ERROR PAGE','ERROR PAGE','ERROR PAGE',1),
				('AHOBPR TIMEOUT','AHOBPR TIMEOUT','AHOBPR TIMEOUT',1),
				('AHOBPR SYSTEM CHECK','AHOBPR SYSTEM CHECK','AHOBPR SYSTEM CHECK',1),
				('AHOBPR ERROR PAGE','AHOBPR ERROR PAGE','AHOBPR ERROR PAGE',1),
				('AHOBPR DEPLOYMENT HISTORY','AHOBPR DEPLOYMENT HISTORY','AHOBPR DEPLOYMENT HISTORY',1))
   AS MasterRecord(CODE, NAME, DESCRIPTION_TEXT, SORT_ORDER)
       )
MERGE
  INTO  [dbo].STD_APPPAGE AS Tgt
  USING MasterData AS Src
  ON    Tgt.[CODE] = Src.CODE
 WHEN  NOT MATCHED
 THEN  INSERT
        ([CODE],[NAME],[DESCRIPTION_TEXT],[SORT_ORDER],[INACTIVE_FLAG],[INACTIVE_DATE],[CREATEDBY],[UPDATEDBY],[CREATED],[UPDATED])
        VALUES
        (Src.CODE, Src.NAME, Src.DESCRIPTION_TEXT, Src.SORT_ORDER, 0, NULL,  @UserName, @UserName, @Today,@Today);

-- Load ROLE_PERMISSIONS
SET @AdminRoleId 		= (SELECT ID FROM STD_ROLE WHERE CODE = 'AHOBPR_REGISTRY_MANAGER'); --100
SET @AdvancedUserRoleId = (SELECT ID FROM STD_ROLE WHERE CODE = 'AHOBPR_ADVANCED_USER'); --101
SET @CareTeamRoleId     = (SELECT ID FROM STD_ROLE WHERE CODE = 'AHOBPR_CARE_TEAM');--102
SET @HelpDeskRoleId     = (SELECT ID FROM STD_ROLE WHERE CODE = 'AHOBPR_HELP_DESK');

DELETE FROM [dbo].ROLE_PERMISSIONS WHERE STD_ROLE_ID IN(@AdminRoleId, @AdvancedUserRoleId, @CareTeamRoleId, @HelpDeskRoleId);

WITH  MasterData
AS    ( 
        SELECT *
        FROM   ( VALUES  
        -- Admin User Role
				(AHOBPR.GetAppPageId('AHOBPR ADMIN EDIT USERS'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ADMIN LIST USERS'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ADMIN TRACKING'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ADMIN USER ROLE'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR CHANGE STATUS'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE FULL'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE SECTION'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE SUMMARY'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANT SEARCH'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANTS'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRY FLAG'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Ad Hoc'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT AGING'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Selected Questionnaire'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Status by Location'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SUBPOPULATION FLAG'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR STATUS HISTORY'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT PARTICIPATION'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT CONTACT INFORMATION'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT AD HOC BY LOCATION'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT ELIGIBILITY REVIEW'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT FOLLOW-UP QUESTIONS'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANT INFORMATION'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRY MESSAGES'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ADMIN EDIT TEMPLATE'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SEND MESSAGE'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR TOOL'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ADMIN ADD FLAGS'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR MY ACCOUNT'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR NAME UPDATE'),@AdminRoleId),				
				(AHOBPR.GetAppPageId('AHOBPR DEPLOYMENT HISTORY'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ERROR PAGE'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR TIMEOUT'),@AdminRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SYSTEM CHECK'),@AdminRoleId),
				(AHOBPR.GetAppPageId('ERROR PAGE'),@AdminRoleId),					
        	-- Advanced User Role
				(AHOBPR.GetAppPageId('AHOBPR ADMIN TRACKING'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR CHANGE STATUS'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE FULL'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE SECTION'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE SUMMARY'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANT SEARCH'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANTS'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Ad Hoc'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT AGING'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Selected Questionnaire'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Status by Location'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SUBPOPULATION FLAG'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR STATUS HISTORY'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT PARTICIPATION'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT CONTACT INFORMATION'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT AD HOC BY LOCATION'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT FOLLOW-UP QUESTIONS'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANT INFORMATION'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ADMIN ADD FLAGS'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR MY TEMPLATES'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR MY ACCOUNT'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR DEPLOYMENT HISTORY'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR TIMEOUT'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ERROR PAGE'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SYSTEM CHECK'),@AdvancedUserRoleId),
				(AHOBPR.GetAppPageId('ERROR PAGE'),@AdvancedUserRoleId),				
		-- Care Team User Role
				(AHOBPR.GetAppPageId('AHOBPR ADMIN TRACKING'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE FULL'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE SECTION'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR QUESTIONNAIRE SUMMARY'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANT SEARCH'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANTS'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Ad Hoc'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT AGING'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Selected Questionnaire'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR Report Status by Location'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SUBPOPULATION FLAG'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR STATUS HISTORY'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT CONTACT INFORMATION'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REPORT AD HOC BY LOCATION'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANT INFORMATION'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR MY ACCOUNT'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR DEPLOYMENT HISTORY'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ERROR PAGE'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR TIMEOUT'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SYSTEM CHECK'),@CareTeamRoleId),
				(AHOBPR.GetAppPageId('ERROR PAGE'),@CareTeamRoleId),				
		-- Help Desk User Role
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANT SEARCH'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR REGISTRANT INFORMATION'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ADMIN TRACKING'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SUBPOPULATION FLAG'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR STATUS HISTORY'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR NAME UPDATE'),@HelpDeskRoleId),				
				(AHOBPR.GetAppPageId('AHOBPR MY ACCOUNT'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR DEPLOYMENT HISTORY'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR SYSTEM CHECK'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR TIMEOUT'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('AHOBPR ERROR PAGE'),@HelpDeskRoleId),
				(AHOBPR.GetAppPageId('ERROR PAGE'),@HelpDeskRoleId)		
				)
   AS MasterRecord(AppPageId, RoleId)
       )
MERGE
  INTO  [dbo].ROLE_PERMISSIONS AS Tgt
  USING MasterData AS Src
  ON    Tgt.[STD_APPPAGE_ID] = Src.AppPageId
  AND   Tgt.STD_ROLE_ID = Src.RoleId
 WHEN  NOT MATCHED
 THEN  INSERT
	   ([STD_ROLE_ID],[STD_APPPAGE_ID],[CREATEDBY],[UPDATEDBY],[CREATED],[UPDATED],[INSERT_FLAG],[UPDATE_FLAG],[DELETE_FLAG])
	VALUES(Src.RoleId, Src.AppPageId, @UserName, @UserName, @Today,@Today, 1, 1, 1);
GO

/*
DATE:  7-31-2017
AUTHOR:  S. WANG
DESCRIPTION: Update REGISTRANT_DEPLOYMENT table with appropriate values
*/ 
-- UPDATE to "Verified" if [USER_VERIFIED_FLAG] = 1
UPDATE [AHOBPR].[REGISTRANT_DEPLOYMENT] SET STD_DEPLOYMENT_STAGE_ID = 3
WHERE [USER_VERIFIED_FLAG] = 1 AND IS_ELIGIBLE_FLAG = 1

-- UPDATE to "Rejected" if registrant status is "Reviewed - Not Eligible"
UPDATE [AHOBPR].[REGISTRANT_DEPLOYMENT] SET STD_DEPLOYMENT_STAGE_ID = 6
WHERE REGISTRANT_DEPLOYMENT_ID IN (SELECT REGISTRANT_DEPLOYMENT_ID FROM [AHOBPR].[REGISTRANT_DEPLOYMENT] A INNER JOIN [AHOBPR].[REGISTRANT] B ON A.REGISTRANT_ID = B.REGISTRANT_ID
WHERE A.[USER_ENTERED_FLAG] = 1 AND B.[STD_REGISTRANT_STATUS_ID] = 5)

-- UPDATE to "Accepted" if registrant status is "Reviewed - Eligible"
UPDATE [AHOBPR].[REGISTRANT_DEPLOYMENT] SET STD_DEPLOYMENT_STAGE_ID = 5
WHERE REGISTRANT_DEPLOYMENT_ID IN (SELECT REGISTRANT_DEPLOYMENT_ID FROM [AHOBPR].[REGISTRANT_DEPLOYMENT] A INNER JOIN [AHOBPR].[REGISTRANT] B ON A.REGISTRANT_ID = B.REGISTRANT_ID
WHERE A.[USER_ENTERED_FLAG] = 1 AND B.[STD_REGISTRANT_STATUS_ID] = 8)

-- UPDATE to "refuted" if IS_ELIGIBLE_FLAG = 1 and USER_VERIFIED_FLAG = 0
UPDATE [AHOBPR].[REGISTRANT_DEPLOYMENT] SET STD_DEPLOYMENT_STAGE_ID = 4
WHERE IS_ELIGIBLE_FLAG = 1 and USER_VERIFIED_FLAG = 0

-- UPDATE to "Not Eligible" if [IS_ELIGIBLE_FLAG] = 0
UPDATE [AHOBPR].[REGISTRANT_DEPLOYMENT] SET STD_DEPLOYMENT_STAGE_ID = 2
WHERE IS_ELIGIBLE_FLAG = 0
GO
/* 
Author: L. Mansfield
Date: 9-14-2017
Description: Alter SP_UserAuditLog
*/

-- Alter the existing Procedure SP_UserAuditLog
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

ALTER PROCEDURE [AHOBPR].[SP_UserAuditLog]
  @userId int = NULL,
  @actionTypeId varchar(30) = NULL, -- Action Type ID
  @fromDateText varchar(100) = NULL,
  @toDateText varchar(100) = NULL,
  @lastName varchar(30) = NULL, -- User last name
  @firstName varchar(30) = NULL, -- User first name
  @userRoleId varchar(30) = NULL -- User role ID
AS

BEGIN
 DECLARE @fromDate DATETIME2(0) = NULL,
          @toDate DATETIME2(0) = NULL

 --Temp table for holding the AUDIT_LOG 
  CREATE TABLE #AUDIT_LOG
  ( 	
	  [TYPE_ID]               [tinyint]       NOT NULL,
	  [USER_ID]               [int]           NULL,
	  [REGISTRANT_ID]         [int]           NULL,
	  [NUMBER_OF_REGISTRANTS] [int]           NULL,
	  [MESSAGE]               [varchar](1000) NULL,
	  [CREATED]               [datetime]      NOT NULL
	 );

  INSERT INTO #AUDIT_LOG
    SELECT [TYPE_ID], [USER_ID],[REGISTRANT_ID],[NUMBER_OF_REGISTRANTS],[MESSAGE],[CREATED] 
	  FROM AHOBPR.AUDIT_LOG
	  WHERE USER_ID IS NOT NULL;

  -- Convert date parameters      
  SET @fromDate = dbo.CleanDate(@fromDateText);
  SET @toDate = dbo.AddMaxTime(dbo.CleanDate(@toDateText));

 IF @fromDate IS NOT NULL 
	  DELETE FROM #AUDIT_LOG WHERE CREATED < @fromDate

 IF @toDate IS NOT NULL 
	  DELETE FROM #AUDIT_LOG WHERE CREATED > @toDate
  
 IF @actionTypeId IS NOT NULL AND @actionTypeId != ''
	  DELETE FROM #AUDIT_LOG WHERE TYPE_ID <> @actionTypeId

 IF @userId IS NOT NULL AND @userId != 0
	  DELETE FROM #AUDIT_LOG WHERE USER_ID <> @userId

  IF @userRoleId IS NOT NULL AND @userRoleId != ''
	  DELETE FROM #AUDIT_LOG 
	    WHERE USER_ID NOT IN (SELECT A.USER_ID FROM DBO.USER_ROLES A,  #AUDIT_LOG B 
													  WHERE A.STD_ROLE_ID = @userRoleId AND A.USER_ID = B.USER_ID);
                                  
  IF @lastName IS NOT NULL AND @lastName != ''
	  DELETE FROM #AUDIT_LOG 
	    WHERE USER_ID NOT IN (SELECT A.USER_ID FROM DBO.USERS A,  #AUDIT_LOG B 
													  WHERE A.USER_ID = B.USER_ID AND A.LAST_NAME LIKE '%' + @lastName + '%');
  IF @firstName IS NOT NULL AND @firstName != ''
	  DELETE FROM #AUDIT_LOG 
	    WHERE USER_ID NOT IN (SELECT A.USER_ID FROM DBO.USERS A,  #AUDIT_LOG B 
													  WHERE A.USER_ID = B.USER_ID AND A.FIRST_NAME LIKE '%' + @firstName + '%');

  SELECT  A.CREATED, 
          B.DESCRIPTION AS ACTION_TYPE, 
          CASE  WHEN A.TYPE_ID = 2 OR A.TYPE_ID = 12 OR A.TYPE_ID = 13 
                THEN CASE WHEN C.LAST_NAME IS NULL THEN ''
					                ELSE ISNULL(C.LAST_NAME, '') + ', ' + ISNULL(C.FIRST_NAME, '') + ' ' + ISNULL(C.MIDDLE_NAME, '')
					                END
				WHEN A.TYPE_ID = 11  THEN ''
				WHEN A.TYPE_ID = 10  THEN 
					CASE WHEN C.LAST_NAME IS NULL THEN A.MESSAGE
					ELSE A.MESSAGE + ', ' + ISNULL(C.LAST_NAME, '') + ', ' + ISNULL(C.FIRST_NAME, '') + ' ' + ISNULL(C.MIDDLE_NAME, '')
					END
	            ELSE A.MESSAGE 
	      END AS DETAILS, 
          REPLACE(CAST(ISNULL(A.NUMBER_OF_REGISTRANTS, 0) AS varchar(10)), '0', '') AS NUM_REGISTRANTS,
		  D.FULL_NAME, A.USER_ID
  FROM #AUDIT_LOG A 
	LEFT OUTER JOIN [AHOBPR].[AUDIT_LOG_TYPE] B ON A.TYPE_ID = B.TYPE_ID
	LEFT OUTER JOIN [AHOBPR].[REGISTRANT] C ON A.REGISTRANT_ID = C.REGISTRANT_ID
	LEFT OUTER JOIN DBO.USERS D ON A.USER_ID = D.USER_ID;
           		
END

GO


---------------------------------------------------------------------------------------------------------
print N'Update complete.';
print 'Completed ' + CAST(GETDATE() as varchar(20)) + '.';
---------------------------------------------------------------------------------------------------------
go

update  [dbo].[DatabaseChangeLog]
  set   ScriptDescription = 'Script Completed Successfully.'
  where [MajorReleaseNumber]  = '$(MajorReleaseNumber)'
    and [MinorReleaseNumber]  = '$(MinorReleaseNumber)'
    and [PointReleaseNumber]  = '$(PointReleaseNumber)'
    and [ScriptName]          = '$(ScriptName)'
    and TargetRegistry        = 'AHOBPR';
   
go
select * from [dbo].[DatabaseChangeLog]
where TargetRegistry        = 'AHOBPR'
  and [MajorReleaseNumber] = '$(MajorReleaseNumber)'
  and [MinorReleaseNumber] = '$(MinorReleaseNumber)'
  and [PointReleaseNumber] = '$(PointReleaseNumber)'
  and [ScriptName] = '$(ScriptName)';  
go