﻿using System;
using System.ServiceProcess;
using System.Configuration;
using System.Timers;
using Quartz;
using Quartz.Impl;
using NLog;
using System.Threading.Tasks;
using System.Collections.Specialized;

namespace BENSClientService
{
    public partial class Scheduler : ServiceBase
    {
        static Logger logger = LogManager.GetCurrentClassLogger();
       // Timer timer;
        BensMQClient client;
        BensMQProperties props;
        
        int nightlyJobHour;
        int nightlyJobMinute;
        int connectionInterval;
        public Scheduler()
        {
            logger.Debug("Constructing Scheduler");
            InitializeComponent();
            logger.Debug("Setting BensMQProperties");
            props = new BensMQProperties(ConfigurationManager.AppSettings);
            logger.Debug("Initializing RestClient");
            RestfulClient restClient = CreateRestClient();
            logger.Debug($"Setting MQCHLLIB to {LogEntryCleaner.Clean(props.MQCHLLIB)}");
            Environment.SetEnvironmentVariable("MQCHLLIB", props.MQCHLLIB);
            logger.Debug($"Setting MQCHLTAB to {LogEntryCleaner.Clean(props.MQCHLTAB)}");
            Environment.SetEnvironmentVariable("MQCHLTAB", props.MQCHLTAB);
            logger.Debug("Creating BensMQClient");
            client = new BensMQClient(props, restClient);
            SetNightlyJobVariables();
            connectionInterval = int.Parse(ConfigurationManager.AppSettings["connectionIntervalInSeconds"]);
            logger.Debug("Created.");
        }

        private static RestfulClient CreateRestClient()
        {
            return new RestfulClient(ConfigurationManager.AppSettings["RestServiceUrl"], ConfigurationManager.AppSettings["secretKey"]);
        }

        private void SetNightlyJobVariables()
        {
            nightlyJobHour = int.Parse(ConfigurationManager.AppSettings["hour"]);
            nightlyJobMinute = int.Parse(ConfigurationManager.AppSettings["minute"]);
        }

        protected override void OnStart(string[] args)
        {
            logger.Info("Starting Bens Client Service");
            try
            {
                RunProgram().GetAwaiter().GetResult();
            }
            catch (ArgumentNullException)
            {
                logger.Error("NullArguments padded to OnStart(). Possible that configuration file is missing.");
            }
        }

        public BensMQClient Client { get { return client; } }
        protected override void OnStop()
        {
            logger.Info("Stopping Bens Client Service");
           
            //timer.Enabled = false;
        }
        private async Task RunProgram()
        {
            try
            {
                NameValueCollection qprops = new NameValueCollection
                {
                    {"quartz.serializer.type", "binary" }
                };
                StdSchedulerFactory factory = new StdSchedulerFactory(qprops);
                IScheduler scheduler = await factory.GetScheduler();
                //start job
                await scheduler.Start();
                await ScheduleNightlyJob(scheduler);
                await ScheduleBensConnectionJob(scheduler);
            }
            catch (SchedulerException ex)
            {
                logger.Error($"In OnTimedEvent: {ex.Message}");
            }
        }

        private async Task ScheduleBensConnectionJob(IScheduler scheduler)
        {
            logger.Debug("Scheduling Bens connection");
            //Job to Contact the BENS Message Queue
            var job = JobBuilder.Create<ContactBensJob>()
                .WithIdentity("BensMQJob")
                .Build();
            job.JobDataMap["client"] = client;
            //Trigger the job to run now, and then repeat every X seconds
            var trigger = TriggerBuilder.Create()
                .WithIdentity("BensMQTrigger")
                .StartNow()
                .WithSimpleSchedule(b => b
                    .WithIntervalInSeconds(connectionInterval)
                    .RepeatForever())
                .Build();

            await scheduler.ScheduleJob(job, trigger);
        }

        private async Task ScheduleNightlyJob(IScheduler scheduler)
        {
            logger.Debug("Scheduling Nightly Job");
            //define the job and tie it to QueueResolver class
            IJobDetail job = JobBuilder.Create<ResolveQueueJob>()
                .WithIdentity("nightlyjob")
                .Build();
            job.JobDataMap["client"] = CreateRestClient();
            //Job will run at nightlyJobHour/nightlyJobMinute
            var trigger = TriggerBuilder.Create()
                .WithIdentity("nightlytrigger")
                .WithSchedule(CronScheduleBuilder
                    .DailyAtHourAndMinute(nightlyJobHour, nightlyJobMinute)
                    .WithMisfireHandlingInstructionFireAndProceed())
                .ForJob("nightlyjob")
                .Build();

            await scheduler.ScheduleJob(job, trigger);
        }
    }

}
