﻿using IBM.XMS;
using NLog;
using System;
using System.Xml.Linq;
using System.Data;
using System.Threading;
using System.Linq;

namespace BENSClientService
{
    public class BensMQClient
    {
        private static Logger logger = LogManager.GetCurrentClassLogger();
        BensMQProperties _mqprops;
        RestfulClient restfulClient;

        public BensMQClient(BensMQProperties mqProperties, RestfulClient restClient)
        {
            logger.Debug("Creating BensMQClient");
            _mqprops = mqProperties;
            this.restfulClient = restClient;
            logger.Debug("Created");
            IsClientConnected = false;
        }

        public bool IsClientConnected { get; set; }
        /// <summary>
        /// Method to connect to queue manager and receive messages.
        /// </summary>
        public void ReceiveMessages()
        {
            logger.Debug("In BensMQClient.ReceiveMessages");
            XMSFactoryFactory factoryFactory;
            IConnectionFactory cf;
            IConnection connectionWMQ;
            ISession sessionWMQ;
            IDestination destination;
            IMessageConsumer durableSubscriber;
            if (IsClientConnected)
            {
                logger.Info("Connection to MQ already open. Exiting ReceiveMessages()");
                return;
            }
            try
            {
                IsClientConnected = true;
                // Get an instance of factory.
                logger.Debug("Creating instance of XMSFactory");
                factoryFactory = XMSFactoryFactory.GetInstance(XMSC.CT_WMQ);

                logger.Debug("Setting properties for XMSFactory");
                // Set the properties
                cf = _mqprops.GetFactory(factoryFactory);

                // Create connection.
                logger.Debug("Creating connection");
                connectionWMQ = cf.CreateConnection();

                logger.Debug("Connection created.");
                logger.Debug("Client ID:  " + connectionWMQ.ClientID);


                // Create session
                logger.Debug("Creating a session");
                sessionWMQ = connectionWMQ.CreateSession(false, AcknowledgeMode.AutoAcknowledge);
                logger.Debug("Session created.");

                // Create destination
                var destinationName = "topic://" + _mqprops.TopicName;
                logger.Debug($"Setting destination to '{destinationName}'");
                destination = sessionWMQ.CreateTopic(destinationName);
                logger.Debug("Destination created");

                // Create consumer
                logger.Debug("Creating consumer");
                durableSubscriber = sessionWMQ.CreateDurableSubscriber(destination, _mqprops.SubscriptionName, _mqprops.Selector, false);
                
                logger.Debug("Message Consumer created. ");
                // Start the connection to receive messages.
                logger.Debug("Creating MessageListener.");
                MessageListener messageListener = new MessageListener(OnNewMessageCallback);
                durableSubscriber.MessageListener = messageListener;
                logger.Debug("Message Listener set. Starting the connection now.");
                logger.Debug("Starting");
                connectionWMQ.Start();
                logger.Debug("Started");
                // Wait for messages. Exit if no message by then
                logger.Debug($"Loitering for {_mqprops.Timeout / 1000} seconds.");
                Thread.Sleep(_mqprops.Timeout);
                // Cleanup
                logger.Debug("Closing subscription.");
                durableSubscriber.Close();
                logger.Debug("Closed");
                logger.Debug("Disposing destination");
                destination.Dispose();
                logger.Debug("Done");
                logger.Debug("Disposing of session");
                sessionWMQ.Dispose();
                logger.Debug("Done");
                logger.Debug("Closing connection");
                connectionWMQ.Close();
                IsClientConnected = false;
                logger.Debug("Done");
            }
            catch (Exception ex)
            {
                IsClientConnected = false;
                logger.Error($"Exception in ReceiveMessages()");
                logger.Debug($"Exception: {ex.Message}");
            }
        }

        private void OnNewMessageCallback(IMessage message)
        {
            // The try block below is unlikely to throw a runtime exception, but as
            // a general good practice an asynchronous consumer's message listener or 
            // callback should catch a potential runtime exception, and optionally 
            // divert the message in question to an application-specific destination.
            try
            {
                logger.Debug($"Message received: {message}");
                ProcessMessage((ITextMessage)message);
                logger.Debug("Message Processed.");
            }
            catch (Exception ex)
            {
                logger.Error("Exception caught in OnMessageCallback: {0}", ex.Message);

                // We have atleast two choices now - (1) rethrow ex. In this case the control
                // passes back to XMS and which may attempt to redeliver the message, depending
                // on session's acknowledge mode, or (2) terminate the process. Alternatively, we
                // may divert the message to an application-specific destination.

                // Terminate the current process
                Environment.Exit(1);
            }
        }

        public bool ProcessMessage(ITextMessage message)
        {
            var addedToDB = false;
            logger.Debug("In ProcessMessage");
            logger.Debug("Calling ExtractEdipi");
            var edipi = ExtractEdipi(message.Text);
            if (!string.IsNullOrEmpty(edipi))
            {
                logger.Debug($"Calling AddEdipiToDatabase with edipi: {edipi}");

                 addedToDB = restfulClient.AddEdipiToDatabase(edipi);
                if (addedToDB)
                {
                    logger.Info($"EDIPI: {edipi} added to AHOBPR database.");
                }
                else
                {
                    logger.Info($"ERROR EDIPI: {edipi} NOT added to AHOBPR database.");
                }
            }
            return addedToDB;
        }
        public string ExtractEdipi(string message)
        {
            logger.Debug("In ExtractEdipi");
            var nodes = from ids in XElement.Parse(message).Elements("primaryId")
                        where (string)ids.Element("idType") == "EDI_PI"
                        select ids.Element("id").Value;
            return nodes != null ? nodes.First() : string.Empty;
        }
    }
}
