﻿using System;
using System.Collections.Generic;
using System.Data.Linq;
using System.Linq;
using System.Text;
using System.Web;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessAHOBPR;
using System.Reflection;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    public class AHOBPRUserManager : AHOBPRBaseBO
    {
        private AHOBPRShared _sharedManager = new AHOBPRShared();
        private int _totalCount = 0;

        public AHOBPR_USER GetUser(int id)
        {
            AHOBPR_USER user = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForAhobprUser(_dbAhopbr);

                user = (from e in _dbAhopbr.AHOBPR_USERs
                        where e.USER_ID == id
                        select e).FirstOrDefault();

            }
            return user;
        }

        private void SetLoadWithForAhobprUser(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<AHOBPR_USER>(e => e.STD_PRIMARY_JOB);
            lo.LoadWith<AHOBPR_USER>(e => e.STD_SECONDARY_JOB);
            lo.LoadWith<AHOBPR_USER>(e => e.STD_WORK_LOCATION);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }


        public void ApproveAccount(int id)
        {
            ApproveAccount(GetUser(id));
        }

        public void DisableAccount(int id)
        {
            EnableAccount(GetUser(id), false);
        }

        public void EnableAccount(int id)
        {
            AHOBPR_USER user = GetUser(id);
            if (user == null)
            {
                USER mainUser = (new UsersManager()).SelectByID("AHOBPR", id);
                CreateAccount(mainUser, true, false, null, null, null, null);
                user = GetUser(id);
            }
            EnableAccount(user, true);
        }

        public void ApproveAccount(AHOBPR_USER user)
        {
            user.SetAsChangeTrackingRoot();
            user.PENDING_FLAG = false;
            user.LAST_LOGIN = null;
            user.SetAsUpdateOnSubmit();
            UpdateUser(user);
        }

        public void EnableAccount(AHOBPR_USER user, bool enabled)
        {
            user.SetAsChangeTrackingRoot();
            user.ACTIVE_FLAG = enabled;
            user.LAST_LOGIN = null;
            user.SetAsUpdateOnSubmit();
            UpdateUser(user);
        }

        public void UpdateLogin(AHOBPR_USER user)
        {
            user.SetAsChangeTrackingRoot();
            user.LAST_LOGIN = DateTime.Now;
            user.SetAsUpdateOnSubmit();
            UpdateUser(user);
        }

        public REGISTRANT GetUserById(int userId)
        {
            REGISTRANT registrant = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForUser(_dbAhopbr);

                registrant = (from e in _dbAhopbr.REGISTRANTs
                              where e.REGISTRANT_ID == userId
                              select e).FirstOrDefault();

            }
            return registrant;
        }

        private void SetLoadWithForUser(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            //lo.LoadWith<AHOBPR_MAIN_USER>(e => e.);
            //lo.LoadWith<USER>(e => e.USER_RE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        public void UpdateLastVistaLogin(AHOBPR_USER user)
        {
            user.SetAsChangeTrackingRoot();
            user.ACTIVE_FLAG = true;
            user.LAST_LOGIN = DateTime.Now;
            user.LAST_VISTA_LOGIN = DateTime.Now;
            user.SetAsUpdateOnSubmit();
            UpdateUser(user);
        }

        public bool CreateAccount(
            String adUsername,
            String fullName,
            String firstName,
            String middleName,
            String surname,
            String maidenName,
            String employeeNumber,
            String jobTitle,
            String emailAddress,
            String phone,
            String fax,
            bool active,
            bool pending,
            DateTime? lastVistaLogin,
            int? primaryJobId,
            int? secondaryJobId,
            int? workLocationId)
        {
            adUsername = adUsername.Trim();
            USER user;
            UsersManager usersManager;
            AHOBPRUserManager _ahobprUserManager = new AHOBPRUserManager();

            if (!(System.Threading.Thread.CurrentPrincipal is UserAccountManager))
            {
                System.Threading.Thread.CurrentPrincipal = new UserAccountManager("", "");
            }
            usersManager = new UsersManager();
            user = usersManager.SelectByUsername(adUsername);
            if (user == null)
            {
                //If the user doesn't exist in the standard schema table, add them:
                user = new USER()
                {
                    ACCOUNT_EXPIRE_DATE = DateTime.Now.AddYears(1),
                    ACCOUNT_LOCK_DATE = null,
                    CREATED = DateTime.Now,
                    CREATEDBY = "AHOBPR",
                    EMAIL_ADDRESS = emailAddress,
                    EMPLOYEE_NUMBER = employeeNumber,
                    FIRST_NAME = firstName,
                    FULL_NAME = fullName,
                    JOB_TITLE = jobTitle,
                    LAST_NAME = surname,
                    MAIDEN_NAME = maidenName,
                    MIDDLE_NAME = middleName,
                    TELEPHONE_NUMBER = phone,
                    FAX_NUMBER = fax,
                    UPDATED = DateTime.Now,
                    UPDATEDBY = "AHOBPR",
                    USERNAME = adUsername,
                    PASSWORD_CREATE_DATE = DateTime.Now,
                    PASSWORD_CHANGE_DATE = DateTime.Now,
                    INACTIVE_FLAG = false
                };
                user.SetAsChangeTrackingRoot();
                user.SetAsInsertOnSubmit();

                usersManager.Update(user);

                user = usersManager.SelectByUsername(adUsername);

            }
            //If successfully created or if an account already exists in the standard table,
            //then add a row to our user table:
            return CreateAccount(user, active, pending, lastVistaLogin, primaryJobId, secondaryJobId, workLocationId);
        }

        public void AddUserToRegistry(USER user, UsersManager usersManager)
        {
            usersManager.AddUserToRegistry(user, "AHOBPR");

            USER_ROLE userRole = new USER_ROLE();
            userRole.STD_ROLE_ID = AHOBPRShared.GetUserRoleId(AHOBPRGlobal.AhobprUserRoleCareTeamCode);
            userRole.USER_ID = user.USER_ID;
            userRole.STD_INSTITUTION_ID = 1000001;
            user.SetAsChangeTrackingRoot();
            user.USER_ROLEs.Add(userRole);
            userRole.SetAsInsertOnSubmit();

            usersManager.Update(user);
        }

        public bool CreateAccount(USER user, bool active, bool pending, DateTime? lastVistaLogin, int? primaryJobId, int? secondaryJobId, int? workLocationId)
        {
            AddUserToRegistry(user, new UsersManager());

            using (_dbAhopbr = GetDataContext())
            {
                AHOBPR_USER bprUser = new AHOBPR_USER()
                {
                    ACTIVE_FLAG = active,
                    USER_ID = user.USER_ID,
                    CREATED = DateTime.Now,
                    CREATEDBY = user.USER_ID.ToString(),
                    UPDATED = DateTime.Now,
                    UPDATEDBY = user.USER_ID.ToString(),
                    PENDING_FLAG = pending,
                    PRIMARY_JOB_ID = primaryJobId,
                    SECONDARY_JOB_ID = secondaryJobId,
                    WORK_LOCATION_ID = workLocationId
                };

                if (lastVistaLogin != null)
                {
                    bprUser.LAST_LOGIN = lastVistaLogin.Value;
                    bprUser.LAST_VISTA_LOGIN = lastVistaLogin.Value;
                }
                _dbAhopbr.AHOBPR_USERs.InsertOnSubmit(bprUser);
                _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);
                return true;
            }
        }

        public IList<SP_UserListForSendEmailResult> SearchUserSendEmail(string lastName, string firstName,
        string userRole, string userStatus, string sortExpression, int maximumRows, int startRowIndex)
        {
            IQueryable<SP_UserListForSendEmailResult> mainQueryGridView;

            const string gridDefaultSort = @"FULL_NAME";

            if (string.IsNullOrEmpty(sortExpression))
            {
                sortExpression = string.Format("{0}", gridDefaultSort);
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.CommandTimeout = AHOBPRGlobal.TimeOutForReports;
                //call stored procedure
                var results = _dbAhopbr.SP_UserListForSendEmail(lastName, firstName, userRole, userStatus).ToList();

                mainQueryGridView = results.AsQueryable();
                //sort
                _totalCount = mainQueryGridView.Count();
            }

            return (mainQueryGridView).SortAndPage(sortExpression, maximumRows, startRowIndex).ToList<SP_UserListForSendEmailResult>();

        }

        /// <summary>
        /// Get the total count for search user 
        /// </summary>
        /// <param name="lastName"></param>
        /// <param name="firstName"></param>
        /// <param name="userRole"></param>
        /// <param name="userStatus"></param>
        /// <returns></returns>
        public int SearchUserSendEmailCount(string lastName, string firstName,
                    string userRole, string userStatus, string sortExpression, int maximumRows, int startRowIndex)
        {

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.CommandTimeout = AHOBPRGlobal.TimeOutForReports;

                //call stored procedure
                return _dbAhopbr.SP_UserListForSendEmail(lastName, firstName, userRole, userStatus).Count();
            }
        }


        /// <summary>
        /// User search
        /// </summary>
        /// <param name="username"></param>
        /// <param name="pending"></param>
        /// <param name="active"></param>
        /// <returns></returns>
        public IQueryable<GuiAhobprUser> UserSearch(String username, bool? pending, bool? active)
        {
            _dbAhopbr = GetDataContext();

            List<GuiAhobprUser> list = new List<GuiAhobprUser>();
            UsersManager usersManager = new UsersManager();
            IEnumerable<GuiAhobprUser> q = from su in usersManager.SelectActive("AHOBPR", "USER_ID", 0, 10000)
                                           join au in _dbAhopbr.AHOBPR_USERs on su.USER_ID equals au.USER_ID into auu
                                           from sub in auu.DefaultIfEmpty()
                                           select new GuiAhobprUser()
                                           {
                                               MainUser = su,
                                               AhobprUser = (sub == null ? new AHOBPR_USER() : sub)

                                           };
            //Filter by username:
            if (!String.IsNullOrEmpty(username))
            {
                q = from dau in q
                    where dau.MainUser.USERNAME.ToLower().Contains(username.ToLower()) ||
                          dau.MainUser.FULL_NAME.ToLower().Contains(username.ToLower())
                    select dau;
            }

            //Filter by pending:
            if (pending.HasValue)
            {
                q = from dau in q
                    where dau.AhobprUser.PENDING_FLAG == pending.Value
                    select dau;
            }

            //Filter by enabled:
            if (active.HasValue)
            {
                q = from dau in q
                    where dau.AhobprUser.ACTIVE_FLAG == active.Value
                    select dau;
            }

            return q.AsQueryable();
        }

        public int UpdateUser(AHOBPR_USER ahobprUser)
        {
            int returnStatus = 0;
            AHOBPR_USER existingUser = GetUser(ahobprUser.USER_ID);

            if (existingUser != null)
            {
                existingUser.SetAsChangeTrackingRoot();

                if (ahobprUser.STD_PRIMARY_JOB != null)
                {
                    if (existingUser.STD_PRIMARY_JOB != null)
                    {
                        if (existingUser.STD_PRIMARY_JOB.STD_PRIMARY_JOB_ID != ahobprUser.STD_PRIMARY_JOB.STD_PRIMARY_JOB_ID)
                        {
                            existingUser.STD_PRIMARY_JOB = ahobprUser.STD_PRIMARY_JOB;
                        }
                    }else
                    {
                        existingUser.STD_PRIMARY_JOB = ahobprUser.STD_PRIMARY_JOB;
                    }
                }
              

                if (ahobprUser.STD_SECONDARY_JOB != null)
                {
                    if (existingUser.STD_SECONDARY_JOB != null)
                    {
                        if (existingUser.STD_SECONDARY_JOB.STD_SECONDARY_JOB_ID != ahobprUser.STD_SECONDARY_JOB.STD_SECONDARY_JOB_ID)
                        {
                            existingUser.STD_SECONDARY_JOB = ahobprUser.STD_SECONDARY_JOB;
                        }
                    }else
                    {
                        existingUser.STD_SECONDARY_JOB = ahobprUser.STD_SECONDARY_JOB;
                    }
                }
                

                if (ahobprUser.STD_WORK_LOCATION != null)
                {
                    if (existingUser.STD_WORK_LOCATION != null)
                    {
                        if (existingUser.STD_WORK_LOCATION.STD_WORK_LOCATION_ID != ahobprUser.STD_WORK_LOCATION.STD_WORK_LOCATION_ID)
                        {
                            existingUser.STD_WORK_LOCATION = ahobprUser.STD_WORK_LOCATION;
                        }
                        
                    }else
                    {
                        existingUser.STD_WORK_LOCATION = ahobprUser.STD_WORK_LOCATION;
                    }     
                    
                }


                existingUser.PENDING_FLAG = ahobprUser.PENDING_FLAG;
                existingUser.ACTIVE_FLAG = ahobprUser.ACTIVE_FLAG;
                existingUser.LAST_LOGIN = ahobprUser.LAST_LOGIN;
                existingUser.LAST_VISTA_LOGIN = ahobprUser.LAST_VISTA_LOGIN;
                existingUser.SetAsUpdateOnSubmit();


                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    existingUser.SynchroniseWithDataContext(_dbAhopbr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                    if (changeSet.Inserts.Count > 0 || changeSet.Updates.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                            returnStatus = 1;

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                }
            }
            return returnStatus;
        }

        public void UpdatePrimaryJob(AHOBPR_USER user, int? id)
        {
            if (id != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    user.STD_PRIMARY_JOB = _dbAhopbr.STD_PRIMARY_JOBs.FirstOrDefault(j => j.STD_PRIMARY_JOB_ID == id);
                }
            }
        }

        public void UpdateSecondaryJob(AHOBPR_USER user, int? id)
        {
            if (id != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    user.STD_SECONDARY_JOB = _dbAhopbr.STD_SECONDARY_JOBs.FirstOrDefault(j => j.STD_SECONDARY_JOB_ID == id);
                }
            }
        }

        public void UpdateWorkLocation(AHOBPR_USER user, int? id)
        {
            if (id != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    user.STD_WORK_LOCATION = _dbAhopbr.STD_WORK_LOCATIONs.FirstOrDefault(j => j.STD_WORK_LOCATION_ID == id);
                }
            }
        }

        /// <summary>
        /// Return Dictionary of current user's roles
        /// </summary>
        /// <param name="roleManager"></param>
        /// <param name="configEntry"></param>
        /// <param name="currentUser"></param>
        /// <returns></returns>
        public Dictionary<int, string> GetCurrentUserRoles(RoleManager roleManager, string configEntry, UserAccountManager currentUser)
        {
            Dictionary<int, string> registryRoles = roleManager
                .SelectActiveByRegistryCode(configEntry)
                .ToDictionary(o => o.ID, o => o.CODE);
            var dupeRoles = registryRoles.Clone();
            int[] roles = currentUser.GetRoles();
            foreach (int key in dupeRoles.Keys)
            {
                if (!roles.Contains(key))
                {
                    registryRoles.Remove(key);
                }
            }
            return registryRoles;
        }

        #region State_Filter
        public bool ShouldRedirectInvalidAdvancedUsers(Dictionary<int, string> roles, int currentUserId)
        {
            bool shouldRedirectUser = false;
            //If an Advanced User hasn't set the states they are to filter on (set in the MyAccount page) then they need to be booted
            if (IsAdvancedUserButNotRegistryManager(roles))
            {
                USER_STATES_FILTER stateFilter = GetStatesFilter(currentUserId);
                if (stateFilter == null || (stateFilter.STATES == String.Empty))
                {
                    shouldRedirectUser = true;
                }
            }
            return shouldRedirectUser;
        }
        /// <summary>
        /// Check if User has the Advanced User Role, but isn't a Registry Manager
        /// </summary>
        /// <param name="roles"></param>
        /// <returns></returns>
        public bool IsAdvancedUserButNotRegistryManager(Dictionary<int, string> roles)
        {
            return (roles.Values.Contains(AHOBPRGlobal.AhobprUserRoleAdvancedUserCode) && !roles.Values.Contains(AHOBPRGlobal.AhobprUserRoleRegistryManagerCode));
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public USER_STATES_FILTER GetStatesFilter(int userId)
        {
            USER_STATES_FILTER userStatesFilter = null;
            using (_dbAhopbr = GetDataContext())
            {
                userStatesFilter = _dbAhopbr.USER_STATES_FILTERs.FirstOrDefault(u => u.USER_ID == userId);
            }
            return userStatesFilter;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="userStateFilter"></param>
        /// <returns></returns>
        public int UpdateUserStateFilter(USER_STATES_FILTER userStateFilter)
        {
            int returnStatus = 0;

            USER_STATES_FILTER existingStateFilter = GetStatesFilter(userStateFilter.USER_ID);
            if (existingStateFilter == null)
            {
                existingStateFilter = userStateFilter;
                existingStateFilter.SetAsChangeTrackingRoot();
                existingStateFilter.SetAsInsertOnSubmit();
            }
            else
            {
                existingStateFilter.SetAsChangeTrackingRoot();
                existingStateFilter.STATES = userStateFilter.STATES;
                existingStateFilter.SetAsUpdateOnSubmit();
            }


            if (existingStateFilter != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    existingStateFilter.SynchroniseWithDataContext(_dbAhopbr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                    if (changeSet.Inserts.Count > 0 || changeSet.Updates.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                            returnStatus = 1;

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                }
            }
            return returnStatus;
        }

        #endregion
    }

    public class GuiAhobprUser
    {
        public AHOBPR_USER AhobprUser { get; set; }
        public USER MainUser { get; set; }
        public String UserId { get { return MainUser.USER_ID.ToString(); } }
        public String Username { get { return MainUser.USERNAME; } }
        public String Name { get { return MainUser.FULL_NAME; } }
        public String LastLogin { get { return AhobprUser.LAST_LOGIN.ToString(); } }
        public String LastVistaLogin { get { return AhobprUser.LAST_VISTA_LOGIN.ToString(); } }
        public override string ToString()
        {
            return MainUser.USER_ID + ":" + MainUser.USERNAME;
        }
    }
}