﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessManagerAHOBPR.ValueTypes;
using System.Reflection;
using VeteransAffairs.Registries.BusinessManager;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    public class AHOBPRRegistrantManager : AHOBPRBaseBO
    {
        private AHOBPRShared _sharedManager = new AHOBPRShared();
        private int _totalCount = 0;

        #region Methods for SelectByStatus

        public IEnumerable<REGISTRANT> SelectByStatus(int statusId, string sort, int startRow, int maxRows)
        {
            if (String.IsNullOrEmpty(sort))
                sort = "LAST_NAME";

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForListViews(_dbAhopbr);
                IEnumerable<REGISTRANT> entities;
                entities = SelectByStatusLinqFilter(statusId).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }
        
        public int SelectByStatusCount(int statusId, string sort, int startRow, int maxRows)
        {
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForListViews(_dbAhopbr);
                return SelectByStatusLinqFilter(statusId).Count();
            }
        }
        private IQueryable<REGISTRANT> SelectByStatusLinqFilter(int statusId)
        {
            IQueryable<REGISTRANT> linqFilter = from e in _dbAhopbr.REGISTRANTs
                                                where e.STD_REGISTRANT_STATUS_ID == statusId
                                                select e;
            return linqFilter;
        }

        private void SetLoadWithForListViews(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.STD_REGISTRANT_STATUS);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }
        #endregion

        #region Methods for SearchRegistrants

        /// <summary>
        /// 
        /// </summary>
        /// <param name="lastName"></param>
        /// <param name="last4SSN"></param>
        /// <param name="dob"></param>
        /// <param name="registryStatusId"></param>
        /// <param name="registryFlagId"></param>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public IEnumerable<REGISTRANT> SearchRegistrants(string lastName, string last4SSN, string dob, string registryStatusId, 
                                            string registryFlagId, string registrantId, string sort, int startRow, int maxRows, int totalRecords)
        {
            if (String.IsNullOrEmpty(sort))
                sort = "LAST_NAME, FIRST_NAME";

            //Added as a precautionary measure per RTC 460132 below
            try
            {
                using (_dbAhopbr = GetDataContext())
                {
                    SetLoadWithForSearchRegistrants(_dbAhopbr);
                    IQueryable<REGISTRANT> entities = SearchRegistrantsLinqFilter(lastName, last4SSN, dob, registryStatusId, registryFlagId, registrantId);

                    _totalCount = totalRecords = entities.Count();
                    entities = entities.SortAndPage(sort, maxRows, startRow);
                    return entities.ToList<REGISTRANT>();
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Search Error", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);

                _totalCount = totalRecords = 0;
                return new List<REGISTRANT>();
            }
        }

        public IEnumerable<REGISTRANT> SearchRegistrantsForChangeStatus(string lastName, string last4SSN, string registryStatusIds,
                                                    string adminFlag, string adminFlagFromDate, string adminFlagToDate, bool exludeReviewHecAdminFlag,
                                                    string statusFromDate, string statusToDate, bool hasStateFilter, string stateFilter, string sort,
                                                    int startRow, int maxRows, int totalRecords)
        {
            if (String.IsNullOrEmpty(sort))
                sort = "LAST_NAME, FIRST_NAME";
            //RTC 460132 - Added Try/Catch for safety, but defect fix should be below.
            try
			{
			using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForChangeStatus(_dbAhopbr);
                IQueryable<REGISTRANT> entities = SearchRegistrantsChangeStatusLinqFilter(lastName, last4SSN, registryStatusIds, adminFlag, adminFlagFromDate, adminFlagToDate, exludeReviewHecAdminFlag, statusFromDate, statusToDate, hasStateFilter, stateFilter);

                _totalCount = totalRecords = entities.Count();
                //output parmeter used to populate the total number of records label on the Changes Status search page
                totalRecords = _totalCount;
				entities = entities.SortAndPage(sort, maxRows, startRow);
                return entities.ToList<REGISTRANT>();
            }
			}
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Search Error", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);

                _totalCount = totalRecords = 0;
                return new List<REGISTRANT>();
            }
        }
        public Dictionary<string, object> ApplyChangeStatusAdvancedUserRules(string registry, RoleManager roleManager,
                                            UserAccountManager currentUser, List<string> baseRegistrantStatusValues, List<string> selectedRegistrantValues)
        {
            var filterInfo = new Dictionary<string, object>()
            {
                {"StatusFilter" , string.Join(",", selectedRegistrantValues) },
                {"StateFilter", new ChangeStatusStatesFilter() }
            };
            ChangeStatusStatesFilter stateFilterInfo = new ChangeStatusStatesFilter();
            AHOBPRUserManager ahobprUserManager = new AHOBPRUserManager();
            Dictionary<int, string> currentUserRoles = ahobprUserManager.GetCurrentUserRoles(roleManager, registry, currentUser);
            bool isAdvancedUserButNotRegistryManager = ahobprUserManager.IsAdvancedUserButNotRegistryManager(currentUserRoles);
            try
            {
                stateFilterInfo = GetStateFilter(isAdvancedUserButNotRegistryManager, ahobprUserManager, currentUser.UserId);
                filterInfo["StateFilter"] = stateFilterInfo;
            }
            catch (Exception e)
            {
                throw new Exception("Failure in AHOBPRRegistrantManager. Current User: {" + currentUser + "}, ", e);
            }
            //If the user is an advanced user but not a registry manager, add all of the status filters
            if (isAdvancedUserButNotRegistryManager)
            {
                if (selectedRegistrantValues.Count() == 0)
                {
                    filterInfo["StatusFilter"] = string.Join(",", baseRegistrantStatusValues);
                }else
                {
                    filterInfo["StatusFilter"] = string.Join(",", selectedRegistrantValues);
                }
            }
            
            return filterInfo;
        }

        public ChangeStatusStatesFilter GetStateFilter(bool shouldApplyFilter, AHOBPRUserManager ahobprUserManager, int userId)
        {
            var cssFilter = new ChangeStatusStatesFilter();
            USER_STATES_FILTER stateFilter = ahobprUserManager.GetStatesFilter(userId);
            //Filter should only be set for an advanced user who is NOT also a registry manager
            if (shouldApplyFilter)
            {
                if (stateFilter != null)
                {
                    cssFilter.HasStateFilter = true;
                    cssFilter.StateFilter = stateFilter.STATES;
                }
                else
                {
                    throw new Exception("UserId {" + userId + "} is an advanced user without any user state filters set.");
                }
            }

            return cssFilter;
        }

        public int SearchRegistrantsForChangeStatusCount(string lastName, string last4SSN, string registryStatusIds,
                                                           string adminFlag, string adminFlagFromDate, string adminFlagToDate, bool exludeReviewHecAdminFlag,
                                                           string statusFromDate, string statusToDate, string sort, int startRow, int maxRows, int totalRecords, bool hasStateFilter, string stateFilter)
        {
            return _totalCount;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="lastName"></param>
        /// <param name="last4SSN"></param>
        /// <param name="dob"></param>
        /// <param name="registryStatusId"></param>
        /// <param name="registryFlagId"></param>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public int SearchRegistrantsCount(string lastName, string last4SSN, string dob, string registryStatusId,
                                          string registryFlagId, string registrantId, string sort, int startRow, int maxRows, int totalRecords)
        {
            return _totalCount;
        }


        private IQueryable<REGISTRANT> SearchRegistrantsLinqFilter(string lastName, string last4SSN, string dob,
                                                                   string registryStatusId, string registryFlagId, string registrantId)
        {
            IQueryable<REGISTRANT> linqFilter = null;
            if (string.IsNullOrEmpty(lastName) && string.IsNullOrEmpty(last4SSN) && string.IsNullOrEmpty(dob)
                && string.IsNullOrEmpty(registryStatusId) && string.IsNullOrEmpty(registryFlagId) && string.IsNullOrEmpty(registrantId))
            {
                linqFilter = from q in _dbAhopbr.REGISTRANTs
                             where q.REGISTRANT_ID == 0
                             select q;
            }
            else
            {
                linqFilter = from q in _dbAhopbr.REGISTRANTs
                             select q;

                if (!String.IsNullOrEmpty(last4SSN))
                {
                    linqFilter = linqFilter.Where(q => q.Snum.EndsWith(last4SSN.Trim()));
                }
                if (!string.IsNullOrEmpty(registrantId))
                {
                    linqFilter = linqFilter.Where(q => q.REGISTRANT_ID == Convert.ToInt32(registrantId));
                }
                if (!String.IsNullOrEmpty(lastName))
                {
                    linqFilter = linqFilter.Where(q => q.LAST_NAME.Contains(lastName.Trim()));
                }
                if (!String.IsNullOrEmpty(dob))
                {
                    try
                    {
                        DateTime dateOfBirth = Convert.ToDateTime(dob);
                        linqFilter = linqFilter.Where(q => q.BIRTH_DATE.Value.Date == dateOfBirth);
                    }
                    catch (Exception ex)
                    {
                        _sharedManager.LogErrorMessage("Invalid Date", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    }
                }
                if (!String.IsNullOrEmpty(registryStatusId))
                {
                    linqFilter = linqFilter.Where(q => q.STD_REGISTRANT_STATUS_ID == Convert.ToInt32(registryStatusId));
                }
                if (!String.IsNullOrEmpty(registryFlagId))
                {
                    linqFilter = linqFilter.Where(q => q.REGISTRANT_REGISTRY_FLAGs.Any(s => s.STD_REGISTRY_FLAG_ID == Convert.ToInt32(registryFlagId)));
                }
            }
            return linqFilter;
        }

        public IQueryable<REGISTRANT> CreateLinqFilter(bool hasStateFilter, string stateFilter)
        {
            IQueryable<REGISTRANT> linqFilter = null;
            if (hasStateFilter && stateFilter != null)
            {
                var statesPredicate = PredicateBuilder.False<REGISTRANT_ADDRESS>();
                string[] states = stateFilter.Split(',');
                foreach (var state in states)
                {
                    statesPredicate = statesPredicate.Or(s => s.STATE.ToLower() == state.ToLower());
                }
                var addresses = _dbAhopbr.REGISTRANT_ADDRESSes.Where(statesPredicate);
                linqFilter = _dbAhopbr.REGISTRANTs.Where(r => addresses.Any(a => r.REGISTRANT_ID == a.REGISTRANT_ID));
            }
            else
            {
                linqFilter = from q in _dbAhopbr.REGISTRANTs
                             select q;
            }
            return linqFilter;
        }

        private IQueryable<REGISTRANT> SearchRegistrantsChangeStatusLinqFilter(string lastName, string last4SSN, string registryStatusIds,
                                                                               string adminFlag, string adminFlagFromDate, string adminFlagToDate, bool exludeReviewHecAdminFlag,
                                                                               string statusFromDate, string statusToDate, bool hasStateFilter, string stateFilter)
        {
            //RTC 460132 - SQL has a limit to how many parameters can be used
            //several pieces of code below were pulling a list of ID's and then passing them as SQL parameters
            //that code has been replaced with a separate query on the REGISTRANT_REGISTRY_FLAG table

            bool blnJoinFlags = false;

            IQueryable<REGISTRANT_REGISTRY_FLAG> flags = from f in _dbAhopbr.REGISTRANT_REGISTRY_FLAGs
                                                         select f;

            IQueryable<REGISTRANT> registrants = CreateLinqFilter(hasStateFilter, stateFilter);

            if (string.IsNullOrEmpty(registryStatusIds) && String.IsNullOrEmpty(last4SSN)
                && String.IsNullOrEmpty(lastName) && string.IsNullOrEmpty(adminFlag))
            {
                registrants = registrants.Where(q => q.REGISTRANT_ID == 0);
            }
            if (!string.IsNullOrEmpty(registryStatusIds))
            {
                string[] statusID = registryStatusIds.Split(",");
                registrants = registrants.Where(q => statusID.Contains(q.STD_REGISTRANT_STATUS_ID.ToString()));

            }
            if (!String.IsNullOrEmpty(last4SSN))
            {
                registrants = registrants.Where(q => q.Snum.EndsWith(last4SSN.Trim()));
            }
            if (!String.IsNullOrEmpty(lastName))
            {
                registrants = registrants.Where(q => q.LAST_NAME.Contains(lastName.Trim()));
            }
            if (!String.IsNullOrEmpty(adminFlag))
            {
                blnJoinFlags = true;
                flags = flags.Where(f => f.STD_REGISTRY_FLAG_ID == Convert.ToInt32(adminFlag));
            }
            if (!String.IsNullOrEmpty(adminFlagFromDate))
            {
                blnJoinFlags = true;
                flags = flags.Where(f => f.CREATED >= Convert.ToDateTime(adminFlagFromDate));
            }
            if (!String.IsNullOrEmpty(adminFlagToDate))
            {               
                blnJoinFlags = true;
                flags = flags.Where(f => f.CREATED < Convert.ToDateTime(adminFlagToDate).AddDays(1));
            }

            if (exludeReviewHecAdminFlag)
            {
                blnJoinFlags = true;
                flags = flags.Where(f => f.STD_REGISTRY_FLAG_ID != (int)AHOBPRShared.PredefinedAdminFlag.ReviewHEC);
            }

            if (!string.IsNullOrEmpty(statusFromDate))
            {
                registrants = registrants.Where(q => q.REGISTRANT_STATUS_LAST_UPDATED_DATE >= Convert.ToDateTime(statusFromDate));
            }

            if (!string.IsNullOrEmpty(statusToDate))
            {
                registrants = registrants.Where(q => q.REGISTRANT_STATUS_LAST_UPDATED_DATE < Convert.ToDateTime(statusToDate).AddDays(1));
            }

            if (blnJoinFlags)
                registrants = registrants.Join(flags, q => q.REGISTRANT_ID, f => f.REGISTRANT_ID, (q, f) => q);

            return registrants;
        }
        
        private void SetLoadWithForSearchRegistrants(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.STD_REGISTRANT_STATUS);
            lo.LoadWith<REGISTRANT>(e => e.STD_SERVICE_STATUS);
            lo.LoadWith<REGISTRANT>(e => e.FORM_RESPONSEs);
            lo.LoadWith<FORM_RESPONSE>(e => e.FORM_RESPONSE_QUESTIONs);
            lo.LoadWith<FORM_RESPONSE>(e => e.FORM_RESPONSE_STATUS);
            lo.LoadWith<FORM_RESPONSE_QUESTION>(e => e.FORM_RESPONSE_ANSWERs);
            lo.LoadWith<FORM_RESPONSE_ANSWER>(e => e.STD_FORM_ANSWER);
            lo.LoadWith<STD_FORM_ANSWER>(e => e.STD_FORM_QUESTION);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_REGISTRY_FLAGs);
            lo.LoadWith<REGISTRANT_REGISTRY_FLAG>(e => e.STD_REGISTRY_FLAG);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }
        private void SetLoadWithForChangeStatus(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.STD_REGISTRANT_STATUS);
            lo.LoadWith<REGISTRANT>(e => e.STD_SERVICE_STATUS);
            lo.LoadWith<REGISTRANT>(e => e.FORM_RESPONSEs);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_REGISTRY_FLAGs);
            lo.LoadWith<REGISTRANT_REGISTRY_FLAG>(e => e.STD_REGISTRY_FLAG);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }
        #endregion

        #region Methods for SelectRegistrantById

        /// <summary>
        /// Select Registrant REGISTRANT_ID
        /// </summary>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        public REGISTRANT GetRegistrantById(int registrantId)
        {
            REGISTRANT registrant = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrant(_dbAhopbr);

                registrant = (from e in _dbAhopbr.REGISTRANTs 
                              where e.REGISTRANT_ID == registrantId
                              select e).FirstOrDefault();

            }
            return registrant;
        }

        /// <summary>
        /// Select Registrant by REGISTRANT_ID for name change purpose
        /// </summary>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        public REGISTRANT GetRegistrantByIdForNameChange(int registrantId)
        {
            REGISTRANT registrant = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrantNameChange(_dbAhopbr);

                registrant = (from e in _dbAhopbr.REGISTRANTs
                              where e.REGISTRANT_ID == registrantId
                              select e).FirstOrDefault();

            }
            return registrant;
        }

        private void SetLoadWithForRegistrantNameChange(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_RECORD_INFO);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_CONTACT_INFO);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }
        /// <summary>
        /// Select Registrant REGISTRANT_ID
        /// </summary>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        public REGISTRANT GetRegistrantByIdLight(int registrantId)
        {
            REGISTRANT registrant = null;
            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;
                registrant = (from e in _dbAhopbr.REGISTRANTs
                              where e.REGISTRANT_ID == registrantId
                              select e).FirstOrDefault();

            }
            return registrant;
        }

        private void SetLoadWithForRegistrantDeploymentsLight(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_DEPLOYMENTs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        /// <summary>
        /// Select Registrant REGISTRANT_ID
        /// </summary>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        public REGISTRANT GetRegistrantDeploymentsById(int registrantId)
        {
            REGISTRANT registrant = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrantDeploymentsLight(_dbAhopbr);
                registrant = (from e in _dbAhopbr.REGISTRANTs
                              where e.REGISTRANT_ID == registrantId
                              select e).FirstOrDefault();
            }
            return registrant;
        }


        /// <summary>
        /// Get Registrnat Status by ID
        /// </summary>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        public string GetRegistrantStatusById(int registrantId)
        {
            string status = string.Empty;

            if (registrantId > 0)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    SetLoadWithForRegistrantLight(_dbAhopbr);
                    status = (from e in _dbAhopbr.REGISTRANTs
                              where e.REGISTRANT_ID == registrantId
                              select e.STD_REGISTRANT_STATUS.REGISTRANT_STATUS).FirstOrDefault();
                }
            }

            return status;
        }

        /// <summary>
        /// Get Registry Flag Name By Id
        /// </summary>
        /// <param name="flagId"></param>
        /// <returns></returns>
        public string GetRegistryFlagNameById(int flagId)
        {
            string flagName = string.Empty;

            if (flagId > 0)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    flagName = (from e in _dbAhopbr.STD_REGISTRY_FLAGs
                              where e.STD_REGISTRY_FLAG_ID == flagId
                              select e.REGISTRY_FLAG_NAME).FirstOrDefault();
                }
            }

            return flagName;
        }

        /// <summary>
        /// Get Registry Flag Name By RegistrantFlagId
        /// </summary>
        /// <param name="registrantFlagId"></param>
        /// <returns></returns>
        public string GetRegistryFlagNameByRegistrantFlagId(int registrantFlagId)
        {
            string flagName = string.Empty;

            if (registrantFlagId > 0)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    flagName = (from e in _dbAhopbr.REGISTRANT_REGISTRY_FLAGs
                                where e.REGISTRANT_REGISTRY_FLAG_ID == registrantFlagId
                                select e.STD_REGISTRY_FLAG.REGISTRY_FLAG_NAME).FirstOrDefault();
                }
            }

            return flagName;
        }

        private void SetLoadWithForRegistrant(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.STD_REGISTRANT_STATUS);
            lo.LoadWith<REGISTRANT>(e => e.STD_SERVICE_STATUS);
            lo.LoadWith<REGISTRANT>(e => e.FORM_RESPONSEs);
            lo.LoadWith<FORM_RESPONSE>(e => e.FORM_RESPONSE_QUESTIONs);
            lo.LoadWith<FORM_RESPONSE>(e => e.FORM_RESPONSE_STATUS);
            lo.LoadWith<FORM_RESPONSE_QUESTION>(e => e.FORM_RESPONSE_ANSWERs);
            lo.LoadWith<FORM_RESPONSE_ANSWER>(e => e.STD_FORM_ANSWER);
            lo.LoadWith<STD_FORM_ANSWER>(e => e.STD_FORM_QUESTION);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_REGISTRY_FLAGs);
            lo.LoadWith<REGISTRANT_REGISTRY_FLAG>(e => e.STD_REGISTRY_FLAG);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_RECORD_INFO);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_CONTACT_INFO);
            lo.LoadWith<REGISTRANT_CONTACT_INFO>(e => e.REGISTRANT_ADDRESSes);
            lo.LoadWith<REGISTRANT_CONTACT_INFO>(e => e.REGISTRANT_PHONEs);
            lo.LoadWith<REGISTRANT_CONTACT_INFO>(e => e.REGISTRANT_EMAILs);
            lo.LoadWith<REGISTRANT>(e => e.STD_SERVICE_STATUS_WHEN_ENROLLED);
            lo.LoadWith<REGISTRANT>(e => e.STD_COMPONENT);
            lo.LoadWith<REGISTRANT>(e => e.STD_COMPONENT_WHEN_ENROLLED); 
            lo.LoadWith<REGISTRANT>(e => e.STD_DISCHARGE_CHARACTER);
            lo.LoadWith<REGISTRANT>(e => e.STD_SEPARATION_REASON);
            lo.LoadWith<REGISTRANT>(e => e.STD_SEPARATION_REASON_WHEN_ENROLLED);
            lo.LoadWith<REGISTRANT>(e => e.REGISTRANT_RETIREMENTs);
            lo.LoadWith<REGISTRANT_RETIREMENT>(e => e.STD_RETIREMENT_TYPE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        private void SetLoadWithForRegistrantLight(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT>(e => e.STD_REGISTRANT_STATUS);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }
        #endregion

        #region Methods for SelectRegistrantNameById

        public string SelectRegistrantNameById(int registrantId)
        {
            string registrant = string.Empty;

            using (_dbAhopbr = GetDataContext())
            {
                registrant = (from e in _dbAhopbr.REGISTRANTs
                              where e.REGISTRANT_ID == registrantId
                              select e.LAST_NAME + ", " + e.FIRST_NAME + " " + e.MIDDLE_NAME).FirstOrDefault();

            }
            return registrant;
        }

        #endregion 

        #region Select all EDIPI mismatched
        public IEnumerable<REGISTRANT> SearchRegistrantsForNameUpdate(string lastName, string last4SSN, string dob, 
                                                                        string sort, int startRow, int maxRows, int totalRecords)
        {
            if (String.IsNullOrEmpty(sort))
                sort = "LAST_NAME, FIRST_NAME";

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForChangeStatus(_dbAhopbr);
                IQueryable<REGISTRANT> entities = SearchRegistrantsNameUpdateLinqFilter(lastName, last4SSN, dob);

                _totalCount = entities.Count();
                //output parmeter used to populate the total number of records label on the Changes Status search page
                totalRecords = _totalCount;

                return (entities).SortAndPage(sort, maxRows, startRow).ToList<REGISTRANT>();
            }
        }
        public int SearchRegistrantsForNameUpdateCount(string lastName, string last4SSN, string dob,
                                                          string sort, int startRow, int maxRows, int totalRecords)
        {
            return _totalCount;
        }

       private IQueryable<REGISTRANT> SearchRegistrantsNameUpdateLinqFilter(string lastName, string last4SSN, string dob)
        {
            IQueryable<REGISTRANT> linqFilter = from q in _dbAhopbr.REGISTRANTs
                                                where q.REGISTRANT_RECORD_INFO.EDIPI_MISMATCHED_FLAG == true
                                                select q; 

            if (!String.IsNullOrEmpty(last4SSN))
            {
                linqFilter = linqFilter.Where(q => q.Snum.EndsWith(last4SSN.Trim()));
            }
            if (!String.IsNullOrEmpty(lastName))
            {
                linqFilter = linqFilter.Where(q => q.LAST_NAME.Contains(lastName.Trim()));
            }
            if (!String.IsNullOrEmpty(dob))
            {
                try
                {
                    DateTime dateOfBirth = Convert.ToDateTime(dob);
                    linqFilter = linqFilter.Where(q => q.BIRTH_DATE.Value.Date == dateOfBirth);
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Invalid Date", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                }
            }

            return linqFilter;
        }


        #endregion

        #region Methods for SelectDeploymentHistory

        public List<REGISTRANT_DEPLOYMENT> SelectRegistrantDeploymentHistory(string registrantId, string sort, int startRow, int maxRows)
        {
            const string gridDefaultSort = @"DEPLOYMENT_START_DATE";

            if (string.IsNullOrEmpty(sort))
            {
                sort = string.Format("{0}", gridDefaultSort);
            }

            //List<REGISTRANT_DEPLOYMENT> registrantDeployments = new List<REGISTRANT_DEPLOYMENT>();
            IQueryable<REGISTRANT_DEPLOYMENT> registrantDeployments;
            int id = string.IsNullOrEmpty(registrantId) ? 0 : Convert.ToInt32(registrantId);

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrantDeployments(_dbAhopbr);

                registrantDeployments = (from e in _dbAhopbr.REGISTRANT_DEPLOYMENTs
                                   where e.REGISTRANT_ID == id
                                   orderby e.DEPLOYMENT_START_DATE 
                                   select e);

                //Used for sorting  
            _totalCount = registrantDeployments.Count();

            return (registrantDeployments).SortAndPage(sort, maxRows, startRow).ToList<REGISTRANT_DEPLOYMENT>();
            }
        }

        public int DeploymentCount(string registrantId, string sort, int startRow, int maxRows)
        {
            return _totalCount;
        }

        public List<REGISTRANT_DEPLOYMENT> SelectRegistrantEligibleDeployments(string registrantId, string sort, int startRow, int maxRows)
        {
            const string gridDefaultSort = @"DEPLOYMENT_START_DATE";

            if (string.IsNullOrEmpty(sort))
            {
                sort = string.Format("{0}", gridDefaultSort);
            }

            //List<REGISTRANT_DEPLOYMENT> registrantDeployments = new List<REGISTRANT_DEPLOYMENT>();
            IQueryable<REGISTRANT_DEPLOYMENT> registrantDeployments;
            int id = string.IsNullOrEmpty(registrantId) ? 0 : Convert.ToInt32(registrantId);

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrantDeployments(_dbAhopbr);

                registrantDeployments = (from e in _dbAhopbr.REGISTRANT_DEPLOYMENTs
                                         where (e.REGISTRANT_ID == id  && e.IS_ELIGIBLE_FLAG == true)
                                         orderby e.DEPLOYMENT_START_DATE
                                         select e);

                //Used for sorting  SelectRegistrantEligibleDeployments
                _totalCount = registrantDeployments.Count();

                return (registrantDeployments).SortAndPage(sort, maxRows, startRow).ToList<REGISTRANT_DEPLOYMENT>();
            }
        }

        public int EligibleDeploymentCount(string registrantId, string sort, int startRow, int maxRows)
        {
            return _totalCount;
        }

        //Use for the Clinically Relevant Summary page for the registrants distinct deployments
        public List<REGISTRANT_DEPLOYMENT> SelectDistinctRegistrantDeploymentHistory(string registrantId)
        {
            List<REGISTRANT_DEPLOYMENT> registrantDeployments = new List<REGISTRANT_DEPLOYMENT>();
            int id = string.IsNullOrEmpty(registrantId) ? 0 : Convert.ToInt32(registrantId);

            using (_dbAhopbr = GetDataContext())
            {
               //SetLoadWithForRegistrantDeployments(_dbAhopbr);

                List<int?> deploymentIds = (from d in _dbAhopbr.FORM_RESPONSE_QUESTIONs
                                           where d.FORM_RESPONSE.REGISTRANT_ID == id
                                           select d.REGISTRANT_DEPLOYMENT_ID).Distinct().ToList();

                registrantDeployments = (from e in _dbAhopbr.REGISTRANT_DEPLOYMENTs
                                         where deploymentIds.Contains(e.REGISTRANT_DEPLOYMENT_ID)
                                         select e).ToList();
            }
            return registrantDeployments;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="formId"></param>
        /// <returns></returns>
        public string GetFollowupFormDescription(int registrantId, int formId)
        {
            string description = string.Empty;

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForFollowupForm(_dbAhopbr);

                FOLLOWUP_FORM_RESPONSE response = (from e in _dbAhopbr.FOLLOWUP_FORM_RESPONSEs
                               where e.REGISTRANT_ID == registrantId 
                                    && e.STD_FORM_ID == formId
                               select e).FirstOrDefault();

                if (response != null)
                {
                    string status = response.COMPLETED_DATE.HasValue ?
                        " (submitted  " + response.COMPLETED_DATE.ToString() + ")" : " (new)";
                    description = response.STD_FORM.NAME + status;
                }
            }

            return description;
        }
        
        public string GetDeploymentDescriptionById(int deploymentId)
        {
            string description = string.Empty;

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrantDeployments(_dbAhopbr);

                REGISTRANT_DEPLOYMENT deployment = (from e in _dbAhopbr.REGISTRANT_DEPLOYMENTs
                                                    where e.REGISTRANT_DEPLOYMENT_ID == deploymentId
                                                    select e).FirstOrDefault();

                if (deployment != null)
                {
                    description = deployment.DeploymentDescription;
                }
            }

            return description;
        }

        public List<string> GetDeploymentDescriptionByRegistrantId(string registrantId)
        {
            List<REGISTRANT_DEPLOYMENT> registrantDeployments = new List<REGISTRANT_DEPLOYMENT>();
            List<string> deploymentsList = new List<string>();
            int id = string.IsNullOrEmpty(registrantId) ? 0 : Convert.ToInt32(registrantId);

            using (_dbAhopbr = GetDataContext())
            {
                registrantDeployments = (from e in _dbAhopbr.REGISTRANT_DEPLOYMENTs
                                         where e.REGISTRANT_ID == id
                                         select e).ToList();

                 foreach (REGISTRANT_DEPLOYMENT d in registrantDeployments)
                      {
                          deploymentsList.Add(d.DeploymentDescriptionForProgressNote);
                      }

           
            }

            return deploymentsList;
        }

        public string GetDeploymentDescriptionByIdForLocSpecfic(int deploymentId)
        {
            string description = string.Empty;

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrantDeployments(_dbAhopbr);

                REGISTRANT_DEPLOYMENT deployment = (from e in _dbAhopbr.REGISTRANT_DEPLOYMENTs
                                                    where e.REGISTRANT_DEPLOYMENT_ID == deploymentId
                                                    select e).FirstOrDefault();

                if (deployment != null)
                {
                    description = deployment.DeploymentDescriptionLocationSpecfic;
                }
            }

            return description;
        }

        public REGISTRANT_VISIT GetMostRecentEncounter(int registrantId)
        {
            IQueryable<REGISTRANT_VISIT> result;
            REGISTRANT_VISIT registrantVisit = new REGISTRANT_VISIT();
            using (_dbAhopbr = GetDataContext())
            {


                result = (from r in _dbAhopbr.REGISTRANT_VISITs
                                   where r.EVALUATION_FLAG == 'N' && r.REGISTRANT_ID == registrantId
                                   group r by r.REGISTRANT_ID into g
                                   select g.OrderByDescending(t => t.VISIT_DATE_TIME).FirstOrDefault());

                //set iQueryable back to object
                registrantVisit = result.Select(r => r).FirstOrDefault();

            }
                        
            return registrantVisit;
        }

        public List<EvaluationGridview> GetAHOBPRExamination(int registrantId)
        {
            IQueryable<REGISTRANT_VISIT> result;
            REGISTRANT_VISIT registrantExam = new REGISTRANT_VISIT();
            EvaluationGridview evaluationdata = new EvaluationGridview();
            List<EvaluationGridview> evaluationGridview = new List<EvaluationGridview>();
            string location;
            using (_dbAhopbr = GetDataContext())
            {
                result = (from r in _dbAhopbr.REGISTRANT_VISITs
                                   where r.EVALUATION_FLAG == 'Y' && r.REGISTRANT_ID == registrantId
                                   group r by r.REGISTRANT_ID into g
                                   select g.OrderByDescending(t => t.VISIT_DATE_TIME).FirstOrDefault());

                registrantExam = result.Select(r => r).FirstOrDefault();

                if (registrantExam != null)
                {

                    int institutionId = (int?)registrantExam.STD_INSTITUTION_ID ?? 0;
                    location = _sharedManager.GetFacilityDescription(institutionId);

                    evaluationdata =
                                       new EvaluationGridview
                                       {
                                           RegistrantId = registrantExam.REGISTRANT_ID,
                                           //EvaluationDate = registrantExam.VISIT_DATE_TIME.HasValue ? registrantExam.VISIT_DATE_TIME.Value.Month + "/" + registrantExam.VISIT_DATE_TIME.Value.Day + "/" + registrantExam.VISIT_DATE_TIME.Value.Year : string.Empty,
                                           EvaluationDate = registrantExam.VISIT_DATE_TIME.HasValue ? Convert.ToDateTime(registrantExam.VISIT_DATE_TIME).ToString("MM/dd/yyyy") : string.Empty,
                                           Location = location,
                                           StopCode = registrantExam.STOP_CODE,
                                           EvaluationUpdated = "AHOBPR Evaluation (Updated " + (Convert.ToDateTime(registrantExam.UPDATED).ToString("MM/dd/yyyy") ?? string.Empty) + ")"
                                           //EvaluationUpdated = "AHOBPR Evaluation (Updated " + (registrantExam.UPDATED.ToShortDateString() ?? string.Empty) + ")" 

                                       };

                    //convert to list for gridview to check the row count
                    evaluationGridview.Add(evaluationdata);
                    

                }

            }

            return evaluationGridview;
        }

        private void SetLoadWithForFollowupForm(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<FOLLOWUP_FORM_RESPONSE>(e => e.STD_FORM);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        private void SetLoadWithForRegistrantDeployments(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT_DEPLOYMENT>(e => e.STD_BRANCH_OF_SERVICE);
            lo.LoadWith<REGISTRANT_DEPLOYMENT>(e => e.STD_PAYPLAN);
            lo.LoadWith<REGISTRANT_DEPLOYMENT>(e => e.FORM_RESPONSE_QUESTIONs);
            lo.LoadWith<REGISTRANT_DEPLOYMENT>(e => e.REGISTRANT_SERVICE_SPECIFIC_OCCUPATIONs);
            lo.LoadWith<REGISTRANT_SERVICE_SPECIFIC_OCCUPATION>(e => e.STD_SERVICE_OCCUPATION);
            lo.LoadWith<REGISTRANT_DEPLOYMENT>(e => e.STD_COMPONENT);
            lo.LoadWith<REGISTRANT_DEPLOYMENT>(e => e.STD_DEPLOYMENT_STAGE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        #endregion

        #region Methods for SelectHealthFactor

        public REGISTRANT_VISIT SelectMostRecentRegistrantHealthEvaluation(int registrantId)
        {
            REGISTRANT_VISIT evaluation = null;

            using (_dbAhopbr = GetDataContext())
            {
                evaluation = (from e in _dbAhopbr.REGISTRANT_VISITs
                              where e.REGISTRANT_ID == registrantId && e.EVALUATION_FLAG == 'Y'
                              orderby e.VISIT_DATE_TIME descending
                              select e).FirstOrDefault();

            }

            return evaluation;
        }

        public REGISTRANT_VISIT SelectRegistrantHealthEvaluation(int registrantEvaluationId)
        {
            REGISTRANT_VISIT evaluation = null;

            //get registrant_Visit_Id

            using (_dbAhopbr = GetDataContext())
            {
                evaluation = (from e in _dbAhopbr.REGISTRANT_VISITs
                              where e.REGISTRANT_ID == registrantEvaluationId && e.EVALUATION_FLAG == 'Y'
                              orderby e.VISIT_DATE_TIME descending
                              select e).FirstOrDefault();

            }

            return evaluation;
        }

        public int GetRegistrantVisitId(int registrantEvaluationId)
        {
            int registrantVisitId = 0;

            using (_dbAhopbr = GetDataContext())
            {
                registrantVisitId = (from e in _dbAhopbr.REGISTRANT_VISITs
                              where e.REGISTRANT_ID == registrantEvaluationId && e.EVALUATION_FLAG == 'Y'
                              orderby e.VISIT_DATE_TIME descending
                              select e.REGISTRANT_VISIT_ID).FirstOrDefault();

            }

            return registrantVisitId;
        }

        public List<REGISTRANT_VISIT> SelectPreviousRegistrantHealthEvaluation(string registrantId)
        {
            int id = string.IsNullOrEmpty(registrantId) ? 0 : Convert.ToInt32(registrantId);
            REGISTRANT_VISIT mostRecentEvaluation = SelectMostRecentRegistrantHealthEvaluation(id);
            List<REGISTRANT_VISIT> previousEvaluations = new List<REGISTRANT_VISIT>();

            if (mostRecentEvaluation != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    previousEvaluations = (from e in _dbAhopbr.REGISTRANT_VISITs
                                  where e.REGISTRANT_ID != mostRecentEvaluation.REGISTRANT_ID
                                        && e.REGISTRANT_ID == id && e.EVALUATION_FLAG == 'Y'
                                  orderby e.VISIT_DATE_TIME descending
                                  select e).ToList();

                }
            }

            return previousEvaluations;
        }


        public List<VIEW_HEALTH_FACTOR> SelectRegistrantHealthFactor(string registrantEvaluationId, string healthFactorCategoryId)
        {
            List<VIEW_HEALTH_FACTOR> registrantHealthFactor = new List<VIEW_HEALTH_FACTOR>();
            int id = string.IsNullOrEmpty(registrantEvaluationId) ? 0 : Convert.ToInt32(registrantEvaluationId);
            int categoryId = string.IsNullOrEmpty(healthFactorCategoryId) ? 0 : Convert.ToInt32(healthFactorCategoryId);

            //get registrant_Visit_Id
            int registrantVisitId = GetRegistrantVisitId(id);
            
            using (_dbAhopbr = GetDataContext())
            {
                registrantHealthFactor = (from e in _dbAhopbr.VIEW_HEALTH_FACTORs
                                          where e.REGISTRANT_VISIT_ID == registrantVisitId
                                            && e.STD_HEALTH_FACTOR_CATEGORY_ID == categoryId
                                         orderby e.SORT_ORDER
                                         select e).ToList();

            }
            return registrantHealthFactor;
        }


        #endregion
        
        #region Methods for SelectProcessMetrics

        public List<PROCESS_METRIC> SelectProcessMetrics(string registrantId, string metricType)
        {
            List<PROCESS_METRIC> processMetrics = new List<PROCESS_METRIC>();
            int id = string.IsNullOrEmpty(registrantId) ? 0 : Convert.ToInt32(registrantId);

            using (_dbAhopbr = GetDataContext())
            {
                processMetrics = (from e in _dbAhopbr.PROCESS_METRICs
                                   where e.REGISTRANT_ID == id
                                    && e.STD_PROCESS_METRICS_TYPE.PROCESS_METRICS == metricType
                                   orderby e.UPDATED
                                   select e).ToList();

            }
            return processMetrics;
        }

        #endregion


        #region Methods for SelectRegistrantFlags

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        public List<REGISTRANT_REGISTRY_FLAG> SelectRegistrantSubpopulationFlags(string registrantId)
        {
            List<REGISTRANT_REGISTRY_FLAG> registrantFlags = new List<REGISTRANT_REGISTRY_FLAG>();
            int id = string.IsNullOrEmpty(registrantId)? 0 : Convert.ToInt32(registrantId);

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrantFlags(_dbAhopbr);

                registrantFlags = (from e in _dbAhopbr.REGISTRANT_REGISTRY_FLAGs
                                   where e.REGISTRANT_ID == id
                                    && e.STD_REGISTRY_FLAG.STD_REGISTRY_FLAG_TYPE_ID == 2
                                   orderby e.CREATED
                                   select e).ToList();

            }
            return registrantFlags;
        }

        public List<REGISTRANT_REGISTRY_FLAG> SelectRegistrantAdminTrackingFlags(string registrantId)
        {
            List<REGISTRANT_REGISTRY_FLAG> registrantFlags = new List<REGISTRANT_REGISTRY_FLAG>();
            int id = string.IsNullOrEmpty(registrantId) ? 0 : Convert.ToInt32(registrantId);

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistrantFlags(_dbAhopbr);

                registrantFlags = (from e in _dbAhopbr.REGISTRANT_REGISTRY_FLAGs
                                   where e.REGISTRANT_ID == id
                                    && e.STD_REGISTRY_FLAG.STD_REGISTRY_FLAG_TYPE_ID == 1
                                   orderby e.CREATED
                                   select e).ToList();

            }
            return registrantFlags;
        }

        private void SetLoadWithForRegistrantFlags(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<REGISTRANT_REGISTRY_FLAG>(e => e.REGISTRANT);
            lo.LoadWith<REGISTRANT_REGISTRY_FLAG>(e => e.STD_REGISTRY_FLAG);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        #endregion


        #region Methods for AddRegistryFlagToRegistrant

        public int AddSupopulationFlagToRegistrant(int registrantId, int registryFlagId)
        {
            int returnStatus = 0;

            if (registryFlagId == 0 || registrantId == 0)
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                //check if subpopulation flag exists
                REGISTRANT_REGISTRY_FLAG registrantflag = (from e in _dbAhopbr.REGISTRANT_REGISTRY_FLAGs
                                                 where e.REGISTRANT_ID == registrantId
                                                    && e.STD_REGISTRY_FLAG_ID == registryFlagId
                                                 select e).FirstOrDefault();

                //insert if not exists
                if (registrantflag == null)
                {
                    registrantflag = new REGISTRANT_REGISTRY_FLAG();

                    registrantflag.REGISTRANT_ID = registrantId;
                    registrantflag.STD_REGISTRY_FLAG_ID = registryFlagId;

                    _dbAhopbr.REGISTRANT_REGISTRY_FLAGs.InsertOnSubmit(registrantflag);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus = 1;
                    }
                    catch (System.Data.SqlClient.SqlException ex)
                    {
                        _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                    catch (ChangeConflictException e)
                    {
                        _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                    catch
                    {
                        returnStatus = 0;
                    }

                }
                // if the registry flag already exists, set returnStatus to 2
                else
                {
                    returnStatus = 2;
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="registryFlagName"></param>
        /// <returns></returns>
        public int AddAdminTrackingFlagToRegistrantByName(int registrantId, string registryFlagName)
        {
            int result = 0;

            AHOBPRAdminManager adminManager = new AHOBPRAdminManager();
            int registryFlagId = adminManager.GetRegistryFlagIdByName(registryFlagName);
            if (registryFlagId == 0)
            {
                adminManager.AddRegistryFlag((int)AhobprRegistryFlagType.AdminFlag, registryFlagName, registryFlagName);
                registryFlagId = adminManager.GetRegistryFlagIdByName(registryFlagName);      
            }
            if (registryFlagId > 0)
            {
                result = AddAdminTrackingFlagToRegistrant(registrantId, registryFlagId); 
            }

            return result;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="registryFlagName"></param>
        /// <returns></returns>
        public int AddAdminTrackingFlagToRegistrantForSentMessage(int registrantId, string registryFlagName)
        {
            int result = 0;
            
            AHOBPRAdminManager adminManager = new AHOBPRAdminManager();
            int registryFlagId = adminManager.GetRegistryFlagIdByName(registryFlagName);
            if (registryFlagId == 0)
            {
                adminManager.AddRegistryFlagForSentMessage((int)AhobprRegistryFlagType.AdminFlag, registryFlagName, registryFlagName);
                registryFlagId = adminManager.GetRegistryFlagIdByName(registryFlagName);
            }
            if (registryFlagId > 0)
            {
                result = AddAdminTrackingFlagToRegistrant(registrantId, registryFlagId);
            }

            return result;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="registryFlagId"></param>
        /// <returns></returns>
        public int AddAdminTrackingFlagToRegistrant(int registrantId, int registryFlagId)
        {
            int returnStatus = 0;

            if (registryFlagId == 0 || registrantId == 0)
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                REGISTRANT_REGISTRY_FLAG registrantflag = new REGISTRANT_REGISTRY_FLAG();

                registrantflag.REGISTRANT_ID = registrantId;
                registrantflag.STD_REGISTRY_FLAG_ID = registryFlagId;

                _dbAhopbr.REGISTRANT_REGISTRY_FLAGs.InsertOnSubmit(registrantflag);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                    returnStatus = 1;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = -1;
                }
                catch (ChangeConflictException e)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                        returnStatus = 1;
                    }
                    catch (Exception ex)
                    {
                        _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                }
                catch
                {
                    returnStatus = 0;
                }
            }
            return returnStatus;
        }

        #endregion


        #region method for RemoveRegistryFlagFromRegistrant

        public bool RemoveRegistryFlagFromRegistrant(int registrantFlagId)
        {
            bool returnStatus = false;
            using (_dbAhopbr = GetDataContext())
            {
                REGISTRANT_REGISTRY_FLAG registrantFlag = (from e in _dbAhopbr.REGISTRANT_REGISTRY_FLAGs
                                                  where e.REGISTRANT_REGISTRY_FLAG_ID == registrantFlagId
                                                  select e).FirstOrDefault();

                _dbAhopbr.REGISTRANT_REGISTRY_FLAGs.DeleteOnSubmit(registrantFlag);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = true;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
                catch (ChangeConflictException)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    returnStatus = false;
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
            }

            return returnStatus;
        }

        #endregion

        #region Update Registrant Status
        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrant"></param>
        /// <param name="oldStatus"></param>
        /// <returns></returns>
        public int UpdateRegistrantStatus(REGISTRANT registrant, int oldStatusId)
        {
            int returnStatus = UpdateRegistrant(registrant);

            int newStatusId = (int) registrant.STD_REGISTRANT_STATUS_ID;

            // Add a new record to PROCESS_METRIC table if the newStatusId = oldStatusId becasue the trigger on Registrant table cannot handle this
            if (newStatusId == oldStatusId)
            {
                returnStatus = AddProcessMetricForStatusChange(registrant.REGISTRANT_ID, newStatusId);
            }

            returnStatus = AddRegistryFlagForStatusChange(registrant.REGISTRANT_ID, GetAdminFlag(oldStatusId, newStatusId));
            
            return returnStatus;
        }

        /// <summary>
        /// Get the admin flag ID by old status ID and new status ID
        /// </summary>
        /// <param name="oldStatusId"></param>
        /// <param name="newStatusId"></param>
        /// <returns></returns>
        public int GetAdminFlag(int oldStatusId, int newStatusId)
        {
            int flagId = 0;

            if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.Review && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ReviewToReviewEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.Review && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ReviewToReviewNotEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.NotEligible && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.NotEligibleToReviewEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.NotEligible && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.NotEligibleToReviewNotEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.Participant && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ParticipantToReviewEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.Participant && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ParticipantToReviewNotEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.NoConcent && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.NoConsentToReviewEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.NoConcent && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.NoConsentToReviewNotEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.Concent && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ConsentToReviewEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.Concent && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ConsentToReviewNotEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.Eligible && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.EligibleToReviewEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.Eligible && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.EligibleToReviewNotEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ReviewNotEligibleToReviewEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ReviewNotEligibleToReviewNotEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ReviewEligibleToReviewEligible;
            }
            else if (oldStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible && newStatusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
            {
                flagId = (int)AHOBPRShared.PredefinedAdminFlag.ReviewEligibleToReviewNotEligible;
            }

            return flagId;
        }

        /// <summary>
        /// GetAdminFlagNameForStatusChange
        /// </summary>
        /// <param name="oldStatusId"></param>
        /// <param name="newStatusId"></param>
        /// <returns></returns>
        public string GetAdminFlagNameForStatusChange(int oldStatusId, int newStatusId)
        {
            string flagName = string.Empty;
            int flagId = GetAdminFlag(oldStatusId, newStatusId);
            flagName = GetRegistryFlagNameById(flagId);
            
            return flagName;
        }

        /// <summary>
        /// Get status name by statusId
        /// </summary>
        /// <param name="statusId"></param>
        /// <returns></returns>
        private string GetStatus(int statusId)
        {
            string status = string.Empty;

            switch (statusId)
            {
                case (int) AHOBPRShared.RegistrantStatus.NoConcent:
                    status = "No Consent";
                    break;
                case (int)AHOBPRShared.RegistrantStatus.Concent:
                    status = "Consent";
                    break;
                case (int)AHOBPRShared.RegistrantStatus.NotEligible:
                    status = "Not Eligible";
                    break;
                case (int)AHOBPRShared.RegistrantStatus.Review:
                    status = "Review";
                    break;
                case (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible:
                    status = "Reviewed - Not Eligible";
                    break;
                case (int)AHOBPRShared.RegistrantStatus.Eligible:
                    status = "Eligible";
                    break;
                case (int)AHOBPRShared.RegistrantStatus.Participant:
                    status = "Participant";
                    break;
            }

            return status;
        }

        private int AddRegistryFlagForStatusChange(int registrantId, int flagId)
        {
            int returnStatus = 0;

            if (flagId == 0 && registrantId == 0)
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                REGISTRANT_REGISTRY_FLAG registrantWithFlag = new REGISTRANT_REGISTRY_FLAG();
                registrantWithFlag.REGISTRANT_ID = registrantId;
                registrantWithFlag.STD_REGISTRY_FLAG_ID = flagId;

                _dbAhopbr.REGISTRANT_REGISTRY_FLAGs.InsertOnSubmit(registrantWithFlag);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                    returnStatus = 1;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = -1;
                }
                catch (ChangeConflictException e)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                        returnStatus = 1;
                    }
                    catch (Exception ex)
                    {
                        _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                }
                catch
                {
                    returnStatus = 0;
                }

            }
            return returnStatus;

        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="statusId"></param>
        /// <returns></returns>
        private int AddProcessMetricForStatusChange(int registrantId, int statusId)
        {
            int returnStatus = 0;

            if (statusId == 0)
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                PROCESS_METRIC processMetric = new PROCESS_METRIC();

                processMetric.REGISTRANT_ID = registrantId;
                processMetric.PROCESS_METRICS_VALUE = GetStatus(statusId);
                processMetric.STD_PROCESS_METRICS_TYPE_ID  = 1;

                _dbAhopbr.PROCESS_METRICs.InsertOnSubmit(processMetric);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                    returnStatus = 1;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = -1;
                }
                catch (ChangeConflictException e)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                        returnStatus = 1;
                    }
                    catch (Exception ex)
                    {
                        _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                }
                catch
                {
                    returnStatus = 0;
                }

            }
            return returnStatus;

        }

        public int UpdateRegistrant(REGISTRANT registrant)
        {
            int returnStatus = 0;

            if (registrant != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    registrant.SynchroniseWithDataContext(_dbAhopbr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                    if (changeSet.Updates.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                            returnStatus = 1;

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "UserId: " + registrant.USER_ID + " Error: " + ex.Message);
                            returnStatus = -1;
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "UserId: " + registrant.USER_ID + " Error: " + ex.Message);
                            returnStatus = -1;
                        }
                    }
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// Update registrant deployment stage if registrant status is changed to "Reviewed Not Eligible" or "Review Eligible"
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="statusId"></param>
        /// <returns></returns>
        public int UpdateRegistrantDeploymentStage(int registrantId, int statusId)
        {
            int returnStatus = 0;
            bool update = false;

            REGISTRANT registrantWithDeployments = GetRegistrantDeploymentsById(registrantId);

            if (registrantId > 0 && statusId > 0 && registrantWithDeployments != null && registrantWithDeployments.REGISTRANT_DEPLOYMENTs.Count > 0)
            {
                registrantWithDeployments.SetAsChangeTrackingRoot();
                foreach (REGISTRANT_DEPLOYMENT deployment in registrantWithDeployments.REGISTRANT_DEPLOYMENTs)
                {
                    if (deployment.USER_ENTERED_FLAG == true)
                    {
                        if (statusId == (int)AHOBPRShared.RegistrantStatus.ReviewEligible)
                        {
                            deployment.STD_DEPLOYMENT_STAGE_ID = (int)AHOBPRShared.DeploymentStage.Accepted;
                        }
                        else if (statusId == (int)AHOBPRShared.RegistrantStatus.ReviewedNotEligible)
                        {
                            deployment.STD_DEPLOYMENT_STAGE_ID = (int)AHOBPRShared.DeploymentStage.Rejected;
                        }
                        deployment.SetAsUpdateOnSubmit();
                        update = true;
                    }
                }
            }

            if (update)
            {
                returnStatus = UpdateRegistrant(registrantWithDeployments);
            }
            return returnStatus;
        }


        public static String RegistrantPortalRegistrantStatusFromRegistrantStatusId(int id)
        {
            switch (id)
            {
                case 1: return "NOT_CONSENT";
                case 2: return "CONSENT";
                case 3: return "NOT_ELIGIBLE";
                case 4: return "REVIEW";
                case 5: return "REVIEWED_NOT_ELIGIBLE";
                case 6: return "ELIGIBLE";
                case 7: return "PARTICIPANT";
                case 8: return "REVIEWED_ELIGIBLE";
            }
            return "";
        }

        #endregion
    }
}
