﻿using System;
using System.Collections.Generic;
using System.Data.Linq;
using System.Linq;
using System.Text;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessAHOBPR;
using System.Reflection;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    public enum AuditLogType
    {
        UserLogin = 1,
        RegistrantOpened = 2,
        DataMaintenance = 3,
        MassFlagSet = 4,
        FollowupGroupSent = 5,
        AdminFlagSet = 6,
        SubpopulationFlagSet = 7,
        MessageTemplateAddedEdited = 8,
        RegistryFlagAddedEdited = 9,
        ChangeStatus = 10,
        AccessFromCPRS = 11,
        ProgressNoteSent = 12,
        RegistrantNameVerified = 13,
        NewFollowupQuestionGroupSubmitted = 14,
        MessageSent = 15,
        EmailSentToUsers = 16,
        UserEmailTemplateAddedEdited = 17
    }

    public class AHOBPRAuditManager : AHOBPRBaseBO
    {
        private AHOBPRShared _sharedManager = new AHOBPRShared();
        private int _totalCount = 0;

        /// <summary>
        /// Add to AUDIT_LOG
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="message"></param>
        /// <param name="registrantId"></param>
        /// <param name="type"></param>
        /// <param name="numberOfRegistrants"></param>
        public void AddToLog(string userId, string message, string registrantId, AuditLogType type, int? numberOfRegistrants)
        {
            using (_dbAhopbr = GetDataContext())
            {
                AUDIT_LOG log = new AUDIT_LOG()
                {
                    CREATED = DateTime.Now,
                    CREATEDBY = userId,
                    MESSAGE = message,
                    REGISTRANT_ID = String.IsNullOrEmpty(registrantId) ? null : (int?)int.Parse(registrantId),
                    TYPE_ID = (byte)(int)type,
                    UPDATED = DateTime.Now,
                    UPDATEDBY = userId,
                    USER_ID = String.IsNullOrEmpty(userId) ? null : (int?)int.Parse(userId),
                    NUMBER_OF_REGISTRANTS = numberOfRegistrants
                };
                log.SetAsChangeTrackingRoot();
                log.SetAsInsertOnSubmit();

                UpdateLog(log);
            }
        }
    
        /// <summary>
        /// Update AUDIT_LOG
        /// </summary>
        /// <param name="log"></param>
        /// <returns></returns>
        public int UpdateLog(AUDIT_LOG log)
        {
            int returnStatus = 0;

            if (log != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    log.SynchroniseWithDataContext(_dbAhopbr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                    if (changeSet.Updates.Count > 0 ||
                        changeSet.Inserts.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                            returnStatus = 1;

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                }
            }
            return returnStatus;
        }


        #region User Action Tracking

        public IList<SP_UserListForActionTrackingResult> SearchUsersForActionTracking(string lastName, string firstName, string userRole, string sort)
        {
            if (String.IsNullOrEmpty(sort))
                sort = "FULL_NAME";
            List<SP_UserListForActionTrackingResult> results = null;
            IQueryable<SP_UserListForActionTrackingResult> mainQueryGridView;

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.CommandTimeout = AHOBPRGlobal.TimeOutForReports;

                //call stored procedure
                results = _dbAhopbr.SP_UserListForActionTracking(lastName, firstName, userRole).ToList();
                mainQueryGridView = results.AsQueryable();
                _totalCount = results.Count();
            }

            return (mainQueryGridView).Sort(sort).ToList<SP_UserListForActionTrackingResult>(); ;

        }
        public int SearchUsersForActionTrackingCount(string lastName, string firstName, string userRole)
        {
            return _totalCount;
        }

        /// <summary>
        /// Get all audit log for a user
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="sort"></param>
        /// <returns></returns>
        public IList<SP_UserAuditLogResult> SearchAuditLog(string userId, string lastName, string firstName, string userRole, string actionType,
                                                                    string fromDate, string toDate, string sort)
        {
            if (string.IsNullOrEmpty(userId) && string.IsNullOrEmpty(lastName) && string.IsNullOrEmpty(firstName) && string.IsNullOrEmpty(userRole)
                 && string.IsNullOrEmpty(actionType) && string.IsNullOrEmpty(fromDate) && string.IsNullOrEmpty(toDate))
            {
                return null;
            }
            else
            {
                const string defaultSort = @"CREATED DESC";

                if (string.IsNullOrEmpty(sort))
                {
                    sort = string.Format("{0}", defaultSort);
                }

                IQueryable<SP_UserAuditLogResult> mainQueryGridView;

                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.CommandTimeout = AHOBPRGlobal.TimeOutForReports;
                    //call stored procedure
                    var results = _dbAhopbr.SP_UserAuditLog(string.IsNullOrEmpty(userId) ? 0 : Convert.ToInt32(userId), 
                        actionType, fromDate, toDate, lastName, firstName, userRole).ToList();

                    mainQueryGridView = results.AsQueryable();
                    //sort
                    _totalCount = mainQueryGridView.Count();
                }

                return (mainQueryGridView).Sort(sort).ToList<SP_UserAuditLogResult>();
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        public int SearchAuditLogCount(string userId, string lastName, string firstName, string userRole, string actionType,
                                                                    string fromDate, string toDate)
        {
            return _totalCount;
        }

        #endregion

    }
}
