﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using System.Reflection;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    public enum AhobprRegistryFlagType
    {
        AdminFlag = 1,
        SubpopulationFlag = 2
    }

    public class AHOBPRAdminManager : AHOBPRBaseBO 
    {
        private AHOBPRShared _sharedManager = new AHOBPRShared();

        /// <summary>
        /// Select all registry flags   
        /// </summary>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public IEnumerable<STD_REGISTRY_FLAG> SelectRegistryFlags(string sort, int startRow, int maxRows,
                                                                  string keyWords, string status, string flagType, string predefined)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = "REGISTRY_FLAG_NAME";
            }

            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForRegistryFlag(_dbAhopbr);
                return SelectRegistryFlagsLinqFilter(keyWords, status, flagType, predefined).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
            }
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public int SelectRegistryFlagsCount(string sort, int startRow, int maxRows,
                                            string keyWords, string status, string flagType, string predefined)
        {
            using (_dbAhopbr = GetDataContext())
            {
                return SelectRegistryFlagsLinqFilter(keyWords, status, flagType, predefined).Count();
            }
        }

        private IQueryable<STD_REGISTRY_FLAG> SelectRegistryFlagsLinqFilter(string keyWords, string status, string flagType, string predefined)
        {
            IQueryable<STD_REGISTRY_FLAG> linqFilter = from e in _dbAhopbr.STD_REGISTRY_FLAGs
                                          select e;

            if (!string.IsNullOrEmpty(keyWords))
            {
                string[] words = keyWords.Split(' ');
                foreach (string word in words)
                {
                    if (!string.IsNullOrEmpty(word))
                    {
                        linqFilter = (from e in linqFilter
                                      where e.REGISTRY_FLAG_NAME.ToLower().Contains(word.ToLower())
                                         || e.REGISTRY_FLAG_DESCRIPTION.ToLower().Contains(word.ToLower())
                                      select e);
                    }
                }
            }

            if (status == "Active")
            {
                linqFilter = (from e in linqFilter
                              where e.ACTIVE_FLAG == true
                              select e);
            }
            else if (status == "Inactive")
            {
                linqFilter = (from e in linqFilter
                              where e.ACTIVE_FLAG == false
                              select e);
            }

            if (flagType == "Administrative Flag")
            {
                linqFilter = (from e in linqFilter
                              where e.STD_REGISTRY_FLAG_TYPE_ID == (int)AhobprRegistryFlagType.AdminFlag
                              select e);
            }
            else if (flagType == "Subpopulation Flag")
            {
                linqFilter = (from e in linqFilter
                              where e.STD_REGISTRY_FLAG_TYPE_ID == (int)AhobprRegistryFlagType.SubpopulationFlag
                              select e);
            }

            if (predefined == "Yes")
            {
                linqFilter = (from e in linqFilter
                              where e.PREDEFINED == true
                              select e);
            }
            else if (predefined == "No")
            {
                linqFilter = (from e in linqFilter
                              where e.PREDEFINED == false
                              select e);
            }

            return linqFilter;
        }

        /// <summary>
        /// Remove a Registry Flag
        /// </summary>
        /// <param name="registryFlagId"></param>
        /// <returns></returns>
        public bool RemoveRegistryFlag(int registryFlagId)
        {
            bool returnStatus = false;
            using (_dbAhopbr = GetDataContext())
            {
                STD_REGISTRY_FLAG registryFlag = (from e in _dbAhopbr.STD_REGISTRY_FLAGs
                                                  where e.STD_REGISTRY_FLAG_ID == registryFlagId
                                                  select e).FirstOrDefault();

                _dbAhopbr.STD_REGISTRY_FLAGs.DeleteOnSubmit(registryFlag);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = true;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
                catch (ChangeConflictException)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    returnStatus = false;
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
            }

            return returnStatus;
        }

        /// <summary>
        /// Add a new registry flag
        /// </summary>
        /// <param name="registryFlagName"></param>
        /// <param name="registryFlagDescription"></param>
        /// <returns></returns>
        public int AddRegistryFlag(int registryFlagTypeId, string registryFlagName, string registryFlagDescription)
        {
            int returnStatus = 0;

            if (string.IsNullOrEmpty(registryFlagName))
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                //check if registry flag exists
                STD_REGISTRY_FLAG registryFlag = (from e in _dbAhopbr.STD_REGISTRY_FLAGs
                                                  where e.REGISTRY_FLAG_NAME == registryFlagName
                                                  select e).FirstOrDefault();

                //insert if not exists
                if (registryFlag == null)
                {
                    registryFlag = new STD_REGISTRY_FLAG();

                    registryFlag.STD_REGISTRY_FLAG_TYPE_ID = registryFlagTypeId;
                    registryFlag.REGISTRY_FLAG_NAME = registryFlagName;
                    registryFlag.REGISTRY_FLAG_DESCRIPTION = registryFlagDescription;
                    registryFlag.ACTIVE_FLAG = true;

                    _dbAhopbr.STD_REGISTRY_FLAGs.InsertOnSubmit(registryFlag);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus = 1;
                    }
                    catch (System.Data.SqlClient.SqlException ex)
                    {
                        _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                    catch (ChangeConflictException e)
                    {
                        _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);
                            RegistriesGlobal.UserRegistryAll = null;
                            AHOBPRGlobal.ViewLookupAll = null;
                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                    catch
                    {
                        returnStatus = 0;
                    }

                }
                // if the registry flag already exists, set returnStatus to 2
                else
                {
                    returnStatus = 2;
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// Add a new registry flag
        /// </summary>
        /// <param name="registryFlagName"></param>
        /// <param name="registryFlagDescription"></param>
        /// <returns></returns>
        public int AddRegistryFlagForSentMessage(int registryFlagTypeId, string registryFlagName, string registryFlagDescription)
        {
            int returnStatus = 0;

            if (string.IsNullOrEmpty(registryFlagName))
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                //check if registry flag exists
                STD_REGISTRY_FLAG registryFlag = (from e in _dbAhopbr.STD_REGISTRY_FLAGs
                                                  where e.REGISTRY_FLAG_NAME == registryFlagName
                                                  select e).FirstOrDefault();

                //insert if not exists
                if (registryFlag == null)
                {
                    registryFlag = new STD_REGISTRY_FLAG();

                    registryFlag.STD_REGISTRY_FLAG_TYPE_ID = registryFlagTypeId;
                    registryFlag.REGISTRY_FLAG_NAME = registryFlagName;
                    registryFlag.REGISTRY_FLAG_DESCRIPTION = registryFlagDescription;
                    //set active_Flag to false so the admin flags don't show up in the dropdownlist for automatic generated sent messages
                    registryFlag.ACTIVE_FLAG = true;
                    registryFlag.PREDEFINED = true;

                    _dbAhopbr.STD_REGISTRY_FLAGs.InsertOnSubmit(registryFlag);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus = 1;
                    }
                    catch (System.Data.SqlClient.SqlException ex)
                    {
                        _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                    catch (ChangeConflictException e)
                    {
                        _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);
                            RegistriesGlobal.UserRegistryAll = null;
                            AHOBPRGlobal.ViewLookupAll = null;
                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                    catch
                    {
                        returnStatus = 0;
                    }

                }
                // if the registry flag already exists, set returnStatus to 2
                else
                {
                    returnStatus = 2;
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registryFlagId"></param>
        /// <returns></returns>
        public STD_REGISTRY_FLAG GetRegistryFlagById(int registryFlagId)
        {
            STD_REGISTRY_FLAG registryFlag = null;

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;
                registryFlag = (from e in _dbAhopbr.STD_REGISTRY_FLAGs
                                where e.STD_REGISTRY_FLAG_ID == registryFlagId
                                select e).FirstOrDefault();

            }

            return registryFlag;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registryFlagName"></param>
        /// <returns></returns>
        public int GetRegistryFlagIdByName(string registryFlagName)
        {
            int registryFlagId = 0;

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;
                registryFlagId = (from e in _dbAhopbr.STD_REGISTRY_FLAGs
                                  where e.REGISTRY_FLAG_NAME == registryFlagName
                                select e.STD_REGISTRY_FLAG_ID).FirstOrDefault();

            }

            return registryFlagId;
        }

        /// <summary>
        /// Update Registry Flag
        /// </summary>
        /// <param name="registryFlag"></param>
        /// <returns></returns>
        public int UpdateRegistryFlagActiveFlag(STD_REGISTRY_FLAG registryFlag)
        {
            int returnStatus = 0;

            if (registryFlag != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;
                
                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    registryFlag.SynchroniseWithDataContext(_dbAhopbr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                    if (changeSet.Updates.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                            returnStatus = 1;

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                            returnStatus = 1;
                        }
                        catch ( Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="db"></param>
        private void SetLoadWithForRegistryFlag(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<STD_REGISTRY_FLAG>(e => e.STD_REGISTRY_FLAG_TYPE);
            lo.LoadWith<STD_REGISTRY_FLAG>(e => e.REGISTRANT_REGISTRY_FLAGs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }


        /// <summary>
        /// Get build version string from assembly info
        /// </summary>
        /// <returns></returns>
        public string GetBuildVersion()
        {
            string buildVersion = string.Empty;
            Assembly assembly = Assembly.GetExecutingAssembly();
            string[] assemblyInfo = assembly.FullName.Split(",");
            foreach (string info in assemblyInfo)
            {
                if (info.Contains("Version="))
                {
                    buildVersion = info.Split("=")[1];
                    break;
                }
            }

            return buildVersion;
        }

        /// <summary>
        /// Get the database name
        /// </summary>
        /// <returns></returns>
        public string GetDatabaseName()
        {
            using (_dbAhopbr = GetDataContext())
            {
                return _dbAhopbr.Connection.Database;
            }
        }

    }
}
