﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using VeteransAffairs.Registries.BusinessManagerAHOBPR;

public static class AhobprExtensions
{
    public enum LoginResults { Disabled90Days, DisabledVista, Disabled, Pending, New, Valid };

    public static String GetCurrentUserNameWithoutDomain(this System.Security.Principal.IPrincipal user)
    {
        string userId = user.Identity.Name;

        //Remove domain name (if necessary):
        int index = userId.IndexOf("\\");
        if (0 < index)
        {
            userId = userId.Substring(index + 1);
        }
        return userId.Trim();
    }

    public static LoginResults IsValid(this VeteransAffairs.Registries.BusinessAHOBPR.AHOBPR_USER ahobprUser)
    {
        if (ahobprUser == null)
        {
            //This is not an AHOBPR user.
            //Redirect them to the create account page:
            Helpers.AddActivityToAuditLog(ahobprUser.USER_ID.ToString(),
                "New user: redirect to create account page.", null, AuditLogType.UserLogin, null);
            return LoginResults.New;
        }
        //First, make sure the account is enabled:
        if (!ahobprUser.ACTIVE_FLAG)
        {
            Helpers.AddActivityToAuditLog(ahobprUser.USER_ID.ToString(),
                "Unsuccessful Login: account disabled.", null, AuditLogType.UserLogin, null);
            return LoginResults.Disabled;
        }

        //Next, check whether the account is pending:
        if (ahobprUser.PENDING_FLAG)
        {
            Helpers.AddActivityToAuditLog(ahobprUser.USER_ID.ToString(),
                "Unsuccessful Login: account pending.", null, AuditLogType.UserLogin, null);
            return LoginResults.Pending;
        }

        //Next, check if the VISTA login has expired:
        if (ahobprUser.LAST_VISTA_LOGIN != null)
        {
            //If the last VISTA login occured more than a year ago,
            if (ahobprUser.LAST_VISTA_LOGIN < DateTime.Now.AddYears(-1))
            {
                //Display a message; give the user the ability to update VistA login
                Helpers.AddActivityToAuditLog(ahobprUser.USER_ID.ToString(),
                    "Unsuccessful Login: the last VISTA login was more than a year ago.", null, AuditLogType.UserLogin, null);
                return LoginResults.DisabledVista;
            }
        }

        //Next, check if the last login is more than 90 days ago:
        if (ahobprUser.LAST_LOGIN != null)
        {
            if (ahobprUser.LAST_LOGIN < DateTime.Now.AddDays(-90))
            {
                Helpers.AddActivityToAuditLog(ahobprUser.USER_ID.ToString(),
                    "Unsuccessful Login: last login was " + (DateTime.Now - ahobprUser.LAST_LOGIN.Value).Days + " days ago.",
                    null, AuditLogType.UserLogin, null);
                return LoginResults.Disabled90Days;
            }
        }
        return LoginResults.Valid;
    }
}
