﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessAHOBPR;
using System.Reflection;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    public enum AhobprToolQuestionStatus
    {
        InProgress = 1,
        Completed = 2,
        Deactivated = 3
    }


    public class AHOBPRToolManager : AHOBPRBaseBO 
    {
        private AHOBPRShared _sharedManager = new AHOBPRShared();
        private int _totalQuestionCount = 0;
        private int _totalFormsCount = 0;
        
        #region Methods for questions search

        /// <summary>
        /// Search Questions by keyWords and type
        /// </summary>
        /// <param name="keyWords"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        public IEnumerable<TOOL_QUESTION> SearchQuestions(string keyWords, string type, string status, string sort, int startRow, int maxRows)
        {
            if (String.IsNullOrEmpty(sort))
                sort = "TOOL_QUESTION_ID";

            using (_dbAhopbr = GetDataContext())
            {
                IQueryable<TOOL_QUESTION> entities = SearchQuestionsLinqFilter(keyWords, type, status);

                _totalQuestionCount = entities.Count();

                return (entities).SortAndPage(sort, maxRows, startRow).ToList<TOOL_QUESTION>();
            }
        }

        private IQueryable<TOOL_QUESTION> SearchQuestionsLinqFilter(string keyWords, string type, string status)
        {
            IQueryable<TOOL_QUESTION> linqFilter = from q in _dbAhopbr.TOOL_QUESTIONs
                                                select q;

            if (!string.IsNullOrEmpty(keyWords))
            {
                string[] words = keyWords.Split(' ');
                foreach (string word in words)
                {
                    if (!string.IsNullOrEmpty(word))
                    {
                        linqFilter = linqFilter.Where(q => q.KEY_WORDS.ToLower().Contains(word.ToLower())
                                                           || q.TEXT.ToLower().Contains(word.ToLower()));
                    }
                }

            }
            if (!string.IsNullOrEmpty(type))
            {
                linqFilter = linqFilter.Where(q => q.TYPE.ToLower() == type.ToLower());

            }
            if (!string.IsNullOrEmpty(status))
            {
                int statusNumber = Convert.ToInt32(status);
                switch (statusNumber)
                {
                    case (int)AhobprToolQuestionStatus.InProgress:
                        linqFilter = linqFilter.Where(q => q.EDITABLE_FLAG == true);
                        break;
                    case (int)AhobprToolQuestionStatus.Completed:
                        linqFilter = linqFilter.Where(q => q.EDITABLE_FLAG == false).Where(q => q.ACTIVE_FLAG == true);
                        break;
                    case (int)AhobprToolQuestionStatus.Deactivated:
                        linqFilter = linqFilter.Where(q => q.EDITABLE_FLAG == false).Where(q => q.ACTIVE_FLAG == false);
                        break;
                }

            }

            return linqFilter;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="text"></param>
        /// <param name="help"></param>
        /// <param name="editable"></param>
        /// <param name="clinicalRelevant"></param>
        /// <param name="questionId"></param>
        /// <returns></returns>
        public int AddAnswer(string text, string help, bool editable, bool clinicalRelevant, string order, int questionId, bool standard)
        {
            int returnStatus = 0;

            if (string.IsNullOrEmpty(text))
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                //check if registry flag exists
                TOOL_ANSWER answer = (from e in _dbAhopbr.TOOL_ANSWERs
                                                  where e.TEXT == text && e.TOOL_QUESTION_ID == questionId
                                                  select e).FirstOrDefault();

                //insert if not exists
                if (answer == null)
                {
                    answer = new TOOL_ANSWER();

                    answer.TOOL_QUESTION_ID = questionId;
                    answer.TEXT = text;
                    answer.EDITABLE = editable;
                    answer.CLINICAL_RELEVANT_FLAG = clinicalRelevant;
                    answer.ORDER = Convert.ToInt32(order);
                    answer.STANDARD_FLAG = standard;
                    if (!string.IsNullOrEmpty(help))
                    {
                        answer.HELP = help;
                    }

                    _dbAhopbr.TOOL_ANSWERs.InsertOnSubmit(answer);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus = 1;
                    }
                    catch (System.Data.SqlClient.SqlException ex)
                    {
                        _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                    catch (ChangeConflictException e)
                    {
                        _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                    catch
                    {
                        returnStatus = 0;
                    }

                }
                // if the registry flag already exists, set returnStatus to 2
                else
                {
                    returnStatus = 2;
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="questionId"></param>
        /// <param name="formId"></param>
        /// <returns></returns>
        public int AddQuestionToForm(int questionId, int formId)
        {
            int returnStatus = 0;

            if (questionId == 0 || formId == 0)
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                //check if registry flag exists
                TOOL_FORM_QUESTION formQuestion = (from e in _dbAhopbr.TOOL_FORM_QUESTIONs
                                                   where e.TOOL_QUESTION_ID == questionId && e.TOOL_FORM_ID == formId
                                                   select e).FirstOrDefault();

                //insert if not exists
                if (formQuestion == null)
                {
                    formQuestion = new TOOL_FORM_QUESTION();

                    formQuestion.TOOL_QUESTION_ID = questionId;
                    formQuestion.TOOL_FORM_ID = formId;

                    _dbAhopbr.TOOL_FORM_QUESTIONs.InsertOnSubmit(formQuestion);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        returnStatus = 1;
                    }
                    catch (System.Data.SqlClient.SqlException ex)
                    {
                        _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        returnStatus = -1;
                    }
                    catch (ChangeConflictException e)
                    {
                        _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                    catch
                    {
                        returnStatus = 0;
                    }

                }
                // if the registry flag already exists, set returnStatus to 2
                else
                {
                    returnStatus = 2;
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="questionId"></param>
        /// <param name="formId"></param>
        /// <returns></returns>
        public bool RemoveQuestionFromForm(int questionId, int formId)
        {
            bool returnStatus = false;
            using (_dbAhopbr = GetDataContext())
            {
                TOOL_FORM_QUESTION formQuestion = (from e in _dbAhopbr.TOOL_FORM_QUESTIONs
                                      where e.TOOL_FORM_ID == formId && e.TOOL_QUESTION_ID == questionId
                                      select e).FirstOrDefault();

                _dbAhopbr.TOOL_FORM_QUESTIONs.DeleteOnSubmit(formQuestion);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = true;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
                catch (ChangeConflictException)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    returnStatus = false;
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
            }

            return returnStatus;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="answerId"></param>
        /// <returns></returns>
        public bool RemoveAnswer(int answerId)
        {
            bool returnStatus = false;
            using (_dbAhopbr = GetDataContext())
            {
                TOOL_ANSWER answer = (from e in _dbAhopbr.TOOL_ANSWERs
                                      where e.TOOL_ANSWER_ID == answerId
                                      select e).FirstOrDefault();

                _dbAhopbr.TOOL_ANSWERs.DeleteOnSubmit(answer);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = true;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
                catch (ChangeConflictException)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    returnStatus = false;
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
            }

            return returnStatus;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="questionId"></param>
        /// <param name="standardAnswer"></param>
        /// <returns></returns>
        public bool RemoveStandardAnswer(int questionId, string standardAnswer)
        {
            bool returnStatus = false;
            using (_dbAhopbr = GetDataContext())
            {
                TOOL_ANSWER answer = (from e in _dbAhopbr.TOOL_ANSWERs
                                      where e.TOOL_QUESTION_ID == questionId
                                        && e.STANDARD_FLAG == true
                                        && e.TEXT == standardAnswer
                                      select e).FirstOrDefault();

                _dbAhopbr.TOOL_ANSWERs.DeleteOnSubmit(answer);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                    returnStatus = true;
                }
                catch (System.Data.SqlClient.SqlException ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
                catch (ChangeConflictException)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    returnStatus = false;
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                    returnStatus = false;
                }
            }

            return returnStatus;
        }

        
        /// <summary>
        /// 
        /// </summary>
        /// <param name="keyWords"></param>
        /// <param name="type"></param>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public int SearchQuestionsCount(string keyWords, string type, string status, string sort, int startRow, int maxRows)
        {
            return _totalQuestionCount;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="keyWords"></param>
        /// <param name="status"></param>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public IEnumerable<TOOL_FORM> SearchForms(string keyWords, string status, string systemWide, string sort, int startRow, int maxRows)
        {
            if (String.IsNullOrEmpty(sort))
                sort = "TOOL_FORM_ID";

            using (_dbAhopbr = GetDataContext())
            {
                IQueryable<TOOL_FORM> entities = SearchFormsLinqFilter(keyWords, status, systemWide);

                _totalFormsCount = entities.Count();

                return (entities).SortAndPage(sort, maxRows, startRow).ToList<TOOL_FORM>();
            }
        }

        private IQueryable<TOOL_FORM> SearchFormsLinqFilter(string keyWords, string status, string systemWide)
        {
            IQueryable<TOOL_FORM> linqFilter = from q in _dbAhopbr.TOOL_FORMs
                                                   select q;

            if (!string.IsNullOrEmpty(keyWords))
            {
                string[] words = keyWords.Split(' ');
                foreach (string word in words)
                {
                    if (!string.IsNullOrEmpty(word))
                    {
                        linqFilter = linqFilter.Where(q => q.NAME.ToLower().Contains(word.ToLower())
                                                           || q.TITLE.ToLower().Contains(word.ToLower()));
                    }
                }
            }
            if (!string.IsNullOrEmpty(status))
            {
                int statusNumber = Convert.ToInt32(status);
                switch (statusNumber)
                {
                    case (int)AhobprToolQuestionStatus.InProgress:
                        linqFilter = linqFilter.Where(q => q.EDITABLE_FLAG == true);
                        break;
                    case (int)AhobprToolQuestionStatus.Completed:
                        linqFilter = linqFilter.Where(q => q.EDITABLE_FLAG == false).Where(q => q.ACTIVE_FLAG == true);
                        break;
                    case (int)AhobprToolQuestionStatus.Deactivated:
                        linqFilter = linqFilter.Where(q => q.EDITABLE_FLAG == false).Where(q => q.ACTIVE_FLAG == false);
                        break;
                }
            }
            if (!string.IsNullOrEmpty(systemWide))
            {
                switch (systemWide)
                {
                    case "yes":
                        linqFilter = linqFilter.Where(q => q.SYSTEM_WIDE_FLAG == true);
                        break;
                    case "no":
                        linqFilter = linqFilter.Where(q => q.SYSTEM_WIDE_FLAG == false);
                        break;
                }
            }

            return linqFilter;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="keyWords"></param>
        /// <param name="status"></param>
        /// <param name="sort"></param>
        /// <param name="startRow"></param>
        /// <param name="maxRows"></param>
        /// <returns></returns>
        public int SearchFormsCount(string keyWords, string status, string systemWide, string sort, int startRow, int maxRows)
        {
            return _totalFormsCount;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="questionId"></param>
        /// <returns></returns>
        public bool RemoveQuestion(int questionId)
        {
            bool returnStatus = false;
            TOOL_QUESTION question = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForQuestion(_dbAhopbr);
                question = (from e in _dbAhopbr.TOOL_QUESTIONs
                                          where e.TOOL_QUESTION_ID == questionId
                                                  select e).FirstOrDefault();
            }
            if (question != null)
            {
                question.SetAsChangeTrackingRoot();
                question.SetAsDeleteOnSubmit();
                foreach (TOOL_ANSWER answer in question.TOOL_ANSWERs)
                {
                    answer.SetAsDeleteOnSubmit();
                }
                try
                {
                    question = SaveQuestion(question);
                    if (question == null)
                    {
                        returnStatus = true;
                    }
                }
                catch
                {
                    returnStatus = false;
                }
            }

            return returnStatus;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="formId"></param>
        /// <returns></returns>
        public bool RemoveForm(int formId)
        {
            bool returnStatus = false;
            TOOL_FORM form = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForForm(_dbAhopbr);
                form = (from e in _dbAhopbr.TOOL_FORMs
                            where e.TOOL_FORM_ID == formId
                            select e).FirstOrDefault();
            }
            if (form != null)
            {
                form.SetAsChangeTrackingRoot();
                form.SetAsDeleteOnSubmit();
                foreach (TOOL_FORM_QUESTION item in form.TOOL_FORM_QUESTIONs)
                {
                    item.SetAsDeleteOnSubmit();
                }
                try
                {
                    form = SaveForm(form);
                    if (form == null)
                    {
                        returnStatus = true;
                    }
                }
                catch
                {
                    returnStatus = false;
                }
            }

            return returnStatus;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="questionId"></param>
        /// <returns></returns>
        public bool ActivateDeactivateQuestion(int questionId)
        {
            bool returnStatus = false;
            TOOL_QUESTION question = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForQuestion(_dbAhopbr);
                question = (from e in _dbAhopbr.TOOL_QUESTIONs
                            where e.TOOL_QUESTION_ID == questionId
                            select e).FirstOrDefault();
            }
            if (question != null)
            {
                question.SetAsChangeTrackingRoot();
                question.SetAsUpdateOnSubmit();
                question.ACTIVE_FLAG = !question.ACTIVE_FLAG;
                try
                {
                    question = SaveQuestion(question);
                    if (question != null)
                    {
                        returnStatus = true;
                    }
                }
                catch
                {
                    returnStatus = false;
                }
            }

            return returnStatus;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="formId"></param>
        /// <returns></returns>
        public bool ActivateDeactivateForm(int formId)
        {
            bool returnStatus = false;
            TOOL_FORM form = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForForm(_dbAhopbr);
                form = (from e in _dbAhopbr.TOOL_FORMs
                        where e.TOOL_FORM_ID == formId
                        select e).FirstOrDefault();
            }
            if (form != null)
            {
                form.SetAsChangeTrackingRoot();
                form.SetAsUpdateOnSubmit();
                form.ACTIVE_FLAG = !form.ACTIVE_FLAG;
                try
                {
                    form = SaveForm(form);
                    if (form != null)
                    {
                        returnStatus = true;
                        AHOBPRGlobal.ViewLookupAll = null;
                    }
                }
                catch
                {
                    returnStatus = false;
                }
            }

            return returnStatus;
        }


        private void SetLoadWithForQuestion(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<TOOL_QUESTION>(e => e.TOOL_ANSWERs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="question"></param>
        /// <returns></returns>
        public TOOL_QUESTION SaveQuestion(TOOL_QUESTION question)
        {
            bool delete = false;
            if (question != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    question.SynchroniseWithDataContext(_dbAhopbr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                    if (changeSet.Updates.Count > 0 || changeSet.Inserts.Count > 0 || changeSet.Deletes.Count > 0)
                    {
                        delete = changeSet.Deletes.Count > 0;

                        //if changes present then submit changes
                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        }
                    }
                }
            }
            return delete ? null : question;
        }


        /// <summary>
        /// Update the order for an answer
        /// </summary>
        /// <param name="answerId"></param>
        /// <param name="order"></param>
        /// <returns></returns>
        public bool UpdateAnswer(int answerId, int order)
        {
            bool results = false;
            if (answerId > 0 && order > 0)
            {
                TOOL_ANSWER answer = GetAnswerById(answerId);

                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;

                    if (answer != null)
                    {
                        answer.ORDER = order;
                        answer.SetAsChangeTrackingRoot();
                        answer.SetAsUpdateOnSubmit();
                        //this line traverses all entities, attaching all of them as appropriate to the data context.
                        answer.SynchroniseWithDataContext(_dbAhopbr);

                        //Check if any actual changes will occur
                        ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                        if (changeSet.Updates.Count > 0)
                        {
                            //if changes present then submit changes
                            try
                            {
                                _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                                results = true;

                            }
                            catch (System.Data.SqlClient.SqlException ex)
                            {
                                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                                results = false;
                            }
                            catch (ChangeConflictException)
                            {
                                _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                                results = true;
                            }
                            catch (Exception ex)
                            {
                                _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                                results = false;
                            }
                        }
                    }

                }
            }
            return results;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="form"></param>
        /// <returns></returns>
        public int SubmitFollowUpFormToStdTables(int toolFormId)
        {
            int result = 0;
            TOOL_FORM toolForm = GetFormById(toolFormId);
            if (toolForm != null)
            {
                result = SaveStdForm(CreateStdFormFromToolForm(toolForm));
                AHOBPRGlobal.ViewLookupAll = null;
            }

            return result;
        }

        /// <summary>
        /// Create a STD_FORM object form TOOL_FORM
        /// </summary>
        /// <param name="toolForm"></param>
        /// <returns></returns>
        private STD_FORM CreateStdFormFromToolForm(TOOL_FORM toolForm)
        {
            STD_FORM stdForm = new STD_FORM();
            stdForm.SetAsChangeTrackingRoot();
            stdForm.SetAsInsertOnSubmit();

            stdForm.NAME = toolForm.NAME;
            stdForm.TITLE = toolForm.TITLE;
            stdForm.BPR_FORM_ID = Guid.NewGuid().ToString();
            stdForm.FOLLOWUP_FLAG = true;
            stdForm.VERSION = DateTime.Today.Year + "." + DateTime.Today.Month + "." + DateTime.Today.Day;
            stdForm.TOOL_FORM_ID = toolForm.TOOL_FORM_ID;

            //section
            STD_FORM_SECTION section = new STD_FORM_SECTION();
            section.TITLE = "Follow up form section";
            section.BRP_FORM_SECTION_ID = 1;
            section.TOOL_TIP = "Follow up form section";
            stdForm.STD_FORM_SECTIONs.Add(section);
            section.SetAsInsertOnSubmit();

            //sub section
            STD_FORM_SUB_SECTION subSection = new STD_FORM_SUB_SECTION();
            subSection.TITLE = "Follow up form sub section";
            subSection.TOOL_TIP = "Follow up form sub section";
            subSection.BRP_FORM_SUB_SECTION_ID = 1;
            stdForm.STD_FORM_SUB_SECTIONs.Add(subSection);
            section.STD_FORM_SUB_SECTIONs.Add(subSection);
            subSection.SetAsInsertOnSubmit();

            //question
            STD_FORM_QUESTION stdQuestion = new STD_FORM_QUESTION();
            List<TOOL_FORM_QUESTION> toolFormQuestions = (from e in toolForm.TOOL_FORM_QUESTIONs
                                                          orderby e.TOOL_FORM_QUESTION_ID
                                                          select e).ToList();
            int sequence = 0;
            foreach (TOOL_FORM_QUESTION toolFormQuestion in toolFormQuestions)
            {
                sequence++;
                stdQuestion = new STD_FORM_QUESTION();
                stdQuestion.TEXT = toolFormQuestion.TOOL_QUESTION.TEXT;
                stdQuestion.TYPE = toolFormQuestion.TOOL_QUESTION.TYPE;
                stdQuestion.MIN = toolFormQuestion.TOOL_QUESTION.MIN;
                stdQuestion.MAX = toolFormQuestion.TOOL_QUESTION.MAX;
                stdQuestion.BRP_FORM_QUESTION_ID = sequence.ToString();
                stdQuestion.FORM_QUESTION_UNIQUE_ID = sequence.ToString();
                stdQuestion.SHORT_TEXT = toolFormQuestion.TOOL_QUESTION.TEXT;
                if (stdQuestion.TYPE == "multiSelect")
                {
                    int nonStandardAnswersCount = (from e in toolFormQuestion.TOOL_QUESTION.TOOL_ANSWERs
                                                where e.STANDARD_FLAG == false
                                                select e).Count();
                    stdQuestion.MAX_CHOICES = nonStandardAnswersCount;
                }
                else
                {
                    stdQuestion.MAX_CHOICES = 1;
                }
                stdForm.STD_FORM_QUESTIONs.Add(stdQuestion);
                section.STD_FORM_QUESTIONs.Add(stdQuestion);
                subSection.STD_FORM_QUESTIONs.Add(stdQuestion);
                stdQuestion.SetAsInsertOnSubmit();

                //answers for the question
                STD_FORM_ANSWER stdAnswer = new STD_FORM_ANSWER();
                List<TOOL_ANSWER> toolAnswers = (from e in toolFormQuestion.TOOL_QUESTION.TOOL_ANSWERs
                                                 orderby e.STANDARD_FLAG, e.ORDER
                                                 select e).ToList();
                int answerSquence = 0;
                foreach (TOOL_ANSWER toolAnswer in toolAnswers)
                {
                    answerSquence++;
                    stdAnswer = new STD_FORM_ANSWER();
                    stdAnswer.TEXT = toolAnswer.TEXT;
                    stdAnswer.HELP = String.IsNullOrEmpty(toolAnswer.HELP) ? "" : toolAnswer.HELP; 
                    stdAnswer.EDITABLE = toolAnswer.EDITABLE;
                    stdAnswer.CLINICAL_RELEVANT_FLAG = toolAnswer.CLINICAL_RELEVANT_FLAG;
                    stdAnswer.ORDER = answerSquence;
                    stdAnswer.BRP_FORM_ANSWER_ID = answerSquence.ToString();

                    stdForm.STD_FORM_ANSWERs.Add(stdAnswer);
                    section.STD_FORM_ANSWERs.Add(stdAnswer);
                    subSection.STD_FORM_ANSWERs.Add(stdAnswer);
                    stdQuestion.STD_FORM_ANSWERs.Add(stdAnswer);
                    stdAnswer.SetAsInsertOnSubmit();
                }

            }

            return stdForm;

        }

        /// <summary>
        /// Save STD_FORM
        /// </summary>
        /// <param name="form"></param>
        /// <returns></returns>
        public int SaveStdForm(STD_FORM form)
        {
            int returnStatus = 0;

            if (form != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    form.SynchroniseWithDataContext(_dbAhopbr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                    if (changeSet.Updates.Count > 0 || changeSet.Inserts.Count > 0)
                    {
                        //if changes present then submit changes
                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);
                            returnStatus = 1;

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                            returnStatus = 1;
                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                            returnStatus = -1;
                        }
                    }
                }
            }
            return returnStatus;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="form"></param>
        /// <returns></returns>
        public TOOL_FORM SaveForm(TOOL_FORM form)
        {
            bool delete = false;
            if (form != null)
            {
                using (_dbAhopbr = GetDataContext())
                {
                    _dbAhopbr.DeferredLoadingEnabled = false;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    form.SynchroniseWithDataContext(_dbAhopbr);

                    //Check if any actual changes will occur
                    ChangeSet changeSet = _dbAhopbr.GetChangeSet();

                    if (changeSet.Updates.Count > 0 || changeSet.Inserts.Count > 0 || changeSet.Deletes.Count > 0)
                    {
                        delete = changeSet.Deletes.Count > 0;

                        //if changes present then submit changes
                        try
                        {
                            _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                        }
                        catch (System.Data.SqlClient.SqlException ex)
                        {
                            _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        }
                        catch (ChangeConflictException)
                        {
                            _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        }
                        catch (Exception ex)
                        {
                            _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                        }
                    }
                }
            }
            return delete ? null : form;
        }

        /// <summary>
        /// Get Question By ID
        /// </summary>
        /// <param name="questionId"></param>
        /// <returns></returns>
        public TOOL_QUESTION GetQuestionById(int questionId)
        {
            TOOL_QUESTION question = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForQuestion(_dbAhopbr);
                _dbAhopbr.DeferredLoadingEnabled = false;
                question = (from e in _dbAhopbr.TOOL_QUESTIONs
                            where e.TOOL_QUESTION_ID == questionId
                            select e).FirstOrDefault();

            }
            return question;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="answerId"></param>
        /// <returns></returns>
        public TOOL_ANSWER GetAnswerById(int answerId)
        {
            TOOL_ANSWER answer = null;
            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;
                answer = (from e in _dbAhopbr.TOOL_ANSWERs
                          where e.TOOL_ANSWER_ID == answerId
                          select e).FirstOrDefault();
            }
            return answer;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="formId"></param>
        /// <returns></returns>
        public List<TOOL_QUESTION> GetQuestionsByFormId(int formId)
        {
            List<TOOL_QUESTION> questions = new List<TOOL_QUESTION>();
            List<TOOL_FORM_QUESTION> formQuestions = new List<TOOL_FORM_QUESTION>();

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;
                SetLoadWithForFormQuestion(_dbAhopbr);
                formQuestions = (from e in _dbAhopbr.TOOL_FORM_QUESTIONs 
                             where e.TOOL_FORM_ID == formId
                             orderby e.TOOL_FORM_QUESTION_ID 
                             select e).ToList();

            }

            foreach (TOOL_FORM_QUESTION item in formQuestions)
            {
                questions.Add(item.TOOL_QUESTION);
            }
            return questions;
        }

        private void SetLoadWithForFormQuestion(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<TOOL_FORM_QUESTION>(e => e.TOOL_QUESTION);


            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="formId"></param>
        /// <returns></returns>
        public TOOL_FORM GetFormById(int formId)
        {
            TOOL_FORM form = null;
            using (_dbAhopbr = GetDataContext())
            {
                SetLoadWithForForm(_dbAhopbr);
                _dbAhopbr.DeferredLoadingEnabled = false;

                form = (from e in _dbAhopbr.TOOL_FORMs
                        where e.TOOL_FORM_ID == formId
                            select e).FirstOrDefault();

            }
            return form;
        }

        private void SetLoadWithForForm(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<TOOL_FORM>(e => e.TOOL_FORM_QUESTIONs);
            lo.LoadWith<TOOL_FORM_QUESTION>(e => e.TOOL_QUESTION);
            lo.LoadWith<TOOL_QUESTION>(e => e.TOOL_ANSWERs);


            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        #endregion
    }
}
