﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessManagerAHOBPR.ValueTypes;
using System.Reflection;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    /// <summary>
    /// ViewModel class for displaying available stations
    /// </summary>
    public class Station
    {
        public string Code { get; private set; }
        public string Description { get; private set; }
        public Station(string code, string description)
        {
            Code = code;
            Description = description;
        }
    }

    public class AHOBPRCprsManager : AHOBPRBaseBO 
    {
        //private AHOBPRShared _sharedManager = new AHOBPRShared();

        #region Methods for FindRegistrant

        /// <summary>
        /// Find registrant by icn or ssn & dob
        /// </summary>
        /// <param name="icn"></param>
        /// <param name="ssn"></param>
        /// <param name="dob"></param>
        /// <returns></returns>
        public REGISTRANT FindRegistrant(string icn, string ssn, string dob)
        {
            REGISTRANT registrant = null;
            using (_dbAhopbr = GetDataContext())
            {

                registrant = (from e in _dbAhopbr.REGISTRANTs
                              where e.ICN == icn
                                  || (e.Snum.Replace("-", "") == ssn.Replace("-", "")
                                        && e.BIRTH_DATE == FormatDob(dob))
                              select e).FirstOrDefault();                         
            }

            return registrant;
        }

        private DateTime FormatDob (string dob)
        {
            DateTime result = DateTime.MinValue;
            if (dob.Length == 8)
            {
                string dobDate = dob.Substring(4, 2) + "/" + dob.Substring(6, 2) + "/" + dob.Substring(0, 4);
                result = Convert.ToDateTime(dobDate);
            }

            return result;
        }
        #endregion

        #region Methods for CprsLog

        /// <summary>
        /// Save an entry to CPRS_LOG table after a note is written to CPRS
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="noteIen"></param>
        /// <param name="noteContent"></param>
        /// <param name="siteCode"></param>
        /// <param name="encounterString"></param>
        /// <param name="patientIen"></param>
        /// <param name="patientIcn"></param>
        /// <param name="providerID"></param>
        /// <param name="providerName"></param>
        /// <returns></returns>
        public bool SaveCprsLog(int registrantId, string noteIen, string noteContent, string siteCode, string encounterString,
                                string patientIen, string patientIcn, string providerID, string providerName)
        {
            bool returnStatus = true;

            if (registrantId == 0)
            {
                return returnStatus;
            }

            using (_dbAhopbr = GetDataContext())
            {
                _dbAhopbr.DeferredLoadingEnabled = false;

                //check if registry flag exists
                CPRS_LOG cprsLog = new CPRS_LOG();
                cprsLog.REGISTRANT_ID = registrantId;
                cprsLog.SITE_CODE = siteCode;
                cprsLog.PROVIDER_ID = providerID;
                cprsLog.PROVIDER_NAME = providerName;
                cprsLog.NOTE_IEN = noteIen;
                cprsLog.NOTE_CONTENT = noteContent;
                cprsLog.ENCOUNTER_STRING = encounterString;
                cprsLog.PATIENT_IEN = patientIen;
                cprsLog.PATIENT_ICN = patientIcn;

                _dbAhopbr.CPRS_LOGs.InsertOnSubmit(cprsLog);

                try
                {
                    _dbAhopbr.SubmitChanges(ConflictMode.ContinueOnConflict);

                    returnStatus = true;
                }
                catch (System.Data.SqlClient.SqlException e)
                {
                    returnStatus = false;
                }
                catch (ChangeConflictException e)
                {
                    _dbAhopbr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                    try
                    {
                        _dbAhopbr.SubmitChanges(ConflictMode.FailOnFirstConflict);

                        returnStatus = true;
                    }
                    catch
                    {
                        returnStatus = false;
                    }
                }
                catch
                {
                    returnStatus = false;
                }

            }

            return returnStatus;
        }

        /// <summary>
        /// Check if a note is in CPRS_LOG table
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="noteIen"></param>
        /// <param name="noteContent"></param>
        /// <param name="siteCode"></param>
        /// <returns></returns>
        public bool InCprsLog(int registrantId, string noteContent, string siteCode)
        {
            bool result = false;
            using (_dbAhopbr = GetDataContext())
            {
                noteContent = noteContent.Replace("<br>", "|").Replace(Environment.NewLine, "|");
                CPRS_LOG log = (from e in _dbAhopbr.CPRS_LOGs
                                where e.REGISTRANT_ID == registrantId
                                    && e.NOTE_CONTENT == noteContent
                                    && e.SITE_CODE == siteCode
                                select e).FirstOrDefault();
                result = (log != null);
            }

            return result;
        }

        #endregion

        #region Methods for FindSiteCodes
        public List<Station> GetStationsByBaseCode(string baseStation)
        {
            List<Station> stations = null;

            using (_dbRegistry = GetRegistryDataContext())
            {
                stations = (from e in _dbRegistry.STD_INSTITUTIONs
                            where e.STATIONNUMBER.Substring(0, 3).ToUpper() == baseStation.ToUpper() 
                            orderby e.STATIONNUMBER
                            select new Station(e.STATIONNUMBER, e.NAME + " (" + e.STATIONNUMBER + ")")).ToList();
            }
            return stations;
        }
        #endregion
    }
}
