﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Linq.Expressions;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManagerAHOBPR;
using VeteransAffairs.Registries.BusinessManagerAHOBPR.ViaEmrServiceReference;
using VeteransAffairs.Registries.BusinessManager.MDWS;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using System.Threading;
using System.Web.Configuration;

namespace VeteransAffairs.Registries.BusinessManager
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class ViaManager : BaseBO
    {
        private EmrServiceImplService _viaClient = new EmrServiceImplService();
        private queryBean _queryBean = new queryBean();
        private MDWSLog _logManager = new MDWSLog();
        private MDWS_LOG _viaLog = new MDWS_LOG();
        private AHOBPRShared _sharedManager = new AHOBPRShared();

        public ViaManager(string appId, string appToken, string appPassword)
        {
            _queryBean.requestingApp = Helpers.GetConfigEntry("ViaAppId");
            _queryBean.consumingAppToken = Helpers.GetConfigEntry("ViaAppToken");
            _queryBean.consumingAppPassword = Helpers.GetConfigEntry("ViaAppToken2");
        }

        public personsTO CprsLaunch(string duz, string dfn, string siteCode)
        {
            personsTO result = null;
            SetLogDefaults(ref _viaLog);
            _viaLog.SERVICE_NAME = "cprsLaunch";
            _viaLog.PARAMETERS = xmlParameters(new string[] { "CPRS Launch", siteCode, duz, dfn });

            _queryBean.provider = new provider();
            _queryBean.patient = new patient();
            _queryBean.provider.userId = duz;
            _queryBean.provider.loginSiteCode = siteCode;
            _queryBean.patient.localPid = dfn;
            _queryBean.patient.localSiteId = siteCode;
            _queryBean.criteria = "AHOBPR";


            // Call CPRS Launch 
            try
            {
                result = _viaClient.cprsLaunch(_queryBean);
                if (result != null)
                {
                    _viaLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _viaLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _viaLog.ERROR_LEVEL = 2;
                }
            }
            catch
            {
                _viaLog.ERROR_LEVEL = 2;
            }

            _logManager.LogMDWSCall(_viaLog);

            return result;
        }

        /// <summary>
        /// Get all note titles 
        /// </summary>
        /// <param name="personTo"></param>
        /// <returns></returns>
        public string getNoteTitleIenFromVIA(personsTO personTo, string ahobprNoteTitle)
        {
            string result = string.Empty;

            SetLogDefaults(ref _viaLog);
            _viaLog.SERVICE_NAME = "getNoteTitles";
            _viaLog.PARAMETERS = xmlParameters(new string[] { personTo.user.siteId, personTo.user.DUZ, "direction=1", "target=" + ahobprNoteTitle });

            try
            {
                _queryBean.provider = new provider();
                _queryBean.patient = new patient();
                _queryBean.provider.userId = personTo.user.DUZ;
                _queryBean.provider.name = personTo.user.name;
                _queryBean.provider.loginSiteCode = personTo.user.siteId;

                _queryBean.direction = "1";
                _queryBean.target = ahobprNoteTitle;
                _queryBean.recordSiteCode = personTo.user.siteId;
                _queryBean.criteria = "AHOBPR";

                taggedTextArray noteTitles = _viaClient.getNoteTitles(_queryBean);
                 if (noteTitles != null)
                {
                    _viaLog.RETURNED_DATA = noteTitles.Serialize();
                    
                     if (noteTitles.fault != null)
                    {
                        _viaLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        if (noteTitles.count > 0)
                        {
                            _viaLog.ERROR_LEVEL = 0;
                            result = (from e in noteTitles.results.FirstOrDefault().taggedResults
                                      where e.textArray[0] == ahobprNoteTitle
                                      select e.tag).FirstOrDefault();                           
                        }
                        else
                        {
                            _viaLog.ERROR_LEVEL = 1;
                        }
                    }
                }
                else
                {
                    _sharedManager.LogErrorMessage("VIA Error", "GetNoteTitleIen", "VIA Service call get Note Titles returned null");
                    _viaLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                 _sharedManager.LogErrorMessage("VIA Error", "GetNoteTitleIen", exc.Message);
                _viaLog.ERROR_LEVEL = 2;
                throw;
            }
            _logManager.LogMDWSCall(_viaLog);

             return result;
        }


        /// <summary>
        /// Write Note to CPRS
        /// </summary>
        /// <param name="cprsServiceTO"></param>
        /// <returns></returns>
        public bool WriteNote(string noteTitleIen, string noteText, personsTO personTo, string registrantId)
        {
            bool result = false;
            noteResultTO writeNoteResult = null;

            string encounterString = "1;" + FormatDateForVIA(DateTime.Now, true) + ";E";

            //For preserving line breaks within a text block that is being written to a note, replace the \r\n or \n characters with a pipe (|).
            noteText = noteText.Replace("<br>", "|").Replace(Environment.NewLine, "|");

            SetLogDefaults(ref _viaLog);
            _viaLog.SERVICE_NAME = "writeNote";
            _viaLog.PARAMETERS = xmlParameters(new string[] { noteTitleIen, encounterString, noteText, personTo.user.DUZ, personTo.user.siteId, personTo.user.name, personTo.patient.localPid, personTo.patient.mpiPid});

            try
            {
                _queryBean.provider = new provider();
                _queryBean.patient = new patient();
                _queryBean.provider.userId = personTo.user.DUZ;
                _queryBean.provider.name = personTo.user.name;
                _queryBean.provider.loginSiteCode = personTo.user.siteId;

                _queryBean.patient.localPid = personTo.patient.localPid;
                _queryBean.patient.localSiteId = personTo.user.siteId;
                _queryBean.patient.vistaLocations = personTo.user.siteId;
                _queryBean.patient.mpiPid = personTo.patient.mpiPid;

                _queryBean.recordSiteCode = personTo.user.siteId;
                _queryBean.criteria = "AHOBPR";

                writeNoteResult = _viaClient.writeNote(noteTitleIen, encounterString, noteText, "", "", "", "", _queryBean);
                if (writeNoteResult != null)
                {
                    _viaLog.RETURNED_DATA = writeNoteResult.Serialize();

                    if (!String.IsNullOrEmpty(writeNoteResult.id))
                    {
                        result = true;
                         //log to CRPS_LOG table
                        AHOBPRCprsManager cprsManager = new AHOBPRCprsManager();
                        cprsManager.SaveCprsLog( Convert.ToInt32(registrantId), noteTitleIen, noteText, personTo.user.siteId, encounterString, personTo.patient.localPid,
                                    personTo.patient.mpiPid + "V" + personTo.patient.mpiChecksum, personTo.user.DUZ, personTo.user.name);
                   }

                    if (writeNoteResult.fault != null)
                    {
                        _viaLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _viaLog.ERROR_LEVEL = 2;
                }
            }
            catch
            {
                _viaLog.ERROR_LEVEL = 2;
            }

            _logManager.LogMDWSCall(_viaLog);

            return result;
        }

        /// <summary>
        /// Add a record to audit log
        /// </summary>
        public void AddToAuditLog(string duz, string dfn, string siteCode, string serverIP, string serverPort, string userID, AuditLogType type)
        {
            if (duz.Length > 0)
            {
                string message = "CPRS Tools menu access.  Site Code: " + siteCode + ", ";
                message = message + "DUZ: " + duz + ", ";
                message = message + "DFN: " + dfn + ",";
                message = message + "ServerIP: " + serverIP + ",";
                message = message + "ServerPort: " + serverPort + ",";

                string userId = userID.Length > 0 ? userID : duz;

                Helpers.AddActivityToAuditLog(userId, message, dfn, type, null);
            }
        }



        public void SetLogDefaults(ref MDWS_LOG log)
        {
            RegistriesCommonManager _commonManager = new RegistriesCommonManager();

            try
            {
                log.STD_REGISTRY_ID = _commonManager.GetRegistryId(WebConfigurationManager.AppSettings.Get("Registry"));
            }
            catch (Exception)
            {
                //This should never happen. It will cause an error on save since STD_REGISTRY_ID field is not nullable;
            }

            log.SENT = System.DateTime.Now;
            log.ERROR_LEVEL = 0;
        }

        /// <summary>
        /// Format input parameter
        /// </summary>
        /// <param name="parameterArray"></param>
        /// <returns></returns>
        public string xmlParameters(string[] parameterArray)
        {
            string result = String.Empty;
            string elementTemplate = "<pNumber>Parameter</pNumber>";
            string element;

            if (parameterArray != null)
            {
                for (int i = 0; i < parameterArray.Length; i++)
                {
                    element = elementTemplate.Replace("Number", (i + 1).ToString()); //Make parameter number 1-based
                    element = element.Replace("Parameter", parameterArray[i]);
                    result = result + element;
                }

                if (!String.IsNullOrEmpty(result))
                    result = "<Parameters>" + result + "</Parameters>";
            }

            return result;
        }

        /// <summary>
        /// VIA: FormatDateForVIA date formatting
        /// </summary>
        /// <param name="dateValue"></param>
        /// <returns></returns>
        private string FormatDateForVIA(DateTime dateValue, bool numeric)
        {
            // CPRS is expecting date in the following format "20000101.000000";

            if (dateValue != null)
            {
                if (numeric)
                {
                    return String.Format("{0:yyyyMMdd}", dateValue);
                }
                else
                {
                    return (dateValue.Year - 1700) + "" + dateValue.ToString("MM") + dateValue.ToString("dd"); // +"." + dateValue.ToString("HH") + dateValue.ToString("mm") + dateValue.ToString("ss");
                }
            }
            else
            {
                return null;
            }
        }
    }
}