package gov.va.med.authentication.kernel.sspi.authentication.manageable;

import weblogic.security.spi.AuditMgmtEvent;
import weblogic.security.spi.AuditSeverity;

/**
 * The Manageable authenticator's management operation event implementation.
 *
 * The Manageable authenticator posts audit events when management
 * operations are invoked on its mbean (eg. when someone creates a user).
 *
 * This class represents these events.
 *
 * 
 * @author VHIT Security and Other Common Services (S&OCS)
 * @version 1.1.0.002
 */
class KaajeeManageableAuthenticatorManagementEvent implements AuditMgmtEvent
{
  private String    realm;
  private String    provider;
  private String    details;
  private Exception failureException;

  /**
   * Constructs a Manageable authenticator user
   * group management event.
   *
   * @param realm a String containing the provider's realm's name.
   *
   * @param provider a String containing the provider's name.
   *
   * @param details a String containing the specifics
   * of a management operation (that is, the operation name,
   * its input parameters (if there are any), and its return value
   * (if it returns a value and the operation succeeded).
   *
   * @param failureException an Exception containing the
   * reason why the management operation failed.  null
   * indicates that the operation succeeded.
   */
  /*package*/ KaajeeManageableAuthenticatorManagementEvent(String realm, String provider, String details, Exception failureException)
  {
    this.realm = realm;
    this.provider = provider;
    this.details = details;
    this.failureException = failureException;
  }

  /**
   * Get the exception associated with this event.
   *
   * @return an Exception containing the reason the
   * management operation failed.  null indicates that the
   * operation succeeded.
   */
  public Exception getFailureException()
  {
    return failureException;
  }
  
  /**
   * Gets this event's severity
   *
   * @return an AuditSeverity of either SUCCESS if the management operation succeeded
   * or FAILURE if the management operation failed.
   */ 
  public AuditSeverity getSeverity()
  {
    return (failureException == null) ? AuditSeverity.SUCCESS : AuditSeverity.FAILURE;
  }

  /**
   * Gets this event's type.
   *
   * @return a String containing the type of this AuditEvent.
   */
  public String getEventType()
  {
    return "KaajeeManageableAuthenticator User/Group Management Event";
  }

  /**
   * Gets the details of this event (that is, the operation
   * invoked, its input arguments, and return value if appropriate.
   *
   * @return a String containing the details of this audit event.
   */
  public String getDetails()
  {
    return details;
  }

  /**
   * Get the name of the realm of the provider that posted this event.
   *
   * @return a String containing the realm's name.
   */
  public String getRealm()
  {
    return realm;
  }

  /**
   * Get the name of the provider that posted this event.
   *
   * @return a String containing the provider's name.
   */
  public String getProvider()
  {
    return provider;
  }

  /**
   * Convert this event to a string form.
   *
   * Generic audit providers may use this form to log this event.
   *
   * Specific audit providers (that know about this event type) may
   * invoke the specific getters instead (ie. getSeverity, getEventType,
   * getFailureException, getDetails, getRealm, getProvider)
   * and invent their own string form.
   *
   * @return a String describing this audit event.
   */
  public String toString()
  {
    StringBuffer buf = new StringBuffer();
    buf.append("EventType:"   + getEventType() + "\n");
    buf.append("\tRealm: "    + realm          + "\n");
    buf.append("\tProvider: " + provider       + "\n");
    buf.append("\tDetails: "  + details        + "\n");
    buf.append("\tSeverity: " + getSeverity().getSeverityString());
    if (getFailureException() != null) {
      buf.append("\n\tFailureException:" + getFailureException());
    }
    return buf.toString();
  }
}
