// $Id: typeahead.js,v 1.5 2003/12/11 19:15:39 jdunning Exp $ 
/* 
    file typeahead.js
    
    Supports select control type-ahead.
    
    Some browsers (IE) don't implement type-ahead in select controls.  
    When the focus is on a select control and the user presses a
    character key, the first entry that starts with that character is
    selected.  If the user presses the same key again, the next entry
    is selected, and so on.  This is not scalable to large lists.
    
    With type-ahead, the selected entry is the first entry that starts
    with the string of characters already entered. 

    Although Mozilla implements type-ahead, its behavior leaves room
    for improvement.  For example, typing a space resets the type-ahead
    buffer, which makes selection of items containing spaces difficult
    (consider city names).
    
    This implementation offers a number of advanced features:

        - Performant to very large lists.  Binary search is used to
          locate matching elements.
        
        - Characters entered are displayed in the status window.

        - The Backspace key undoes previously entered characters
          (one at a time).
          
        - The Escape key restores the control to its original state.

        - The Up and Down arrow keys can be used to move the selection
          relative to the current selection.

        - The Space key auto-completes as much text as possible, stopping
          at word boundaries.

          Consider the following list:

              Akron
              Albany
              Albuquerque
              Alexander City
              Alexandria Home
              Alexandria VAMC
              Alexandria, LA
              Alice

          Assume the user has already entered 'Al'.  "Albany" is
          therefore selected.  The following table shows the effect of
          a sequence of keystrokes (space key responses are explained
          below:)

              Key       Selection           Typeahead Buffer
              ---       --------------      ----------------
              space     Albany              "AL"
              e         Alexander City      "ALE"
              space     Alexander City      "ALEXAND"
              r         Alexandria          "ALEXANDR"
              space     Alexandria Home     "ALEXANDRIA"
              space     Alexandria Home     "ALEXANDRIA "
              v         Alexandria VAMC     "ALEXANDRIA V"

          The first space has no effect, since there is no entry 
          "Al " and the set of choices that begin with the characters
          "Al" have no common sequence of subsequent characters.

          The second space auto-completes the common sequence of
          characters shared by all choices that start with "Ale".
          
          The third space auto-completes through the end of the current
          word.
          
          The fourth space inserts a space, since there is a choice
          that has a space at that character offset.

        - Key matching: If the typeahead value doesn't match any option
          text, but it does match one or more option values (i.e.
          keys), option values are searched instead of option text.  
          This is useful when users have memorized object keys and they
          are shorter or more easily distinguished than object names.
          Auto-completion is not available during key matching.

    Requirements:
    
    - Typeahead is only suitable for single-selection controls.

    - This implementation requires the text content of selection
      control options to be in ascending order.  (Key matching does not
      require option values to be in ascending order.)

    Use:
    
    1.  Source this script within the head element of the page.
    
            <head>
              ...
              <script language="JavaScript" src="scripts/typeahead.js" 
                  type="text/javascript">
              </script>
              ...
            </head>

    2.  For each selection list to have typeahead, add an onkeyup event
        handler that calls the typeahead method, passing 'this' and the
        event:
        
            <select name="country"
                onkeyup="return typeahead(event)">
            
        Note that this must be the 'onkeyup' event, not 'onkeydown'
        or 'onkeypress'.  This is because select event handlers are 
        installed on the 'onkeyup' event.  This may irritate typers
        who overlap keypresses, but they would have this problem
        anyway.
        
    Limitations:
    
    - This implementation does not support international character sets.

    - Typeahead is not case-sensitive (browsers appear to force keys to
      uppercase or numeric values within select boxes)

    - Typeahead doesn't work for special characters only reachable
      by using the shift key like '(' and ':'. (for the same reason)

    Tested in Mozilla 1.3, IE 6.0, and Opera 7.20.
*/

/*
    A convenience function.
*/
function typeahead_stringStartsWithIgnoreCase(s, lookingFor)
{
    return s.toUpperCase().indexOf(lookingFor.toUpperCase()) == 0;
}

/*
    A convenience function.
*/
function typeahead_stringContains(s, lookingFor)
{
    return s.indexOf(lookingFor) >= 0;
}


/*
    A convenience function.
*/
function typeahead_stringContainsIgnoreCase(s, lookingFor)
{
    return s.toUpperCase().indexOf(lookingFor.toUpperCase()) >= 0;
}

/*
    When the backspace key is pressed in a select control,
    ignore it.  This method must be installed as an event handler for
    the body element.  It is only needed in IE, which treats backspace
    as a back button shortcut.
*/
function typeahead_cancelBackspaceInSelect()
{
    if ( event.keyCode == 8 
        && event.srcElement.tagName.toUpperCase() == "SELECT" )
    {
        event.cancelBubble = true;
        event.returnValue  = false;
    }
}

/*
    Install an event override function for the body element.
    This function installs an event handler for 'eventName' that
    invokes the function named 'overrideName' (note this is a name, not
    a function).  If an event handler already exists, it adds a call to
    the supplied override function in the event handler.
    
    param eventName 
        the name of the event to be overridden
    
    param overrideName 
        the name of the function to be called; this function
*/
function typeahead_installGlobalEventOverride(eventName, overrideName)
{
    var body = document.all ? document.all.tags("body")[0]
                            : document.getElementsByTagName("body")[0];

    var f = body[eventName];
    if ( f == null )
    {
        // no event handler for this event
        body[eventName] = 
            new Function(overrideName + "(); return true;");
    }
    else
    if ( typeahead_stringContains(f.toString(), overrideName) )
    {
        // there's an event handler for this event, 
        // but the desired override hasn't been installed

        var functionBody = f.toString();
        var p = functionBody.indexOf('{');
        functionBody = functionBody.substring(p + 1);
        p = functionBody.lastIndexOf('}');
        functionBody = functionBody.substring(0, p);
        
        body[eventName] =
            new Function(
                overrideName + "();"
                + functionBody
                );
    }
}

/*
    The offset of the first select control option with a text (i.e.
    displayed) value that is equal to or greater than a supplied value.
    
    param selectOptions 
        the options array of the select control; text content must be
        in ascending order
    
    param sought 
        the string to be matched

    return
        The largest offset i such that, for every j < i, 
        selectOptions[j].text < sought.  
        If sought is greater than every selectionOption.text, 
        selectOptions.length
*/
function typeahead_lowerBoundIgnoreCase(selectOptions, sought)
{
    // adapted from JAL
    
    var first = 0;
    var length = selectOptions.length;

    var ucSought = sought.toUpperCase();
    var half;
    var middle;
    while ( length > 0 )
    {
        half = Math.floor(length / 2);
        middle = first + half;
        if ( selectOptions[middle].text.toUpperCase() < ucSought ) 
        {
            first = middle + 1;
            length -= half + 1;
        }
        else
        {
            length = half;
        }
    }

    return first;
}

/*
    The offset of the last select control option with a text (i.e.
    displayed) value that is equal to or greater than a supplied value.
    
    param selectOptions 
        the options array of the select control; text content must be
        in ascending order
    
    param sought 
        the string to be matched

    return
        The largest offset i such that, for every j < i, 
        sought >= selectOptions[j].text
*/
function typeahead_upperBoundIgnoreCase(selectOptions, sought)
{
    // adapted from JAL
    
    var first = 0;
    var length = selectOptions.length;

    var ucSought = sought.toUpperCase();
    var half;
    var middle;
    while ( length > 0 )
    {
        half = Math.floor(length / 2);
        middle = first + half;
        if ( ucSought < selectOptions[middle].text.toUpperCase() ) 
        {
            length = half;
        }
        else
        {
            first = middle + 1;
            length -= half + 1;
        }
    }

    return first;
}

/*
    A comparison function used for sorting option arrays.
    Orders in ascending order by text value.
*/
function typeahead_optionCompare(optionA, optionB)
{
    return optionA.text < optionB.text ? -1 
         : optionB.text < optionA.text ? 1
         : 0;
}

/*
    A convenience function to find the current typeahead selection.
    Slightly different behavior based on whether the typeahead is
    operating on option texts or option values.

    param selectControl
        the select control currently in focus
        
    param options
        a possibly null array of option values.  Each value
        must support the 'text' property.
        
    param typed
        the current value of the typeahead buffer.

    return 
        If options is null, the largest offset i such that, 
        for every j < i, selectControl[j].text < typed.  
        If options is non-null, the largest offset i such that,
        for every j < i, options[j].text < typed.  
*/
function typeahead_findSelection(selectControl, options, typed)
{
    var result;
    if ( options == null )
    {
        result = typeahead_lowerBoundIgnoreCase(selectControl.options, typed);
        if ( result >= selectControl.options.length )
        {
            result == -1;
        }
        else
        if ( ! typeahead_stringStartsWithIgnoreCase(
                selectControl.options[result].text, typed) )
        {
            result = -1;
        }
    }
    else
    {
        var i = typeahead_lowerBoundIgnoreCase(options, typed);
        if ( i >= options.length )
        {
            result = -1;
        }
        else
        if ( ! typeahead_stringStartsWithIgnoreCase(
                options[i].text, typed) )
        {
            result = -1;
        }
        else
        {
            var option = options[i];
            result = option.i;
            if ( ! typeahead_stringContainsIgnoreCase(
                    selectControl.options[result].text, option.text) )
            {
                selectControl.options[result].text += 
                    " [" + option.text + "]";
            }
        }
    }

    return result;
}

/*
    Convert an input key code to a character.
    This function is limited by the capabilities of select controls,
    which support only uppercase characters and special characters
    entered without the shift key.

    param code
        The keyCode obtained from an event.
        
    return 
        the character represented by the code, 
        or null if the character is not a character key code.
*/
function typeahead_convertFromKeyCode(code)
{
    return (
        code == 188 ? ','
      : code == 189 ? '-'
      : code == 190 ? '.'
      : code == 191 ? '/'
      : code == 222 ? "'"
      : code >= 32 && code < 122 ? String.fromCharCode(code)
      : null);
}

function Typeahead(selectControl)
{
    this.controlName = selectControl.name;
    this.typed = "";
    this.selectedIndex = this.originalSelectedIndex = 
        selectControl.selectedIndex;
    this.valueOptions = null;
}

/*
    A dictionary holding typeahead information for the currently
    active select control.
*/
var typeahead_ = null;

/*
    Select the first select option that matches the value typed so far
    while the focus has been in this select control.
    
    param selectControl 
        the select control currently in focus
    
    parm event 
        the keyboard event
*/
function typeahead(event) 
{
    var selectControl = event.target ? event.target : event.srcElement;

    if ( selectControl.options.length == 0 )
    {
        return false;
    }

    if ( typeahead_ == null
        || typeahead_.controlName != selectControl.name )
    {
        typeahead_ = new Typeahead(selectControl);

        // ensure backspace works as desired in IE
        // (in IE, backspace is equivalent to the back button)
        if ( typeahead_stringContains(
                navigator.appName, "Microsoft Internet Explorer")
            && ! typeahead_stringContains(navigator.userAgent, "Opera") )
        {
            typeahead_installGlobalEventOverride(
                "onkeydown", "typeahead_cancelBackspaceInSelect");
        }

        // when field loses focus, 
        // reset window status and typeahead state
        // and set the selection field index to the correct value
        selectControl.onblur = 
            function()
            {
                // Mozilla ignores the selectedIndex value when set 
                // during keyup event processing.  Fortunately, it
                // responds correctly here.

                if ( typeahead_ != null )
                {
                    selectControl.selectedIndex = 
                        typeahead_.selectedIndex;
    
                    window.status = '';
                    typeahead_ = null;
                }
            };
    }

    var keyCode = event.keyCode;
    var keyChar;
    if ( keyCode == 8 )
    {
        // backspace: remove last character and select match
        if ( typeahead_.typed != "" )
        {
            window.status = typeahead_.typed = 
                typeahead_.typed.substring(
                    0, typeahead_.typed.length - 1);
            var temp = 
                typeahead_findSelection(selectControl, 
                    typeahead_.valueOptions, typeahead_.typed);
            if ( temp == -1 )
            {
                temp = 0;
            }

            selectControl.selectedIndex = typeahead_.selectedIndex = temp;
        }
    }
    else
    if ( keyCode == 13 )
    {
        // enter: ensure select control has selection
        selectControl.selectedIndex = 
            typeahead_.selectedIndex;
    }
    if ( keyCode == 27 )
    {
        // escape: reset to original state
        selectControl.selectedIndex = 
            typeahead_.originalSelectedIndex;
        window.status = "";
        typeahead_ = null;
    }
    else
    if ( keyCode == 38 )
    {
        // up arrow: let browser handle it
        window.status = typeahead_.typed = "";
        typeahead_.valueOptions = null;
        return true;
    }
    else
    if ( keyCode == 40 )
    {
        // down arrow: let browser handle it
        window.status = typeahead_.typed = "";
        typeahead_.valueOptions = null;
        return true;
    }
    else
    if ( (keyChar = typeahead_convertFromKeyCode(keyCode)) == null )
    {
        // invalid character: ignore and let browser handle it
        return true;
    }
    else
    {
        // normal character: select first matching entry

        typeahead_.typed += keyChar;
        selectControl.selectedIndex = typeahead_findSelection(
            selectControl, typeahead_.valueOptions, typeahead_.typed);

        if ( selectControl.selectedIndex == -1 
            && typeahead_.valueOptions == null )
        {
            // no entry starts with the value entered so far...

            // Clear the selection.                
            selectControl.selectedIndex = -1;

            if ( typeahead_.typed.length == 1 
                || ! isNaN(typeahead_.typed) )
            {
                // first character failed to match any entry,
                // or current typeahead buffer doesn't match any entry
                // and typeahead string is numeric

                // check option values...
                
                // Option values are probably not in ascending order,
                // so make a partial copy of the options array
                // sufficient to search by option value.
                // Objects in this array have two properties:
                // the value of the option element value attribute, 
                // and the offset into the original options array.
                var options = new Array(selectControl.options.length);
                var i = options.length;
                while ( --i >= 0 )
                {
                    options[i] = new Object();
                    options[i].text = selectControl.options[i].value;
                    options[i].i = i;
                }

                options.sort(typeahead_optionCompare);

                var test = typeahead_findSelection(
                    selectControl, options, typeahead_.typed);

                if ( test >= 0 )
                {
                    // found a matching option value...

                    // store the option value array so it won't have to
                    // be recreated on the next keypress
                    typeahead_.valueOptions = options;

                    // select the associated option
                    selectControl.selectedIndex = options[test].i;
                }
            }
            else
            if ( keyCode == 32 )
            {
                // user entered a space and there's no matching entry
                // with a space at this offset:  auto-complete
                typeahead_.typed = typeahead_.typed.substring(
                    0, typeahead_.typed.length - 1);

                // get the range
                var iFirst = typeahead_lowerBoundIgnoreCase(
                    selectControl.options, typeahead_.typed);
                var first  = selectControl.options[iFirst].text;

                var iLast  = typeahead_upperBoundIgnoreCase(
                    selectControl.options, typeahead_.typed + "z") - 1;
                var last   = selectControl.options[iLast].text;

                // auto-complete all common leading characters in the 
                // first and last values
                var p = typeahead_.typed.length - 1;
                var c;
                var lim = Math.max(first.length, last.length);
                while ( ++p < lim 
                    && (c = first.charAt(p)) == last.charAt(p) 
                    && c >= "A")
                {
                    typeahead_.typed += c;
                }

                if ( c == " " && c == last.charAt(p) )
                {
                    typeahead_.typed += c;
                }

                selectControl.selectedIndex =
                    typeahead_findSelection(selectControl, 
                        typeahead_.valueOptions, typeahead_.typed);
            }
        }

        // save the selected index value for Mozilla    
        typeahead_.selectedIndex = 
            selectControl.selectedIndex;

        // display the typeahead buffer to the user
        window.status = typeahead_.typed;
    }

    // cancel event handling (at least theoretically)
    return false;
}
