// $Id: dependents.js,v 1.3 2005/06/21 18:25:20 kackley Exp $
/*
    file dependents.js
    
    Supports select control dependencies to allow the items displayed
    in a selection list to vary based on a selection made in a
    different selection list.

    Use:

    Examples below are for Country/State.
    
    1.  Source this script within the head element of the page.

            <head>
              ...
              <script language="JavaScript" src="scripts/dependents.js" 
                  type="text/javascript">
              </script>
              ...
            </head>

    2.  Place the controlling selection list before the controlled
        selection lists to encourage the user to complete that field
        first.

    3.  Define id attributes for both the controlling and dependent
        selection lists.

            <select name="country" id="country">
            ...
            <select name="state" id="state">

    4.  In the onchange event handler for the controlling selection
        list, call the notifyDependents method, passing the controlling
        selection list object:

            <select name="country" id="country"
                onchange="return notifyDependents(this)">

    5.  If you'd like a dependent field to completely vanish if there
        aren't any entries specifically dependent on the controlling
        field value, define an id attribute for the element that 
        contains both the field label and the field control.

            <tr id="stateContainer">
              <th>State</th>
              <td><select name="state" id="state">
                ...
              </td>
            </tr>
    
    6.  In the dependent selection list, include an option for every
        value of the depended-on selection list.  Give each dependent
        option a class attribute with a value identical to the 
        'value' attribute of the option in the controlling list for the
        depended-on object.
        
        Any options that don't have a class attribute or whose content
        is empty will always be displayed.  Options that have a 
        class attribute will only be displayed if one of their classes
        corresponds to the value of the controlling field.

            <select name="country" id="country"
                onchange="return notifyDependents(this)">
              ...
              <option value="c840">United States</option>
              ...
            </select>
            
            <select name="state" id="state">
              <option value=""></option>
              <option value="1" class="c840">Alabama</option>
              <option value="2" class="c840">Arkansas</option>
              ...
              <option value="58" class="c124">Alberta</option>
              <option value="59" class="c124">British Columbia</option>
              ...
            </select>

    7.  In the body element's onload handler, initialize the dependency
        relationship by calling the 'registerDependent' function, 
        passing the controlling field's id, the dependent field's id, 
        and, optionally, the dependent container's id.
        
            <body onload="registerDependent(
                'country', 'state', 'stateContainer'); return true;">
    
        For convenience, a callback function generator is provided 
        that can be invoked at the point where the dependent field
        appears in the page.  The 'dependentInitCallback' function
        returns a Function object that invokes the 'registerDependent'
        function.
        
            <script language="javascript"...>
                var initCallbacks = new Array();
                function init() 
                {
                    var i = -1;
                    while ( initCallbacks.length > ++i )
                    {
                        initCallbacks[i]();
                    }
                }
            </script>
            ...
            <body onload="init()">
            ...
            <script language="javascript"...>
                initCallbacks[initCallbacks.length] = 
                    dependentInitCallback(
                        'country', 'state', 'stateContainer');
            </script>
            <select name="state" id="state">
            ...
                
    Tested in Mozilla 1.3, IE 6.0, and Opera 7.20.

    See http://www.quirksmode.org/js/options.html for how to do 
    this with traditional Javascript objects instead of the DOM.
    
    See http://glish.com/articles/apple/rs_with_iframe.html for how to 
    do this by issuing a server request without reloading the page.
*/

var dependentRegistry_ = new Object();

/*
    Convenience function to obtain a Function object to invoke 
    registerDependent when the document has finished loading.
*/
function dependentInitCallback(
    controlId, dependentControlId, dependentContainer)
{
    return new Function("registerDependent('"
        + controlId + "','" 
        + dependentControlId + "','" 
        + dependentContainer + "')");
}

/*
    Register a select control as a dependent of another select control,
    and initialize the dependent control based on the initial value of
    the controlling control.
    
    This function must be called after the document has been loaded.
*/
function registerDependent(controlId, dependentControlId, dependentContainer)
{
    if ( dependentRegistry_[controlId] == null )
    {
        dependentRegistry_[controlId] = new Array();
    }

    var dependents = dependentRegistry_[controlId];

    var dependentEntry = new Object();
    dependentEntry.dependentControlId = dependentControlId;
    dependentEntry.dependentContainer = dependentContainer;

    dependents[dependents.length] = dependentEntry;

    notifyDependents_(controlId);
}

/**
    Notify the dependents of a controlling control that the value has
    changed. 

    This function is called in the onchange event of the controlling
    field. 
**/
function notifyDependents(control)
{
    return notifyDependents_(control.id);
}

function notifyDependents_(controlId)
{
    var dependents = dependentRegistry_[controlId];
    if ( dependents != null )
    {
        var dependent;
        var i = dependents.length;
        while ( --i >= 0 )
        {
            dependent = dependents[i];
            displayDependents_(
                document.getElementById(controlId), 
                dependent.dependentControlId,
                dependent.dependentContainer);
        }
    }

    return true;
}

/*
    Change the user's view of a dependent selection list to display
    only those entries that are dependent on the value in a controlling
    list.  If a dependent container is specified, and there are no
    dependent entries, the container is hidden.
    
    This function hides the original dependent list and creates a
    visible copy, removing all options except for those that have 
    either a class corresponding to the value of the controlling field
    or no class.

    This function is called in the onchange event of the controlling
    field. 
*/
function displayDependents_(
    control, dependentControlId, dependentContainerId)
{
    // Some browsers have problems if the original select element
    // ID is changed, so instead the label reference is retargeted 
    // to the inserted filter control.

    var filteredId = dependentControlId + "filtered";
    var filteredControl;

    var originalControl = document.getElementById(dependentControlId);
    var filteredControl = document.getElementById(filteredId);

    if ( filteredControl == null )
    {
        // Clone the original node (options will be inserted later)
        filteredControl = originalControl.cloneNode(false);

        // Remove the name from original control so it's not submitted
        originalControl.name = "";

        // Change the ID so it doesn't get in the way
        filteredControl.id = filteredId;

        // Change the label, if any, to refer to the ID of the 
        // filtered control.
        var labels = document.getElementsByTagName("label");
        var i = labels.length;
        while ( --i >= 0 )
        {
            if ( labels[i].getAttribute('for') == dependentControlId )
            {
                labels[i].setAttribute('for', filteredId);
                break;
            }
        }

        // display the filtered control; hide the original control
        filteredControl.style.display = "";
        originalControl.style.display = "none";
    }
    else
    {
        // subsequent times...

        // work around an IE bug:
        // if you clone an element and change the clone's id, 
        // then insert the clone in front of the original,
        // document.getElementById(originalId) returns the clone
        // in some cases.
    
        if ( originalControl == filteredControl )
        {
            originalControl = originalControl.nextSibling;
        }

        // temporarily remove the filtered node so IE can resize the
        // control
        filteredControl.parentNode.removeChild(filteredControl);

        // remove all the options
        var children = filteredControl.childNodes;
        var i = children.length;
        while ( --i >= 0 )
        {
            filteredControl.removeChild(children[i]);
        }
    }
    
    // Add appropriate options to the filtered control

    var controlClass = control.options[control.selectedIndex].className;
    var children = originalControl.childNodes;
    var child;
    var childClass;
    var empty = true;
    var lim = children.length;
    var i = -1;
    while ( ++i < lim )
    {
        child = children[i];
        childClass = child.className;
        if ( ((childClass == null) || (childClass == ''))
            && (child.value == '') // leave the null option in
            )
        {
            filteredControl.appendChild(child.cloneNode(true));
        }
        else
        if ( child.nodeName.toUpperCase() == 'OPTION' )
        {
            // for efficiency, scan first before creating an object
            if (   ((childClass != '') && (childClass != null))
                && ((controlClass != '') && (controlClass != null))
                && (   (childClass.indexOf(controlClass) >= 0)
                    || (controlClass.indexOf(childClass) >= 0)
                    )
                )
            {
                empty = false;
                filteredControl.appendChild(child.cloneNode(true));
            }
        }
    }

    // When filteredControl is constructed via filteredControl.appendChild the selectedIndex 
    // setting is retained for Mozilla Firefox and is not for Internet Explorer. 

    if (filteredControl.selectedIndex == 0)
    {
        // check through the options for a matching select 
        var i = filteredControl.length;
        while ( --i >= 0 )
        // if the select index matches then alter filteredControl selectedIndex 
        if (filteredControl[i].value == originalControl[originalControl.selectedIndex].value 
            && filteredControl[i].text == originalControl[originalControl.selectedIndex].text)
        {
           filteredControl.selectedIndex = i;
        }
     }


    // if a dependent container has been supplied, and the filter has
    // removed all the options, hide the container
    if ( dependentContainerId != null )
    {
        var hideControl = document.getElementById(dependentContainerId);

        if ( hideControl != null )
        {
            hideControl.style.display = empty ? "none" : "";
        }
    }

    // insert the filtered control next to the original control
    originalControl.parentNode.insertBefore(
        filteredControl, originalControl);

    return true;
}
