package gov.va.med.esr.voa.validator;

import gov.va.med.esr.common.model.lookup.EmploymentStatus;
import gov.va.med.esr.common.util.AbstractCommonTestCase;
import gov.va.med.esr.voa.webservice.AddressInfo;
import gov.va.med.esr.voa.webservice.EmploymentInfo;
import gov.va.med.esr.voa.webservice.ObjectFactory;

import org.springframework.validation.BindException;

public class EmploymentInfoValidatorTest extends AbstractCommonTestCase {

	private EmploymentInfoValidator validator;

	private EmploymentInfo employmentInfo;

	@Override
	public void onSetUpInTransaction() throws Exception {
		super.onSetUpInTransaction();
		validator = new EmploymentInfoValidator();
		validator.setLookupService(getLookupService());

		ObjectFactory factory = new ObjectFactory();
		employmentInfo = factory.createEmploymentInfo();

		employmentInfo.setEmploymentStatus(EmploymentStatus.CODE_FULL_TIME
				.getCode());

		employmentInfo.setEmployerName("test");
		employmentInfo.setEmployerPhone("2121212222");
		employmentInfo.setRetirementDate("01/01/1999");

		AddressInfo addressInfo = factory.createAddressInfo();
		employmentInfo.setEmployerAddress(addressInfo);

		addressInfo.setLine1("line1");
		addressInfo.setLine2("line2");
		addressInfo.setLine3("line3");
		addressInfo.setCity("Dallas");
		addressInfo.setState("TX");
		addressInfo.setZipCode("75024");
		addressInfo.setPhoneNumber("1234567890");
		addressInfo.setCountry("USA");
	}

	@Override
	public void onTearDownInTransaction() {
		super.onTearDownInTransaction();
		validator = null;
		employmentInfo = null;
	}

	public void testInsuranceInfo() throws Exception {
		BindException errors = new BindException(employmentInfo,
				"employmentInfo");
		validator.validate(employmentInfo, errors);

		assertFalse(errors.hasErrors());
	}

	public void testEmptyEmployer() throws Exception {
		employmentInfo.setEmployerName(null);
		BindException errors = new BindException(employmentInfo,
				"employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 1);
	}

	public void testEmploymentStatus() throws Exception {
		employmentInfo.setEmploymentStatus(null);
		BindException errors = new BindException(employmentInfo,
				"employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 1);
	}

	public void testInvalidEmploymentStatus() throws Exception {
		employmentInfo.setEmploymentStatus("99999999999");
		BindException errors = new BindException(employmentInfo,
				"employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 1);
	}

	public void testRetirment() throws Exception {
		employmentInfo.setEmploymentStatus(EmploymentStatus.CODE_RETIRED
				.getCode());
		employmentInfo.setRetirementDate(null);
		BindException errors = new BindException(employmentInfo,
				"employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 1);

		employmentInfo.setRetirementDate("12345");
		errors = new BindException(employmentInfo, "employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 1);

		employmentInfo.setRetirementDate("01/01/2020");
		errors = new BindException(employmentInfo, "employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 1);
	}

	public void testFulltime() throws Exception {
		employmentInfo.setEmploymentStatus(EmploymentStatus.CODE_FULL_TIME
				.getCode());
		employmentInfo.setEmployerName(null);
		BindException errors = new BindException(employmentInfo,
				"employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 1);

		employmentInfo.setEmployerName("test");
		employmentInfo.setEmployerAddress(null);
		errors = new BindException(employmentInfo, "employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 1);

		ObjectFactory factory = new ObjectFactory();
		AddressInfo addressInfo = factory.createAddressInfo();
		employmentInfo.setEmployerAddress(addressInfo);

		addressInfo.setLine1("line1 + 123456789k01234567890123456789012345");
		addressInfo.setZipCode("123");
		addressInfo.setCountry("USA");

		errors = new BindException(employmentInfo, "employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 4);

		addressInfo.setCity("Dallas");
		addressInfo.setState("invalid");

		errors = new BindException(employmentInfo, "employmentInfo");
		validator.validate(employmentInfo, errors);

		assertEquals(errors.getErrorCount(), 3);
	}
}
