package gov.va.med.esr.voa.validator;

import gov.va.med.esr.voa.common.ErrorCode;
import gov.va.med.esr.voa.util.ServiceUtils;
import gov.va.med.esr.voa.webservice.SpouseInfo;
import gov.va.med.esr.voa.webservice.SsnCollection;
import gov.va.med.esr.voa.webservice.SsnInfo;
import gov.va.med.fw.util.StringUtils;

import java.util.List;

import org.springframework.validation.Errors;
import org.springframework.validation.ValidationUtils;

/**
 * Validate Spouse information
 */
public class SpouseInfoValidator extends AbstractValidator {

	@SuppressWarnings("rawtypes")
	@Override
	public boolean supports(Class clazz) {
		return SpouseInfo.class.equals(clazz);
	}

	@Override
	public void validate(Object target, Errors errors) {
		logger.debug("Spouse info validation...");

		if (target != null) {
			SpouseInfo spouseInfo = (SpouseInfo) target;

			String lastName = spouseInfo.getFamilyName();
			if (StringUtils.isBlank(lastName)) {
				errors.reject(ErrorCode.VOA_0113,
						"Missing required field:  Spouse last name.");
			} else {
				boolean isValid = ServiceUtils.hasLessThan30Chars(lastName);
				if (!isValid) {
					errors.reject(ErrorCode.VOA_0114,
							"Invalid Spouse last name.");
				}
			}

			String firstName = spouseInfo.getGivenName();
			if (StringUtils.isBlank(firstName)) {
				ValidationUtils.rejectIfEmpty(errors, "givenName",
						ErrorCode.VOA_0115,
						"Missing required field:  Spouse first name.");
			} else {
				boolean isValid = ServiceUtils.hasLessThan30Chars(firstName);
				if (!isValid) {
					errors.reject(ErrorCode.VOA_0116,
							"Invalid Spouse first name.");
				}
			}

			String middleName = spouseInfo.getMiddleName();
			if (StringUtils.isNotBlank(middleName)) {
				boolean isValid = ServiceUtils.hasLessThan30Chars(middleName);
				if (!isValid) {
					errors.reject(ErrorCode.VOA_0117,
							"Invalid Spouse middle name.");
				}
			}

			//CCR 13311 remove spouse maiden name

			SsnCollection ssnCollection = spouseInfo.getSsns();
			if (ssnCollection == null) {
				errors.reject(ErrorCode.VOA_0120,
						"Missing required field:  Spouse Social Security Number.");
			} else {
				List<SsnInfo> ssnList = ssnCollection.getSsn();
				if (ssnList.isEmpty()) {
					errors.reject(ErrorCode.VOA_0120,
							"Missing required field:  Spouse Social Security Number.");
				} else {
					for (SsnInfo ssnInfo : ssnList) {
						String ssnTxt = ssnInfo.getSsnText();
						boolean isValid = ServiceUtils.validateSsn(ssnTxt);
						if (!isValid) {
							errors.reject(ErrorCode.VOA_0121,
									"Invalid value for Spouse's Social Security Number.");
							break;
						}
					}
				}
			}

			String dob = spouseInfo.getDob();
			if (StringUtils.isBlank(dob)) {
				ValidationUtils.rejectIfEmpty(errors, "dob",
						ErrorCode.VOA_0122,
						"Missing required field:  Spouse's Date of Birth.");
			} else {
				boolean isValid = ServiceUtils.validateDate(dob);
				if (!isValid) {
					errors.reject(ErrorCode.VOA_0123,
							"Invalid Spouse date of birth.");
				} else {
					boolean isFutureDate = ServiceUtils.isFutureDate(dob);
					if (isFutureDate) {
						errors.reject(ErrorCode.VOA_0123,
								"Invalid Spouse date of birth.");
					}
				}
			}

			String startDate = spouseInfo.getStartDate();
			if (StringUtils.isNotBlank(startDate)) {
				boolean isValidDate = ServiceUtils.validateDate(startDate);
				boolean isValidYear = ServiceUtils.isValidYear(startDate);
				if (isValidDate || isValidYear) {
					boolean isFutureDate = false;
					if (isValidDate) {
						isFutureDate = ServiceUtils.isFutureDate(startDate);
					} else if (isValidYear) {
						isFutureDate = ServiceUtils.isFutureYear(startDate);
					}
					if (isFutureDate) {
						errors.reject(ErrorCode.VOA_0124,
								"Yr required if Marr/Sep, or spouse entered.");
						logger.debug("Yr required if Marr/Sep, or spouse entered.");
					}
				} else {
					errors.reject(ErrorCode.VOA_0124,
							"Yr required if Marr/Sep, or spouse entered.");
				}
			}
		}

		logger.debug("Spouse info validation - Done");
	}

}
