package gov.va.med.esr.voa.validator;

import gov.va.med.esr.voa.common.ErrorCode;
import gov.va.med.esr.voa.util.ServiceUtils;
import gov.va.med.esr.voa.webservice.MilitaryServiceEpisodeInfo;
import gov.va.med.fw.util.StringUtils;

import org.springframework.validation.Errors;
import org.springframework.validation.ValidationUtils;

/**
 * Validate MilitaryServiceEpisode Information
 */
public class MilitaryServiceEpisodeInfoValidator extends AbstractValidator {

	@SuppressWarnings("rawtypes")
	@Override
	public boolean supports(Class clazz) {
		return MilitaryServiceEpisodeInfo.class.equals(clazz);
	}

	@Override
	public void validate(Object target, Errors errors) {
		logger.debug("MilitaryServiceEpisodeInfo validation...");

		if (target != null) {
			MilitaryServiceEpisodeInfo militaryServiceEpisodeInfo = (MilitaryServiceEpisodeInfo) target;

			String serviceBranch = militaryServiceEpisodeInfo
					.getServiceBranch();
			if (StringUtils.isBlank(serviceBranch)) {
				errors.reject(ErrorCode.VOA_0103,
						"Missing Required Field:  Last branch of service.");
			} else {
				// validate by reference data
				try {
					getLookupService().getServiceBranchByCode(serviceBranch);
				} catch (Exception e) {
					errors.reject(ErrorCode.VOA_0104,
							"Invalid Branch of service entered.");
					logger.debug("Invalid Branch of service entered. "
							+ e.getMessage());
				}
			}

			String startDate = militaryServiceEpisodeInfo.getStartDate();
			if (StringUtils.isBlank(startDate)) {
				ValidationUtils.rejectIfEmpty(errors, "startDate",
						ErrorCode.VOA_0105,
						"Missing Required Field:  Last service entry date.");
			} else {
				boolean isValidDate = ServiceUtils.validateDate(startDate);
				boolean isValidMonthYear = ServiceUtils
						.isValidMonthYear(startDate);
				if (isValidDate || isValidMonthYear) {
					boolean isFutureDate = false;
					if (isValidDate) {
						isFutureDate = ServiceUtils.isFutureDate(startDate);
					} else if (isValidMonthYear) {
						isFutureDate = ServiceUtils
								.isFutureMonthYear(startDate);
					}
					if (isFutureDate) {
						errors.reject(ErrorCode.VOA_0106,
								"Invalid format for Last Service Entry Date.");
						logger.debug("Invalid format for Last Service Entry Date.");
					}
				} else {
					errors.reject(ErrorCode.VOA_0106,
							"Invalid format for Last Service Entry Date.");
					logger.debug("Invalid format for Last Service Entry Date.");
				}
			}

			String endDate = militaryServiceEpisodeInfo.getEndDate();
			if (StringUtils.isBlank(endDate)) {
				ValidationUtils
						.rejectIfEmpty(errors, "endDate", ErrorCode.VOA_0107,
								"Missing Required Field:  Last service discharge date.");
			} else {
				boolean isValidDate = ServiceUtils.validateDate(endDate);
				boolean isValidMonthYear = ServiceUtils
						.isValidMonthYear(endDate);

				if (isValidDate || isValidMonthYear) {
					boolean isFutureDate = false;
					if (isValidDate) {
						isFutureDate = ServiceUtils.isFutureDate(endDate);
					} else if (isValidMonthYear) {
						isFutureDate = ServiceUtils.isFutureMonthYear(endDate);
					}
					if (isFutureDate) {
						errors.reject(ErrorCode.VOA_0108,
								"Invalid format for Last Service Discharge Date.");
						logger.debug("Invalid format for Last Service Discharge Date.");
					}
				} else {
					errors.reject(ErrorCode.VOA_0108,
							"Invalid format for Last Service Discharge Date.");
					logger.debug("Invalid format for Last Service Discharge Date.");
				}
			}

			String dischargeType = militaryServiceEpisodeInfo
					.getDischargeType();
			if (StringUtils.isBlank(dischargeType)) {
				errors.reject(ErrorCode.VOA_0109,
						"Missing Required Field: Type of Discharge.");
				logger.debug("Missing Required Field: Type of Discharge."
						+ dischargeType);
			} else {
				try {
					getLookupService().getDischargeTypeByCode(dischargeType);
				} catch (Exception e) {
					errors.reject(ErrorCode.VOA_0110,
							"Invalid discharge type value.");
					logger.debug("Invalid discharge type value. "
							+ e.getMessage());
				}
			}

			//CCR 13311 remove military service number
		}

		logger.debug("MilitaryServiceEpisodeInfo validation - Done.");
	}
}
