/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.ui.util;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import gov.va.med.esr.common.infra.ImpreciseDate;
import gov.va.med.esr.common.infra.TriState;
import gov.va.med.esr.common.model.CommonEntityKeyFactory;
import gov.va.med.esr.common.model.ee.AgentOrangeExposure;
import gov.va.med.esr.common.model.ee.Application;
import gov.va.med.esr.common.model.ee.CDCondition;
import gov.va.med.esr.common.model.ee.CDDiagnosis;
import gov.va.med.esr.common.model.ee.CDProcedure;
import gov.va.med.esr.common.model.ee.CancelDecline;
import gov.va.med.esr.common.model.ee.CatastrophicDisability;
import gov.va.med.esr.common.model.ee.ConflictExperience;
import gov.va.med.esr.common.model.ee.EGTSetting;
import gov.va.med.esr.common.model.ee.Eligibility;
import gov.va.med.esr.common.model.ee.EligibilityVerification;
import gov.va.med.esr.common.model.ee.EnrollmentDetermination;
import gov.va.med.esr.common.model.ee.EnrollmentOverride;
import gov.va.med.esr.common.model.ee.EnvironmentalContaminationExposure;
import gov.va.med.esr.common.model.ee.FeeBasis;
import gov.va.med.esr.common.model.ee.IncompetenceRuling;
import gov.va.med.esr.common.model.ee.IneligibilityFactor;
import gov.va.med.esr.common.model.ee.MedicaidFactor;
import gov.va.med.esr.common.model.ee.MilitaryService;
import gov.va.med.esr.common.model.ee.MilitaryServiceEpisode;
import gov.va.med.esr.common.model.ee.MilitaryServiceSiteRecord;
import gov.va.med.esr.common.model.ee.MilitarySexualTrauma;
import gov.va.med.esr.common.model.ee.MonetaryBenefit;
import gov.va.med.esr.common.model.ee.MonetaryBenefitAward;
import gov.va.med.esr.common.model.ee.NoseThroatRadium;
import gov.va.med.esr.common.model.ee.POWEpisode;
import gov.va.med.esr.common.model.ee.PrisonerOfWar;
import gov.va.med.esr.common.model.ee.PurpleHeart;
import gov.va.med.esr.common.model.ee.RadiationExposure;
import gov.va.med.esr.common.model.ee.RatedDisability;
import gov.va.med.esr.common.model.ee.ReceivedEligibility;
import gov.va.med.esr.common.model.ee.ReceivedEnrollment;
import gov.va.med.esr.common.model.ee.SHAD;
import gov.va.med.esr.common.model.ee.ServiceConnectionAward;
import gov.va.med.esr.common.model.ee.VerificationInfo;
import gov.va.med.esr.common.model.financials.BeneficiaryTravel;
import gov.va.med.esr.common.model.financials.Debt;
import gov.va.med.esr.common.model.financials.DependentFinancials;
import gov.va.med.esr.common.model.financials.FinancialStatement;
import gov.va.med.esr.common.model.financials.Hardship;
import gov.va.med.esr.common.model.financials.Income;
import gov.va.med.esr.common.model.financials.IncomeTest;
import gov.va.med.esr.common.model.financials.IncomeTestStatus;
import gov.va.med.esr.common.model.financials.PatientVisitSummary;
import gov.va.med.esr.common.model.financials.SpouseFinancials;
import gov.va.med.esr.common.model.insurance.InsurancePlan;
import gov.va.med.esr.common.model.insurance.InsurancePolicy;
import gov.va.med.esr.common.model.insurance.InsuranceSubscriber;
import gov.va.med.esr.common.model.insurance.Medicare;
import gov.va.med.esr.common.model.insurance.PrivateInsurance;
import gov.va.med.esr.common.model.lookup.AddressChangeSource;
import gov.va.med.esr.common.model.lookup.AddressType;
import gov.va.med.esr.common.model.lookup.AffectedExtremity;
import gov.va.med.esr.common.model.lookup.AgentOrangeExposureLocation;
import gov.va.med.esr.common.model.lookup.AssociationType;
import gov.va.med.esr.common.model.lookup.BadAddressReason;
import gov.va.med.esr.common.model.lookup.CDDeterminationMethod;
import gov.va.med.esr.common.model.lookup.CancelDeclineNotProcessedReason;
import gov.va.med.esr.common.model.lookup.CancelDeclineReason;
import gov.va.med.esr.common.model.lookup.Condition;
import gov.va.med.esr.common.model.lookup.ConflictLocation;
import gov.va.med.esr.common.model.lookup.Country;
import gov.va.med.esr.common.model.lookup.DecorationStatus;
import gov.va.med.esr.common.model.lookup.Diagnosis;
import gov.va.med.esr.common.model.lookup.Disability;
import gov.va.med.esr.common.model.lookup.DischargeType;
import gov.va.med.esr.common.model.lookup.EligibilityFactor;
import gov.va.med.esr.common.model.lookup.EligibilityStatus;
import gov.va.med.esr.common.model.lookup.EligibilityType;
import gov.va.med.esr.common.model.lookup.EligibilityVerificationSource;
import gov.va.med.esr.common.model.lookup.EmailSourceOfChange;
import gov.va.med.esr.common.model.lookup.EmailType;
import gov.va.med.esr.common.model.lookup.EnrollmentCategory;
import gov.va.med.esr.common.model.lookup.EnrollmentPriorityGroup;
import gov.va.med.esr.common.model.lookup.EnrollmentSource;
import gov.va.med.esr.common.model.lookup.EnrollmentStatus;
import gov.va.med.esr.common.model.lookup.FeeBasisProgram;
import gov.va.med.esr.common.model.lookup.FeeBasisTreatmentType;
import gov.va.med.esr.common.model.lookup.FilipinoVeteranProof;
import gov.va.med.esr.common.model.lookup.Gender;
import gov.va.med.esr.common.model.lookup.IncomeTestType;
import gov.va.med.esr.common.model.lookup.IncomeType;
import gov.va.med.esr.common.model.lookup.InsurancePlanType;
import gov.va.med.esr.common.model.lookup.InsuredRelationship;
import gov.va.med.esr.common.model.lookup.MeansTestStatus;
import gov.va.med.esr.common.model.lookup.MilitarySexualTraumaStatus;
import gov.va.med.esr.common.model.lookup.MonetaryBenefitType;
import gov.va.med.esr.common.model.lookup.NTRVerificationMethod;
import gov.va.med.esr.common.model.lookup.NameType;
import gov.va.med.esr.common.model.lookup.POWCampLocation;
import gov.va.med.esr.common.model.lookup.POWDataSource;
import gov.va.med.esr.common.model.lookup.PhoneSourceOfChange;
import gov.va.med.esr.common.model.lookup.PhoneType;
import gov.va.med.esr.common.model.lookup.PrisonerOfWarLocation;
import gov.va.med.esr.common.model.lookup.Procedure;
import gov.va.med.esr.common.model.lookup.RadiationExposureMethod;
import gov.va.med.esr.common.model.lookup.SSNType;
import gov.va.med.esr.common.model.lookup.ServiceBranch;
import gov.va.med.esr.common.model.lookup.ServicePeriod;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.common.model.party.Address;
import gov.va.med.esr.common.model.party.Email;
import gov.va.med.esr.common.model.party.Phone;
import gov.va.med.esr.common.model.person.Association;
import gov.va.med.esr.common.model.person.BirthRecord;
import gov.va.med.esr.common.model.person.DeathRecord;
import gov.va.med.esr.common.model.person.Dependent;
import gov.va.med.esr.common.model.person.Name;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.SSN;
import gov.va.med.esr.common.model.person.SignatureImage;
import gov.va.med.esr.common.model.person.Spouse;
import gov.va.med.esr.common.model.person.id.PersonIdEntityKey;
import gov.va.med.esr.service.EGTService;
import gov.va.med.esr.service.LookupService;
import gov.va.med.esr.service.PersonHelperService;
import gov.va.med.esr.service.PersonService;
import gov.va.med.esr.service.impl.ChangeEvent;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.UserPrincipalImpl;
import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ServiceConfigurationException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.ps.model.PersonVPID;

/**
 * @author DNS   KATIKM
 */
public class MockPersonBuilder extends AbstractComponent {
		
	private static final Log log = LogFactory.getLog(MockPersonBuilder.class);
	
	public MockPersonBuilder()
    {
    }
	
	public Person getPerson (String personId) throws Exception
	{
		//get the person from the database with the given id
		Person person = null;

        if (personId == null)
        {
            log.info("No person Id specified so returning mock person.");
        }
        else
        {
            try
            {
                if (personIdValid(personId))
                {
                    PersonIdEntityKey personKey = CommonEntityKeyFactory.createPersonIdEntityKey(personId);
                    person = getPersonService().getPerson(personKey);
                }
            }
            catch (ServiceException se)
            {
                log.error("Could not retrive the person with the id " + personId, se);
            }
        }

        // If we weren't able to retrieve a person or if a person Id wasn't specified, get the mock person.
		if (person == null)
        {
			try
            {
				person = buildPerson();
				MockPersonBuilder.setIdentifier(person, new BigDecimal(12345));
			}
            catch (Exception ex)
            {
				throw new ServiceOutageException ("Unable to build Person Object",ex);
			}
		}
		else 
		{
			// Set additional required data if needed
			setAdditionalData(person);
		}
		return person;
	}
	
	public boolean personIdValid(String personId)
    {
		try
        {
			Integer.parseInt(personId);
		}
		catch (NumberFormatException nfe)
        {
			log.debug("Invalid Person Id: '" + personId + "'.", nfe);
            return false;
		}
        return true;
	}
	
	public Person getSimplePerson () throws Exception
	{
		return buildSimplePerson();
	}
	
	/**
	 * build and create person in database 
	 * @return
	 * @throws Exception
	 */
	public Person createPersonInDB() throws Exception {		
		disableSetIdentifier();		
		Person person = buildPerson();
		personService.save(person);			
		enableSetIdentifier();
		return person;
	}
	
	
	
	//included from abstractcommontestcase
	   private LookupService lookupService = null;
	   private PersonHelperService personHelperService = null;
	   private PersonService personService = null;
       private EGTService egtService = null;

	   protected Calendar createCalendar(int year) {
	      Calendar calendar = Calendar.getInstance();
	      calendar.clear();
	      calendar.set( Calendar.YEAR, year );
	      return calendar;
	   }

	   protected Calendar createCalendar(int year, int month) {
	      Calendar calendar = this.createCalendar( year );
	      calendar.set( Calendar.MONTH, month - 1 );
	      return calendar;
	   }

	   protected Calendar createCalendar(int year, int month, int day) {
	      Calendar calendar = this.createCalendar( year, month );
	      calendar.set( Calendar.DAY_OF_MONTH, day );
	      return calendar;
	   }

	   protected Calendar createCalendar(int year, int month, int day, int hour) {
	      Calendar calendar = this.createCalendar( year, month, day );
	      calendar.set( Calendar.HOUR_OF_DAY, hour );
	      return calendar;
	   }

	   protected Calendar createCalendar(int year, int month, int day, int hour,
	            int minute) {
	      Calendar calendar = this.createCalendar( year, month, day, hour );
	      calendar.set( Calendar.MINUTE, minute );
	      return calendar;
	   }

	   protected Calendar createCalendar(int year, int month, int day, int hour,
	            int minute, int second) {
	      Calendar calendar = this.createCalendar( year, month, day, hour, minute );
	      calendar.set( Calendar.SECOND, second );
	      return calendar;
	   }

	   protected Calendar createCalendar(int year, int month, int day, int hour,
	            int minute, int second, int millisecond) {
	      Calendar calendar = this.createCalendar( year, month, day, hour, minute,
	               second );
	      calendar.set( Calendar.MILLISECOND, millisecond );
	      return calendar;
	   }

	   protected ImpreciseDate createImpreciseDate(int year) {
	      return new ImpreciseDate( createCalendar( year ) );
	   }

	   protected ImpreciseDate createImpreciseDate(int year, int month) {
	      return new ImpreciseDate( createCalendar( year, month ) );
	   }

	   protected ImpreciseDate createImpreciseDate(int year, int month, int day) {
	      return new ImpreciseDate( createCalendar( year, month, day ) );
	   }

	   protected ImpreciseDate createImpreciseDate(int year, int month, int day,
	            int hour) {
	      return new ImpreciseDate( createCalendar( year, month, day, hour ) );
	   }

	   protected ImpreciseDate createImpreciseDate(int year, int month, int day,
	            int hour, int minute) {
	      return new ImpreciseDate( createCalendar( year, month, day, hour, minute ) );
	   }

	   protected ImpreciseDate createImpreciseDate(int year, int month, int day,
	            int hour, int minute, int second) {
	      return new ImpreciseDate( createCalendar( year, month, day, hour, minute,
	               second ) );
	   }

	   protected ImpreciseDate createImpreciseDate(int year, int month, int day,
	            int hour, int minute, int second, int millisecond) {
	      return new ImpreciseDate( createCalendar( year, month, day, hour, minute,
	               second, millisecond ) );
	   }

	   /*
	    * Since Messaging defines a bean of type PersonService also, then autowire
	    * by type will not work (when extended and used by Messaging). Therefore, we
	    * will set this lazily in getter method... public void
	    * setPersonService(PersonService personService) { this.personService =
	    * personService; }
	    */



	   /**
	    * 
	    * @return Generate an unique person id
	    */
	   protected Serializable generatePersonID() throws Exception {
	      //return getPersonService().getGenerator().generate();
	   		return null;
	   }

		/**
		 * Builds a basic person with some default settings.
		 * 
		 * @return
		 * @throws Exception
		 */
		protected Person buildPerson() throws Exception  
		{
			Person person = this.buildSimplePerson();
			
			ServiceConnectionAward serviceConnectionAward = new ServiceConnectionAward();	
			serviceConnectionAward.setServiceConnectedPercentage(new Integer(60));
			serviceConnectionAward.setCombinedServiceConnectedPercentageEffectiveDate(createImpreciseDate(2004, 5, 5));
			serviceConnectionAward.addRatedDisability( this.createRatedDisability(
                Disability.CODE_RESPIRATORY_CONDITION_GENERAL.getName(), new Integer(60)  ) );
			serviceConnectionAward.setUnemployable(Boolean.TRUE);
			serviceConnectionAward.setPermanentAndTotal(Boolean.TRUE);
			serviceConnectionAward.setPermanentAndTotalEffectiveDate(createCalendar(2000, 2, 1).getTime());
			
			
			person.setServiceConnectionAward( serviceConnectionAward );
			
			// Birth record
			person.setBirthRecord( this.createBirthRecord( createImpreciseDate(1969, 1, 1) ) );
			
			//set death record
			person.setDeathRecord(this.createDeathRecord());

			
			// Add some income test information
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_MEANS_TEST.getName(),
                MeansTestStatus.MT_STATUS_EXEMPT_LTC_COPAY_EXEMPTION_TEST.getName(), new Integer(2004) );
			incomeTest.setSiteConductingTest(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
			incomeTest.setEffectiveDate(this.createCalendar(1995, 2, 1).getTime());
			person.setIncomeTest(new Integer(2004), incomeTest );
			
			//Add Signature
			attachSignature(person);

			// Monetary Benefit Award
			MonetaryBenefitAward mba = new MonetaryBenefitAward();
			mba.setCheckAmount( new BigDecimal( 20 ) );
			mba.addMonetaryBenefit( this.createMonetaryBenefit(MonetaryBenefitType.CODE_HOUSEBOUND.getName()));
			mba.addMonetaryBenefit( this.createMonetaryBenefit(MonetaryBenefitType.CODE_AID_AND_ATTENDANCE.getName()));
			mba.addMonetaryBenefit( this.createMonetaryBenefit(MonetaryBenefitType.CODE_DISABILITY_COMPENSATION.getName()));
			mba.addMonetaryBenefit( this.createMonetaryBenefit(MonetaryBenefitType.CODE_VA_PENSION.getName()));

			person.setMonetaryBenefitAward( mba );
			
			// Special Factor
			person.addSpecialFactor( this.createAgentOrange() );
			
			//enrollment
			person.setEnrollmentDetermination(createEnrollmentDetermination());
			person.setCancelDecline( createCancelDecline() );
			
			EnrollmentOverride override = new EnrollmentOverride();
			override.setOverride(Boolean.TRUE);
			override.setOverrideReason(null);
			override.setOverrideComment("Test Comment");
			person.setEnrollmentOverride(override);
			
			attachCatastrophicDisability(person);
			attachMilitarySexualTrauma(person);
			attachNoseThroatRadium(person);
			
			attachPOW(person);
			//attachPOW(person);
			attachPurpleHeartWithPHStatusPending(person);
			//attachPurpleHeartWithPHRejectRemark(person);
			
			attachRatedIncompetent(person);
			attachIneligibleFactor(person);
			attachMedicaidFactor(person);
			
			//eligibility verification
			person.setEligibilityVerification(createEligibilityVerification());
			setAACIndicator("2", person); //AAC Queried, Pending
			
			//Application
			attachApplication(person);
			
			//Address
			person.addAddress(createPrimaryAddress());
			person.addAddress(createTemporayCorrespondanceAddress());
			person.addAddress(createConfidentialAddress());			
			
			//Phone
			person.addPhone(createHomePhone(getNextId()));
			person.addPhone(createBusinessPhone(getNextId()));
			
			//Email
			person.addEmail(createPrivateEmail(getNextId()));
			person.addEmail(createBusinessEmail(getNextId()));
			
			
			//Association
			attachAssociation(person);
			
			attachAgentOrangeExposure(person);
			attachEnvironmentalContaminationExposure(person); 
			attachRadiationExposure(person);
			attachEnvironmentalContaminationExposure(person);
			attachRadiationExposure(person);
			 
			//Non-Veteran, should not be with veteran
			attachTriCare(person);
			attachOtherFederalAgency(person);			
			   
			attachMilitaryService(person);
			
			//Preferred Facility
			attachPreferredFacility(person);
			
			//attach patient visit summaries and Fee based services info
			attachPatientVisitSummaries(person);
			
			//attach insurance
			attachInsurance(person);
			
			//attach financials
			attachFinancialInfo(person, new Integer(2005));
			attachFinancialInfo(person, new Integer(2003));

			return person;
		}
		
		/**
		 * Customize this to add additional data onto the person retrieved from DB
		 * @param person
		 * @throws Exception
		 */
		protected void setAdditionalData(Person person) throws Exception
		{
		}

		// Build a "bare-bones" person
		protected  Person buildSimplePerson() throws Exception  {
			Person person = new Person();
			person.addName(createName( "TEST", "PERSON", NameType.LEGAL_NAME.getName()));
			person.addSsn(createSSN("456010001",SSNType.CODE_ACTIVE.getName()));
			person.setBirthRecord( this.createBirthRecord(createImpreciseDate(1950, 1, 1)));
			person.setGender( lookupService.getGenderByCode(Gender.MALE.getName()));
			//set VPID
            PersonVPID.initializeVPID(person,"12345678");//Andrew Check ??
			person.setSensitiveRecord(Boolean.TRUE);
			person.setVeteran(Boolean.TRUE);
			return person;
		}
		
		protected SSN createSSN(String text, String ssnTypeCode) throws Exception{
		    SSN ssn = new SSN();
		    ssn.setSsnText(text);
		    ssn.setType(lookupService.getSSNTypeByCode(ssnTypeCode));
		    return ssn;
		}
		
		protected  IncomeTest createIncomeTest(String testType, String status, Integer incomeYear ) throws Exception {		
			IncomeTest incomeTest = new IncomeTest();
			incomeTest.setIncomeYear( incomeYear );
			incomeTest.setType( lookupService.getIncomeTestTypeByCode( testType ) );
            incomeTest.setIncomeVerificationStatus(lookupService.getMeansTestStatusByCode(
                MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED.getName()));
            incomeTest.setGmtThresholdAmount(new BigDecimal(1000));
            incomeTest.setThresholdA(new BigDecimal(2000));
            incomeTest.setAgreesToPayDeductible(Boolean.TRUE);
            incomeTest.setDiscloseFinancialInformation(Boolean.FALSE);
            incomeTest.setTotalNumberOfDependents(new Integer(0));

            IncomeTestStatus incomeTestStatus = new IncomeTestStatus();
			incomeTestStatus.setStatus( lookupService.getMeansTestStatusByCode( status ) );
			incomeTestStatus.setType( lookupService.getIncomeTestTypeByCode( testType ) );
            incomeTestStatus.setCompletedDate(new Date());
			incomeTest.addStatus( incomeTestStatus );

			return incomeTest;
		}
		
		protected  ReceivedEligibility createReceivedEligibility( String type ) throws Exception {
			ReceivedEligibility receivedEligibility = new ReceivedEligibility();
			receivedEligibility.setType( lookupService.getEligibilityTypeByCode( type ) );	

			return receivedEligibility;
		}	
		
		protected  void setServiceConnectedPercentage( Integer percentage, Person person ) {
			ServiceConnectionAward scw = person.getServiceConnectionAward();
			scw.setServiceConnectedPercentage( percentage );
		}
		
		protected MonetaryBenefit createMonetaryBenefit( String type ) throws Exception {
			
			MonetaryBenefit monetaryBenefit = new MonetaryBenefit();
			monetaryBenefit.setType( lookupService.getMonetaryBenefitTypeByCode( type ) );
			return monetaryBenefit;		
		}
		protected CDCondition createCDCondition() throws Exception
		{
			CDCondition condition = new CDCondition();
			condition.setCondition(lookupService.getConditionByCode(Condition.CODE_FIM.getName()));
			//condition.setPermanent(Boolean.FALSE);
			condition.setScore(new Integer(45));
			
			return condition;
		}
		protected CDDiagnosis createCDDiagnosis() throws Exception
		{
			CDDiagnosis diagnosis = new CDDiagnosis();
			diagnosis.setDiagnosis(lookupService.getDiagnosisByCode(Diagnosis.CODE_QUADRIPLEGIA_C1_C4_COMPLETE.getName()));
			
			return diagnosis;
		}
		protected CDProcedure createCDProcedure() throws Exception
		{
			CDProcedure procedure = new CDProcedure();
			procedure.setAffectedExtremity(lookupService.getAffectedExtremityByCode(AffectedExtremity.CODE_RIGHT_UPPER.getName()));
			procedure.setProcedure(lookupService.getProcedureByCode(Procedure.CODE_TOE_AMPUTATION.getName()));
			return procedure;
		}
		protected CatastrophicDisability createCatastrophicDisability(Date reviewDate, Date determinationDate, String cdDeterminationType, String determineBy)throws Exception {
			CatastrophicDisability catastrophicDisability = new CatastrophicDisability();

			catastrophicDisability.setDecisionDate(determinationDate);
			catastrophicDisability.setReviewDate(reviewDate);
			catastrophicDisability.setDeterminedBy(determineBy);
			catastrophicDisability.setVeteranRequestedCDReviewDate(createCalendar(2004, 5, 5).getTime());
			catastrophicDisability.setFacilityInitiatedCDReviewDate(createCalendar(2004, 5, 9).getTime());
			catastrophicDisability.setVeteranWasNotifiedCDDate(createCalendar(2004, 6, 9).getTime());
	 
			CDDeterminationMethod cdDeterminationMethod = lookupService.getCdDeterminationMethodByCode(cdDeterminationType);
			catastrophicDisability.setDeterminationMethod(cdDeterminationMethod);
			catastrophicDisability.setDeterminationFacility(getVaFacility(VAFacility.CODE_NEW_BEDFORD_CBOC.getName()));
			catastrophicDisability.setCatastrophicallyDisabled(Boolean.TRUE);
			catastrophicDisability.addCondition(createCDCondition());
			catastrophicDisability.addDiagnosis(createCDDiagnosis());
			catastrophicDisability.addProcedure(createCDProcedure());
			
			return catastrophicDisability;
		}
		
		protected Name createName( String familyName, String givenName, String type ) throws Exception {
			Name name = new Name();
			NameType nameType = lookupService.getNameTypeByCode( type );
			name.setFamilyName(  familyName );
			name.setGivenName( givenName );
			name.setType( nameType );
		    return name;
		}
	
		protected RatedDisability createRatedDisability( String type, Integer percent ) throws Exception {
			RatedDisability ratedDisability = new RatedDisability();		
			ratedDisability.setPercentage(new Integer(60));
			ratedDisability.setServiceConnected(Boolean.TRUE);
			ratedDisability.setDisability( lookupService.getDisabilityByCode( type ) );
			ratedDisability.setDiagnosticExtremity(null);
			ratedDisability.setOriginalEffectiveRatingDate(createImpreciseDate(1999, 12, 12));
			ratedDisability.setRatingDate(createImpreciseDate(1999, 12, 20));
			return ratedDisability;	    
		}
		
		protected BirthRecord createBirthRecord( ImpreciseDate date ) {
			BirthRecord birthRecord = new BirthRecord();		
			birthRecord.setBirthDate(date);
		    return birthRecord;
		}

		protected AgentOrangeExposure createAgentOrange() throws Exception {
			AgentOrangeExposure agentOrangeExposure = new AgentOrangeExposure();
			agentOrangeExposure.setLocation( lookupService.getAgentOrangeExposureLocationByCode(
                AgentOrangeExposureLocation.CODE_VIETNAM.getName()));
		    
		    return agentOrangeExposure;
		}
		
		protected EnvironmentalContaminationExposure createEnvironmentalContaminationExposure() throws Exception {
	 
		    return new EnvironmentalContaminationExposure();
		}	
		
		protected RadiationExposure createRadiationExposure() throws Exception {
		    RadiationExposure radiationExposure = new RadiationExposure();
		    radiationExposure.setExposureMethod( lookupService.getRadiationExposureMethodByCode(
                RadiationExposureMethod.CODE_ATMOSPHERIC_NUCLEAR_TESTING.getName()));
		    return radiationExposure;
		}	
		
		
		protected void setAACIndicator( String code, Person person ) throws Exception {
		   this.personHelperService.setAACIndicator( code, person );
		}
		protected void setEligibilityStatus( String status, Person person ) throws Exception {
		   this.personHelperService.setEligibilityStatus( status, person );
		}
		
		protected void setEligibilityStatusToVerified( Person person ) throws Exception {
		    setEligibilityStatus( EligibilityStatus.CODE_VERIFIED.getName(), person );
		}
		
		protected void setEligibilityStatusToPendingVerification( Person person ) throws Exception {
            setEligibilityStatus( EligibilityStatus.CODE_PENDING_VERIFICATION.getName(), person );
		}
		
		protected void setEligibilityStatusToPendingReverification( Person person ) throws Exception {
            setEligibilityStatus( EligibilityStatus.CODE_PENDING_REVERIFICATION.getName(), person );
		}
			

		protected String getEligibilityStatusCode( Person person ) throws Exception {
		    EligibilityStatus status =  this.personHelperService.getEligibilityStatus(person);
		    return status != null ? status.getCode() : null;
		}	
		
		
		protected void setEnrollmentStatus( String status, Person person ) throws Exception {
		   this.personHelperService.setEnrollmentStatus( status, person );
		}	
		
		protected String getEnrollmentCategoryCode( Person person ) throws Exception {
		    EnrollmentCategory status = this.personHelperService.getEnrollmentCategory(person); 
			return (status != null) ? status.getCode() : null ;   
		}	
		
		protected String getEnrollmentStatusCode( Person person ) throws Exception {
		    EnrollmentStatus status = this.personHelperService.getEnrollmentStatus( person );
			return (status != null) ? status.getCode() : null ;   
		}		
		
		protected void setPurpleHeartStatus( String status, Person person ) throws Exception {
		   this.personHelperService.setPurpleHeartStatus( status, person );
		}
		
		protected String getPurpleHeartStatusCode(Person person) throws Exception {
	        DecorationStatus decorationStatus = this.personHelperService
	                .getPurpleHeartStatus(person);
	        return (decorationStatus != null) ? decorationStatus.getCode() : null;
	    }		

		protected void assessIfHasPrimaryRanking( String type, Person person) {
		    
		    assertTrue( this.hasPrimaryEligibility( type, person ) );
		    assertFalse( this.hasSecondaryEligibility( type, person ) );
		    assertFalse( this.hasOtherEligibility( type, person ) );	    	    
		}	
		
		protected void assessIfHasOtherRanking( String type, Person person) {
		    
		    assertFalse( this.hasPrimaryEligibility( type, person ) );
		    assertFalse( this.hasSecondaryEligibility( type, person ) );
		    assertTrue( this.hasOtherEligibility( type, person ) );		    	    
		}	
		
		protected void assessIfHasSecondaryRanking( String type, Person person) {
		    
		    assertFalse( this.hasPrimaryEligibility( type, person ) );
		    assertTrue( this.hasSecondaryEligibility( type, person ) );
		    assertFalse( this.hasOtherEligibility( type, person ) );		    	    
		}	
		
		protected void attachMeansTestWithMtCopayRequired( Person person,
		        BigDecimal gmtThresholdAmount,
		        BigDecimal thresholdA ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_MEANS_TEST.getName(),
			        MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED.getName(), new Integer(2004) );	
		    incomeTest.setGmtThresholdAmount( gmtThresholdAmount );
		    incomeTest.setThresholdA( thresholdA );		
			person.setIncomeTest(new Integer(2004), incomeTest );
		}
		
		protected void attachMeansTestWithNoLongerRequired( Person person ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_MEANS_TEST.getName(),
			        MeansTestStatus.MT_STATUS_NO_LONGER_REQUIRED.getName(), new Integer(2004) );	
			person.setIncomeTest(new Integer(2004), incomeTest );
		}	
		
		protected void attachMeansTestWithRequired( Person person ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_MEANS_TEST.getName(),
			        MeansTestStatus.MT_STATUS_REQUIRED.getName(), new Integer(2004) );	
			person.setIncomeTest(new Integer(2004), incomeTest );
		}
		
		protected void attachMeansTestWithMtCopayExempt( Person person ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(
			        IncomeTestType.CODE_MEANS_TEST.getName(),
			        MeansTestStatus.MT_STATUS_MT_COPAY_EXEMPT.getName(), new Integer(2004) );	
			person.setIncomeTest(new Integer(2004), incomeTest );
		}	
		
		protected void attachPharmacyCopayTestWithRXCopayExempt( Person person ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_CO_PAY_EXEMPTION_TEST.getName(),
			        MeansTestStatus.MT_STATUS_EXEMPT.getName(), new Integer(2004) );	
			person.setIncomeTest(new Integer(2004), incomeTest );
		}	
		
		protected void attachPharmacyCopayTestWithNonExempt( Person person ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_CO_PAY_EXEMPTION_TEST.getName(),
			        MeansTestStatus.MT_STATUS_NON_EXEMPT.getName(), new Integer(2004) );	
			person.setIncomeTest(new Integer(2004), incomeTest );
		}		

		protected void attachPharmacyCopayTestWithNoLongerApplicable( Person person ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_CO_PAY_EXEMPTION_TEST.getName(),
			        MeansTestStatus.MT_STATUS_NP_LONGER_APPLICABLE_RX_COPAY_ONLY.getName(), new Integer(2004) );	
			person.setIncomeTest(new Integer(2004), incomeTest );
		}		

		protected void attachLongTermCareTestWithExempt( Person person ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_LTC_CO_PAY_EXEMPTION_TEST.getName(),
			        MeansTestStatus.MT_STATUS_EXEMPT.getName(), new Integer(2004) );	
			person.setIncomeTest(new Integer(2004), incomeTest );
		}			

		protected void attachLongTermCareTestWithNonExempt( Person person ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_LTC_CO_PAY_EXEMPTION_TEST.getName(),
			        MeansTestStatus.MT_STATUS_NON_EXEMPT.getName(), new Integer(2004) );	
			person.setIncomeTest(new Integer(2004), incomeTest );
		}		
		
		protected void attachMeansTestWithGmtCopayRequired( Person person,
		        BigDecimal gmtThresholdAmount,
		        BigDecimal thresholdA ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_MEANS_TEST.getName(),
			        MeansTestStatus.MT_STATUS_GMT_COPAY_REQUIRED.getName(), new Integer(2004) );	
		    incomeTest.setGmtThresholdAmount( gmtThresholdAmount );
		    incomeTest.setThresholdA( thresholdA );		
			person.setIncomeTest(new Integer(2004), incomeTest );
		}	
		
		protected void attachMeansTestWithHardship( Person person ) throws Exception {
		    
			IncomeTest incomeTest = person.getIncomeTest(new Integer(2004));
			if ( incomeTest == null ){
			    incomeTest = this.createIncomeTest(IncomeTestType.CODE_MEANS_TEST.getName(),
			            MeansTestStatus.MT_STATUS_MT_COPAY_REQUIRED.getName(), new Integer(2004) );
				person.setIncomeTest(new Integer(2004), incomeTest );
			}
			Hardship hardship = new Hardship();
			hardship.setEffectiveDate( new Date() );
			hardship.setHardshipGranted( Boolean.TRUE );
			incomeTest.setHardship( hardship );
		}	
		
		protected void attachMeansTestWithMtStatusPendingAdjudication( Person person,
		        BigDecimal gmtThresholdAmount,
		        BigDecimal thresholdA ) throws Exception {
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_MEANS_TEST.getName(),
			        MeansTestStatus.MT_STATUS_PENDING_ADJUDICATION.getName(), new Integer(2004) );	
		    incomeTest.setGmtThresholdAmount( gmtThresholdAmount );
		    incomeTest.setThresholdA( thresholdA );		
			person.setIncomeTest(new Integer(2004), incomeTest );
		}	
		
		protected void attachServiceConnectionAward( Person person, Integer percent ) {
		    this.attachServiceConnectionAward( person );
			person.getServiceConnectionAward().setServiceConnectedPercentage( percent );
		}
		
		protected void attachServiceConnectionAward( Person person ) {
			ServiceConnectionAward serviceConnectionAward = person.getServiceConnectionAward();
			if ( serviceConnectionAward == null ) {
			    serviceConnectionAward = new ServiceConnectionAward();
			    person.setServiceConnectionAward( serviceConnectionAward );
			}
		}
		
		protected void attachNoseThroatRadium( Person person ) throws Exception {
		    NoseThroatRadium noseThroatRadium =  createNoseThroatRadium( new Date(),
                VAFacility.CODE_CONNECTICUT_HCS.getName(), NTRVerificationMethod.CODE_MILITARY_RECORD.getName());
		    person.addClinicalDetermination( noseThroatRadium );	    
		}
		
		protected void attachVaDisabilityCompensation( Person person ) throws Exception {
		    MonetaryBenefit monetaryBenefit = createMonetaryBenefit(MonetaryBenefitType.CODE_DISABILITY_COMPENSATION.getName());
		    this.attachMonetaryBenefitAward( person );
		    person.getMonetaryBenefitAward().addMonetaryBenefit( monetaryBenefit );
		}
		
		protected void attachVaPension( Person person ) throws Exception {
		    MonetaryBenefit monetaryBenefit = createMonetaryBenefit(MonetaryBenefitType.CODE_VA_PENSION.getName());
		    this.attachMonetaryBenefitAward( person );
		    person.getMonetaryBenefitAward().addMonetaryBenefit( monetaryBenefit );
		}	
		
		protected void attachHousebound( Person person ) throws Exception {
		    MonetaryBenefit monetaryBenefit = createMonetaryBenefit( MonetaryBenefitType.CODE_HOUSEBOUND.getName() );
		    this.attachMonetaryBenefitAward( person );
		    person.getMonetaryBenefitAward().addMonetaryBenefit( monetaryBenefit );
		}
		
		protected void attachAidAndAttendance( Person person ) throws Exception {
		    MonetaryBenefit monetaryBenefit = createMonetaryBenefit( MonetaryBenefitType.CODE_AID_AND_ATTENDANCE.getName() );
		    this.attachMonetaryBenefitAward( person );
		    person.getMonetaryBenefitAward().addMonetaryBenefit( monetaryBenefit );
		}
		
		protected void attachMonetaryBenefitAward( Person person ) throws Exception {
		    MonetaryBenefitAward monetaryBenefitAward = person.getMonetaryBenefitAward();
		    if ( monetaryBenefitAward == null ) {
		        monetaryBenefitAward = new MonetaryBenefitAward();
		        person.setMonetaryBenefitAward( monetaryBenefitAward );
		    }
		}	
		
		protected void attachDischargeDueToDisability( Person person) 
		throws Exception
		{
		    attachMilitaryService( person );
		    person.getMilitaryService().setDischargeDueToDisability( Boolean.TRUE );   
		}
		
		protected void attachMilitaryRetirementInLieuOfVaCompensation( Person person ) throws Exception {
		    attachMilitaryService( person );
	       person.getMilitaryService().setDisabilityRetirementIndicator(Boolean.TRUE);
		}
		
		protected void attachMilitaryService(Person person)  throws Exception
		{
		    MilitaryService militaryService = person.getMilitaryService();
		    if(militaryService == null) 
		    {
		        militaryService = createMilitaryService();
		        person.setMilitaryService(militaryService);
		    }
		}
		
	   protected void attachReceivingMilitaryRetirement(Person person) throws Exception {

	       attachMilitaryService( person );
	       person.getMilitaryService().setDisabilityRetirementIndicator(Boolean.FALSE);
	   }
	   
	   protected void attachPermanentAndTotal( Person person ) {
	       this.attachServiceConnectionAward( person );
	       person.getServiceConnectionAward().setPermanentAndTotal( Boolean.TRUE );
	   }
	   
	   protected void attachUnemployable( Person person ) {
	       this.attachServiceConnectionAward( person );
	       person.getServiceConnectionAward().setUnemployable( Boolean.TRUE );
	   }
	   
	   protected void attachAgentOrangeExposure( Person person ) throws Exception {
	       person.addSpecialFactor( this.createAgentOrange() );
	   }
	   
	   protected void attachEnvironmentalContaminationExposure( Person person ) throws Exception {
	       person.addSpecialFactor( this.createEnvironmentalContaminationExposure() );
	   }    
	   
	   protected void attachRadiationExposure( Person person ) throws Exception {
	       person.addSpecialFactor( this.createRadiationExposure() );
	   }       
	   
	   protected void attachEmployee( Person person ) throws Exception {
			person.addReceivedSecondaryEligibility( createReceivedEligibility(EligibilityType.EMPLOYEE.getName()));
	   }
	   protected void attachEmployeeAsPrimary( Person person ) throws Exception {
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.EMPLOYEE.getName() ) );
	   }    
	   
	   protected void attachMexicanBorderWar( Person person ) throws Exception {
			person.setBirthRecord( createBirthRecord(createImpreciseDate(1906, 12, 31)) );        
			person.addReceivedSecondaryEligibility( createReceivedEligibility( EligibilityType.MEXICAN_BORDER_WAR.getName() ) );
	   }
	   
	   protected void attachMexicanBorderWarAsPrimary( Person person ) throws Exception {
			person.setBirthRecord( createBirthRecord(createImpreciseDate(1906, 12, 31)) );        
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.MEXICAN_BORDER_WAR.getName() ) );
	   }     
	   
	   protected void attachAlliedVeteran( Person person ) throws Exception {
			person.addReceivedSecondaryEligibility( createReceivedEligibility( EligibilityType.ALLIED_VETERAN.getName() ) );
	   }        
	   
	   protected void attachAlliedVeteranAsPrimary( Person person ) throws Exception {
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.ALLIED_VETERAN.getName() ) );
	   }    

	   protected void attachWWI( Person person ) throws Exception {
			person.setBirthRecord( createBirthRecord(createImpreciseDate(1907, 12, 31)) );        
			person.addReceivedSecondaryEligibility( createReceivedEligibility( EligibilityType.WORLD_WAR_I.getName() ) );
	   }

	   
	   protected void attachWWIAsPrimary( Person person ) throws Exception {
			person.setBirthRecord( createBirthRecord(createImpreciseDate(1907, 12, 31)) );        
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.WORLD_WAR_I.getName() ) );
	   } 
	   
	   protected void attachTriCare( Person person ) throws Exception {
			person.addReceivedSecondaryEligibility( createReceivedEligibility( EligibilityType.TRICARE_CHAMPUS.getName() ) );
	   }    
	   
	   protected void attachTriCareAsPrimary( Person person ) throws Exception {
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.TRICARE_CHAMPUS.getName() ) );
	   }    
		
	   protected void attachSharingAgreement( Person person ) throws Exception {
			person.addReceivedSecondaryEligibility( createReceivedEligibility( EligibilityType.SHARING_AGREEMENT.getName() ) );
	   }
	   
	   protected void attachSharingAgreementAsPrimary( Person person ) throws Exception {
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.SHARING_AGREEMENT.getName() ) );
	   }     
		
	   protected void attachChampVA( Person person ) throws Exception {
			person.addReceivedSecondaryEligibility( createReceivedEligibility( EligibilityType.CHAMPVA.getName() ) );
	   }
	   
	   protected void attachChampVAAsPrimary( Person person ) throws Exception {
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.CHAMPVA.getName() ) );
	   }     
		
	   protected void attachCollateralOfVeteran( Person person ) throws Exception {
			person.addReceivedSecondaryEligibility( createReceivedEligibility( EligibilityType.COLLATERAL_OF_VETERAN.getName() ) );
	   }
	   
	   protected void attachCollateralOfVeteranAsPrimary( Person person ) throws Exception {
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.COLLATERAL_OF_VETERAN.getName() ) );
	   }     
		
	   protected void attachOtherFederalAgency( Person person ) throws Exception {
			person.addReceivedSecondaryEligibility( createReceivedEligibility( EligibilityType.OTHER_FEDERAL_AGENCY.getName() ) );
	   }       
	   
	   protected void attachOtherFederalAgencyAsPrimary( Person person ) throws Exception {
			person.setReceivedPrimaryEligiblity( createReceivedEligibility( EligibilityType.OTHER_FEDERAL_AGENCY.getName() ) );
	   }    
	   
	   protected void attachIneligibleFactor( Person person ) throws Exception {
	       person.setIneligibilityFactor( createIneligibilityFactor() );            
	   }
	   
	   protected void attachIneligibleFactorFromSite( Person person ) throws Exception {
	       person.setIneligibilityFactor( createIneligibilityFactor() );
	       person.getIneligibilityFactor().setFacilityReceived( getVaFacility(VAFacility.CODE_NEW_BEDFORD_CBOC.getName()));
	   }
	   
	   protected IneligibilityFactor createIneligibilityFactor() throws Exception {
			Calendar calendar = Calendar.getInstance();
			calendar.set(1999,5,23);         
	       IneligibilityFactor ineligibilityFactor =  new IneligibilityFactor() ;
	       ineligibilityFactor.setIneligibleDate( calendar.getTime() );
	       ineligibilityFactor.setReason( "this person is ineligible");
	       ineligibilityFactor.setFacilityReceived( getVaFacility(VAFacility.CODE_HEC.getName()));
	       return ineligibilityFactor;
	   }
	   
	   protected void attachCatastrophicDisability( Person person ) throws Exception {
			Calendar calendar = Calendar.getInstance();			
			calendar.set(1983,5,15);
			Date reviewDate = calendar.getTime();        
			calendar.set(1983,7,25);
			Date dateDecision = calendar.getTime();		
			
			CatastrophicDisability catastrophicDisability = createCatastrophicDisability(reviewDate,
					dateDecision,
					CDDeterminationMethod.CODE_AUTOMATED_RECORD_REVIEW.getName(),
					"test reason");
			person.addClinicalDetermination( catastrophicDisability );
	   }
	   
	   protected void attachSHAD( Person person ) throws Exception {
	       person.setShad( new SHAD() );
	   }

	   protected void attachMedicaidFactor( Person person ) {
			person.setMedicaidFactor( this.createMedicaidFactor() );        
	   }
	   
	   protected MedicaidFactor createMedicaidFactor() {
			Calendar calendar = Calendar.getInstance();        
			MedicaidFactor medicaidFactor = new MedicaidFactor(); 
			medicaidFactor.setEligibleForMedicaid( Boolean.TRUE );
			calendar.set(2004,3,23);
			medicaidFactor.setLastUpdateDate( calendar.getTime() );			
	       
	       return medicaidFactor;
	   }
		protected MilitarySexualTrauma createMilitarySexualTrauma() throws Exception{
			Calendar calendar = Calendar.getInstance();
			calendar.set(2005,1,12);
			Date statusChangeDate = calendar.getTime();

			MilitarySexualTraumaStatus militarySexualTraumaStatus = 
			   lookupService.getMilitarySexualTraumaStatusByCode("Y");

			MilitarySexualTrauma militarySexualTrauma = new MilitarySexualTrauma();
			militarySexualTrauma.setStatus(militarySexualTraumaStatus);
			militarySexualTrauma.setDeterminationFacility(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
			calendar.set(1986,3,24);
			//militarySexualTrauma.//setDeterminationDate(calendar.getTime());
			militarySexualTrauma.setStatusChangeDate(statusChangeDate);
			return militarySexualTrauma;
		}    
		
		protected MilitarySexualTrauma createMilitarySexualTrauma(String status)
	           throws Exception {
	       MilitarySexualTrauma militarySexualTrauma = createMilitarySexualTrauma();
	       militarySexualTrauma.setStatus(lookupService
	               .getMilitarySexualTraumaStatusByCode(MilitarySexualTraumaStatus.CODE_SCREENED_REPORTS_MST.getName()));
	       return militarySexualTrauma;
	   }
		
		protected void attachMilitarySexualTrauma ( Person person ) throws Exception {
		    person.addClinicalDetermination( this.createMilitarySexualTrauma() );
		}
		
		protected ReceivedEnrollment createReceivedEnrollment() throws Exception {
			Calendar calendar = Calendar.getInstance();
			ReceivedEnrollment receivedEnrollment = new ReceivedEnrollment();
			EnrollmentSource enrollmentSource = 
			   lookupService.getEnrollmentSourceByCode(EnrollmentSource.CODE_HEC.getName());
			receivedEnrollment.setCalculationSource( enrollmentSource );		
			calendar.set(1986,1,1);
			receivedEnrollment.setEnrollmentDate(calendar.getTime());
			receivedEnrollment.setEffectiveDate(calendar.getTime());
			calendar.set(1987,1,1);
			receivedEnrollment.setEndDate(calendar.getTime());
			EnrollmentPriorityGroup enrollmentPriorityGroup = 
			   lookupService.getEnrollmentPriorityGroupByCode(EnrollmentPriorityGroup.GROUP_1.getName());
			receivedEnrollment.setPriorityGroup(enrollmentPriorityGroup);
			receivedEnrollment.setPrioritySubGroup(null);
			receivedEnrollment.setFacilityReceived(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
			return receivedEnrollment;
		}	
		
		protected void attachReceivedEnrollment( Person person ) throws Exception {
		    person.setReceivedEnrollment( this.createReceivedEnrollment() );
		}
		
		protected void attachPreferredFacility( Person person ) throws Exception {
		    person.setMostRecentPreferredFacility( getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
		}
		

		
		protected CancelDecline createCancelDecline() throws Exception {
		    CancelDecline cancelDecline = new CancelDecline();
		    cancelDecline.setCancelDeclineIndicator(Boolean.TRUE);
		    cancelDecline.setReason(lookupService.getCancelDeclineReasonByCode(
                CancelDeclineReason.CODE_OTHER.getName()));
		    cancelDecline.setReasonNotProcessed(lookupService.getCancelDeclineNotProcessedReasonByCode(
                CancelDeclineNotProcessedReason.CODE_PENDING_TELEPHONE_CALLS.getName()));
		    cancelDecline.setEffectiveDate( new Date() );
		    cancelDecline.setRemarks( "test cancel decline remarks");
		    return cancelDecline;
		}
		
		protected void attachCancelDecline ( Person person ) throws Exception {
		    person.setCancelDecline( this.createCancelDecline() );
		}

		protected IncompetenceRuling createIncompetenceRuling() {
			IncompetenceRuling incompetenceRuling = new IncompetenceRuling();
			ImpreciseDate rd = createImpreciseDate(1999, 1, 1);			
			incompetenceRuling.setCivilRulingDate( rd );
			return incompetenceRuling;
		}
		
	   protected void attachMilitaryServiceWithConflictExperience(Person person, Date expire)
	   throws Exception
	   {
	       	attachMilitaryService(person);
			person.getMilitaryService().setCombatVeteranEligibilityEndDate(expire);
			person.getMilitaryService().addMilitaryServiceSiteRecord(this.createMilitaryServiceSiteRecord());
	   }	
	   
	   protected MilitaryServiceSiteRecord createMilitaryServiceSiteRecord() throws Exception
	   {
	       	MilitaryServiceSiteRecord militaryServiceSiteRecord = new MilitaryServiceSiteRecord();
		    militaryServiceSiteRecord.setSite(getVaFacility(VAFacility.CODE_HEC.getName()));
			ConflictExperience conflictExperience = new ConflictExperience();
			// TODO set attributes on conflictExperience
			militaryServiceSiteRecord.addConflictExperience(conflictExperience);
			return militaryServiceSiteRecord;
	   }	
	   
		protected void attachIncompetenceRuling( Person person ) {
		    person.setIncompetenceRuling( this.createIncompetenceRuling() );
		}
		
		protected void attachRatedIncompetent( Person person ) 
		{
		    IncompetenceRuling incompetenceRuling = new IncompetenceRuling();
		    incompetenceRuling.setIncompetent(Boolean.TRUE);
		    incompetenceRuling.setCivilRulingDate(createImpreciseDate(1999, 1, 1));
			incompetenceRuling.setVaRulingDate(createImpreciseDate(1999, 2, 2));
		    person.setIncompetenceRuling(incompetenceRuling);
		}	
		
		protected void attachDeathRecord( Person person ) throws Exception {
		    person.setDeathRecord( this.createDeathRecord() );
		}
		
		protected void attachDeathRecordFromSite( Person person ) throws Exception {
		    person.setDeathRecord( this.createDeathRecord() );
		    person.getDeathRecord().setFacilityReceived( getVaFacility(VAFacility.CODE_NEW_BEDFORD_CBOC.getName()));
		}

		protected MilitaryService createMilitaryService() throws Exception
	 	{
		    MilitaryService militaryService = new MilitaryService();
		    
		    militaryService.setCombatVeteranEligibilityEndDate(new Date());
		    militaryService.setDischargeDueToDisability(Boolean.TRUE);	    
		    militaryService.setDisabilityRetirementIndicator(Boolean.TRUE);
		   
		    MilitaryServiceEpisode militaryServiceEpisode = null;
		    ConflictExperience conflictExperience = null;

		    ServiceBranch serviceBranch = null;
		    DischargeType dischargeType = null;
		    ImpreciseDate serviceEntryDate = null;
		    ImpreciseDate serviceSeparationDate = null;

		    MilitaryServiceSiteRecord militaryServiceSiteRecord = null;
		    
		    //Add a Site Record 1
		    militaryServiceSiteRecord = new MilitaryServiceSiteRecord();
		    setIdentifier(militaryServiceSiteRecord,getNextId());
		    militaryServiceSiteRecord.setSite(getVaFacility(VAFacility.CODE_MICHAEL_E_DEBAKEY_VA_MEDICAL_CENTER.getName()));
		    militaryServiceSiteRecord.setServicePeriod(
                getLookupService().getServicePeriodByCode(ServicePeriod.CODE_SPANISH_AMERICAN.getName()));
		    militaryServiceSiteRecord.setFilipinoVeteranProof(
                getLookupService().getFilipinoVeteranProofByCode(
                    FilipinoVeteranProof.CODE_VERIFICATION_OF_NATURALIZATION.getName()));
		    militaryService.addMilitaryServiceSiteRecord(militaryServiceSiteRecord);

		    serviceBranch = getLookupService().getServiceBranchByCode(ServiceBranch.CODE_NAVY.getName());
		    
		    dischargeType = getLookupService().getDischargeTypeByCode(DischargeType.CODE_GENERAL.getName());
		    serviceEntryDate = new ImpreciseDate("01/01/1980");
		    serviceSeparationDate = new ImpreciseDate("01/01/1985");
		    
		    militaryServiceEpisode = createMilitaryServiceEpisode("123-45-6789",serviceBranch,dischargeType,serviceEntryDate,serviceSeparationDate);
		    militaryServiceSiteRecord.addMilitaryServiceEpisode(militaryServiceEpisode);
		    
		    serviceBranch = getLookupService().getServiceBranchByCode(ServiceBranch.CODE_MARINE_CORPS.getName());
		    dischargeType = getLookupService().getDischargeTypeByCode(DischargeType.CODE_OTHER_THAN_HONORABLE.getName());
		    serviceEntryDate = new ImpreciseDate("01/01/1986");
		    serviceSeparationDate = new ImpreciseDate("01/01/1990");
		    
		    militaryServiceEpisode = createMilitaryServiceEpisode("123-45-6790",serviceBranch,dischargeType,serviceEntryDate,serviceSeparationDate);
		    militaryServiceSiteRecord.addMilitaryServiceEpisode(militaryServiceEpisode);
		    
		    conflictExperience = createConflictExperience(ConflictLocation.CODE_WORLD_WAR_II_PACIFIC.getName());
		    militaryServiceSiteRecord.addConflictExperience(conflictExperience);
		    
		    conflictExperience = createConflictExperience(ConflictLocation.CODE_KOREA.getName());
		    militaryServiceSiteRecord.addConflictExperience(conflictExperience);
		    
		    //Add a Site Record 2
		    militaryServiceSiteRecord = new MilitaryServiceSiteRecord();
		    setIdentifier(militaryServiceSiteRecord,getNextId());
		    militaryServiceSiteRecord.setSite(getVaFacility(VAFacility.CODE_IRON_MOUNTAIN_VA_MEDICAL_CENTER.getName()));
		    militaryServiceSiteRecord.setServicePeriod(
                getLookupService().getServicePeriodByCode(ServicePeriod.CODE_POST_KOREAN.getName()));
		    militaryServiceSiteRecord.setFilipinoVeteranProof(
                getLookupService().getFilipinoVeteranProofByCode(FilipinoVeteranProof.CODE_BIRTH_ABROAD_OF_US_CITIZEN.getName()));
		    militaryService.addMilitaryServiceSiteRecord(militaryServiceSiteRecord);

		    serviceBranch = getLookupService().getServiceBranchByCode(ServiceBranch.CODE_COAST_GUARD.getName());
		    dischargeType = getLookupService().getDischargeTypeByCode(DischargeType.CODE_UNDESIRABLE.getName());
		    serviceEntryDate = new ImpreciseDate("01/01/1991");
		    serviceSeparationDate = new ImpreciseDate("01/01/1992");
		    
		    militaryServiceEpisode = createMilitaryServiceEpisode("123-45-6791",serviceBranch,dischargeType,serviceEntryDate,serviceSeparationDate);
		    militaryServiceSiteRecord.addMilitaryServiceEpisode(militaryServiceEpisode);
		    
		    serviceBranch = getLookupService().getServiceBranchByCode(ServiceBranch.CODE_OTHER.getName());
		    dischargeType = getLookupService().getDischargeTypeByCode(DischargeType.CODE_BAD_CONDUCT.getName());
		    serviceEntryDate = new ImpreciseDate("01/01/1993");
		    serviceSeparationDate = new ImpreciseDate("01/01/1999");
		    
		    militaryServiceEpisode = createMilitaryServiceEpisode("123-45-6792",serviceBranch,dischargeType,serviceEntryDate,serviceSeparationDate);
		    militaryServiceSiteRecord.addMilitaryServiceEpisode(militaryServiceEpisode);
		    
		    conflictExperience = createConflictExperience(ConflictLocation.CODE_PERSIAN_GULF_WAR.getName());
		    militaryServiceSiteRecord.addConflictExperience(conflictExperience);
		    
		    conflictExperience = createConflictExperience(ConflictLocation.CODE_YUGOSLAVIA_CONFLICT.getName());
		    militaryServiceSiteRecord.addConflictExperience(conflictExperience);
		    
		    
		    //Add a HEC Record
		    militaryServiceSiteRecord = new MilitaryServiceSiteRecord();
		    setIdentifier(militaryServiceSiteRecord,getNextId());
		    militaryServiceSiteRecord.setSite(getVaFacility(VAFacility.CODE_HEC.getName()));
		    militaryServiceSiteRecord.setServicePeriod(
                getLookupService().getServicePeriodByCode(ServicePeriod.CODE_WORLD_WAR_I.getName()));
		    militaryServiceSiteRecord.setFilipinoVeteranProof((FilipinoVeteranProof)getLookupService().getByCode(FilipinoVeteranProof.class,"BC"));
		    militaryService.addMilitaryServiceSiteRecord(militaryServiceSiteRecord);
		    
		    serviceBranch = getLookupService().getServiceBranchByCode(ServiceBranch.CODE_ARMY.getName());
		    dischargeType = getLookupService().getDischargeTypeByCode(DischargeType.CODE_HONORABLE.getName());
		    serviceEntryDate = new ImpreciseDate("01/01/1970");
		    serviceSeparationDate = new ImpreciseDate("01/01/1975");
		    
		    militaryServiceEpisode = createMilitaryServiceEpisode("123-45-6793",serviceBranch,dischargeType,serviceEntryDate,serviceSeparationDate);
		    militaryServiceSiteRecord.addMilitaryServiceEpisode(militaryServiceEpisode);

            serviceBranch = getLookupService().getServiceBranchByCode(ServiceBranch.CODE_AIR_FORCE.getName());
             dischargeType = getLookupService().getDischargeTypeByCode(DischargeType.CODE_DISHONORABLE.getName());
		    serviceEntryDate = new ImpreciseDate("01/01/1976");
		    serviceSeparationDate = new ImpreciseDate("01/01/1999");
		    
		    militaryServiceEpisode = createMilitaryServiceEpisode("123-45-6794",serviceBranch,dischargeType,serviceEntryDate,serviceSeparationDate);
		    militaryServiceSiteRecord.addMilitaryServiceEpisode(militaryServiceEpisode);

            conflictExperience = createConflictExperience(ConflictLocation.CODE_VIETNAM.getName());
		    militaryServiceSiteRecord.addConflictExperience(conflictExperience);

            conflictExperience = createConflictExperience(ConflictLocation.CODE_OTHER.getName());
		    militaryServiceSiteRecord.addConflictExperience(conflictExperience);

		    return militaryService;
	 	}
	 	
	 	
	 	protected MilitaryServiceEpisode createMilitaryServiceEpisode(String serviceNumber, 
	 	        													  ServiceBranch serviceBranch, 
	 	        													  DischargeType dischargeType, 
	 	        													  ImpreciseDate startDate, 
	 	        													  ImpreciseDate endDate)
	 	throws Exception
	 	{
	 	    MilitaryServiceEpisode militaryServiceEpisode = new MilitaryServiceEpisode();
	 	    setIdentifier(militaryServiceEpisode,getNextId());
	 	    militaryServiceEpisode.setServiceNumber(serviceNumber);
	 	    militaryServiceEpisode.setServiceBranch(serviceBranch);
	 	    militaryServiceEpisode.setDischargeType(dischargeType);
	 	    
	 	    militaryServiceEpisode.setStartDate(startDate);
	 	    militaryServiceEpisode.setEndDate(endDate);

	 	    return militaryServiceEpisode;
	 	}
	 	
	 	protected ConflictExperience createConflictExperience(String conflictlocationCode)
	 	throws Exception
	 	{
	 	    ConflictExperience conflictExperience = new ConflictExperience();
	 	    setIdentifier(conflictExperience,getNextId());
	 	    ConflictLocation conflictLocation = getLookupService().getConflictByCode(conflictlocationCode);
	 	    if(conflictLocation != null)
	 	    {
	 	        conflictExperience.setConflictLocation(conflictLocation); 
	 	        conflictExperience.setStartDate(createImpreciseDate(1998, 7, 1));
	 	        conflictExperience.setEndDate(createImpreciseDate(1999, 7, 1));
	 	    }
	 	    return conflictExperience;
	 	}
	 	
	 	
	 	protected void attachAssociation(Person person)
	 	throws Exception
	 	{
	 	    //Spouse
	 	    AssociationType type = getLookupService().getAssociationTypeByCode(
                 AssociationType.CODE_PRIMARY_NEXT_OF_KIN.getName());
	 	    NameType nameType = getLookupService().getNameTypeByCode(NameType.LEGAL_NAME.getName());
	 	    AddressType addressType = getLookupService().getAddressTypeByCode(AddressType.CODE_PERMANENT_ADDRESS.getName());
	 	    String orgName = "Good Corp";
	 	    Name name =  new Name();
	 	    name.setFamilyName("LastName");
	 	    name.setGivenName("FirstName");
	 	    name.setMiddleName("K");
	 	    name.setPrefix("Mrs.");
	 	    name.setType(nameType);
	 	    person.addAssociation(createAssociation(type,"Spouse",orgName,name,addressType));
	 	    
	 	    //Mother
	 	    type = getLookupService().getAssociationTypeByCode(AssociationType.CODE_OTHER_NEXT_OF_KIN.getName());
	 	    name =  new Name();
	 	    name.setFamilyName("LastName");
	 	    name.setGivenName("FirstName");
	 	    name.setMiddleName("K");
	 	    name.setPrefix("Mrs.");
	 	    name.setType(nameType);
	 	    person.addAssociation(createAssociation(type,"Mother",orgName,name,addressType));
	 	    
	 	    //Father
            type = getLookupService().getAssociationTypeByCode(AssociationType.CODE_EMERGENCY_CONTACT.getName());
	 	    name =  new Name();
	 	    name.setFamilyName("LastName");
	 	    name.setGivenName("FirstName");
	 	    name.setMiddleName("K");
	 	    name.setPrefix("Mr.");
	 	    name.setSuffix("Jr.");
	 	    name.setType(nameType);
	 	    person.addAssociation(createAssociation(type,"Father",orgName,name,addressType));
	 	    
	 	    //Son
            type = getLookupService().getAssociationTypeByCode(AssociationType.CODE_POWER_OF_ATTORNEY.getName());
	 	    name =  new Name();
	 	    name.setFamilyName("LastName");
	 	    name.setGivenName("FirstName");
	 	    name.setMiddleName("K");
	 	    name.setPrefix("Mr.");
	 	    name.setSuffix("Jr.");
	 	    name.setType(nameType);
	 	    person.addAssociation(createAssociation(type,"Son",orgName,name,addressType));
	 	    
	 	    //Nephew
            type = getLookupService().getAssociationTypeByCode(AssociationType.CODE_GUARDIAN_VA.getName());
	 	    name =  new Name();
	 	    name.setFamilyName("LastName");
	 	    name.setGivenName("FirstName");
	 	    name.setMiddleName("K");
	 	    name.setType(nameType);
	 	    person.addAssociation(createAssociation(type,"Nephew",orgName,name,addressType));
	 	    
	 	    //Niece
            type = getLookupService().getAssociationTypeByCode(AssociationType.CODE_GUARDIAN_CIVIL.getName());
	 	    name =  new Name();
	 	    name.setFamilyName("LastName");
	 	    name.setGivenName("FirstName");
	 	    name.setMiddleName("K");
	 	    name.setPrefix("Ms.");
	 	    name.setType(nameType);
	 	    person.addAssociation(createAssociation(type,"Niece",orgName,name,addressType));
	 	}
	 	
	 	protected Association createAssociation(AssociationType type,String relationship,String orgName, Name name, AddressType addressType)
	 	{
	 	   Association association = new Association();
	 	   setIdentifier(association,getNextId());
	 	   association.setOrganizationName(orgName);
	 	   association.setRelationship(relationship);
	 	   association.setType(type);
	 	   
	 	   association.setRepresentativeName(name);
	 	   
	 	   Address address = new Address();
	 	   association.setAddress(address);
	 	   address.setType(addressType);
	 	   address.setLine1("1300 EDS Drive, Suite 300");
	 	   address.setLine2("3rd Floor, Quad B");
	 	   address.setLine3("Cube A3N-B13");
	 	   address.setCity("Herndon");
	 	   address.setCounty("Fairfax");
	 	   address.setCountry(Country.CODE_USA.getName());
	 	   address.setState("VA");
	 	   address.setPostalCode("20191-1234");
	 	   address.setZipCode("20191");
	 	   address.setZipPlus4("1234");
	 	   
	 	   association.setPrimaryPhone("(703)122-2324");
	 	   association.setPrimaryPhoneChangeDate(new java.sql.Date(System.currentTimeMillis()));
	 	   
	 	   association.setModifiedBy(new UserPrincipalImpl("Mock Builder"));
	 	   return association;
	 	}
	 	
		
		protected DeathRecord createDeathRecord() throws Exception {
			DeathRecord deathRecord = new DeathRecord(); 
			deathRecord.setDeathDate( createImpreciseDate(2000, 1, 1) );
			deathRecord.setFacilityReceived(getVaFacility(VAFacility.CODE_HEC.getName()));
			deathRecord.setDeathReportDate( new Date() );
			return deathRecord;
		}
		
		protected Application createApplication() {
			Application application = new Application();
			application.setApplicationDate( new Date() );
			return application;
		}
		protected void attachApplication( Person person ) {
		    person.setApplication( this.createApplication() );
		}
		
		protected EnrollmentDetermination getEnrollmentDetermination( Person person ) {
			return this.getPersonHelperService().getEnrollmentDetermination( person );	    
		}
		
		protected VerificationInfo createVerificationInfoFromORFZ11() {
		    return new VerificationInfo("0L", "ORFZ11-E", null );
		}
		
		protected void assessPriorityResult( Person person, String priority ) {
			// EnrollmentDetermination ed = getEnrollmentDetermination( person );
			//assertNotNull( "enrollment determination s/b non-null", ed );
			//assertNotNull( "enrollment priority group s/b non-null", ed.getPriorityGroup() );
			//assertEquals( ed.getPriorityGroup().getCode(), priority );			
		}
		
		protected void assessPrioritySubGroupResult( Person person, String subPriority ) {
			// EnrollmentDetermination ed = getEnrollmentDetermination( person );
			//assertNotNull( "enrollment determination s/b non-null", ed );
			//assertNotNull( "enrollment priority group s/b non-null", ed.getPrioritySubGroup() );
			//assertEquals( ed.getPrioritySubGroup().getCode(), subPriority );			
		}	
		
		protected void assessEnrollmentStatusResult(Person person, String status)
	            throws Exception {
	        //EnrollmentStatus enrollmentStatus = this.personHelperService
	         //       .getEnrollmentStatus(person);
	        //assertNotNull(enrollmentStatus);
	        //assertEquals(status, enrollmentStatus.getCode());
	    }	
		/*
		protected ReceivedEnrollment createReceivedEnrollment( String status ) throws Exception {
			//??ReceivedEnrollment receivedEnrollment = this.createReceivedEnrollment();
			//receivedEnrollment.setStatus( lookupService.getEnrollmentStatusByCode( status ) );
			return receivedEnrollment;
		}
*/
		protected VAFacility getVaFacility(String stationNumber) throws Exception {
		    return  this.getLookupService().getVaFacilityByStationNumber( stationNumber );
		}
		
		protected NoseThroatRadium createNoseThroatRadium( Date determinationDate, String stationNumber, String verificationMethodCode) throws Exception {
		    NoseThroatRadium noseThroatRadium = new NoseThroatRadium();
		    
		    //noseThroatRadium.setDeterminationDate( determinationDate );
		    noseThroatRadium.setDeterminationFacility( getVaFacility(stationNumber) );
		    noseThroatRadium.setVerificationFacility( getVaFacility(stationNumber) );
		    noseThroatRadium.setVerificationDate( determinationDate );
		    noseThroatRadium.setVerificationMethod( lookupService.getNTRVerificationMethodByCode(NTRVerificationMethod.CODE_QUALIFYING_MILITARY_SERVICE.getName()));
		    noseThroatRadium.setDiagnosisVerifiedDate(createCalendar(2004,12,1).getTime());
		    noseThroatRadium.setFromAviatorServiceBefore1955( new Boolean(true) );
		    noseThroatRadium.setFromSubmarinerServiceBefore1965( new Boolean(true) );
		    noseThroatRadium.setReceivingTreatment( TriState.TRUE );
		    noseThroatRadium.setDiagnosedWithCancer(Boolean.TRUE);
		    
		    return noseThroatRadium;	    
		}
		
		protected boolean hasOtherEligibility( String type, Person person ) {
			boolean contained = false;
		    EnrollmentDetermination enrollmentDetermination = getEnrollmentDetermination( person );		
			Set others = enrollmentDetermination.getOtherEligibilities();
			if ( others != null && others.size() > 0 ) {
				
				Iterator iter = others.iterator();

				while ( iter.hasNext() )
				{
					Eligibility eligibility = (Eligibility)iter.next();
					if ( type.equals( eligibility.getType().getCode() ) )
					{
						contained = true;
						break;
					}
				}			
			}
			
			return contained; 	    
		}
		
		protected boolean hasSecondaryEligibility( String type, Person person ) {
		    EnrollmentDetermination enrollmentDetermination = getEnrollmentDetermination( person );
			boolean contained = false;
			Set secondaries = enrollmentDetermination.getSecondaryEligibilities();
			if ( secondaries != null && secondaries.size() > 0 ) {
				
				Iterator iter = secondaries.iterator();

				while ( iter.hasNext() )
				{
					Eligibility eligibility = (Eligibility)iter.next();
					if ( type.equals( eligibility.getType().getCode() ) )
					{
						contained = true;
						break;
					}
				}			
			}
			
			return contained; 	    
		}
		
		protected boolean hasPrimaryEligibility( String type, Person person ) {
		    EnrollmentDetermination enrollmentDetermination = getEnrollmentDetermination( person );
		    boolean contained = false;
		    Eligibility primary = enrollmentDetermination.getPrimaryEligibility();
		    if ( primary != null ) {
		        
		        if ( type.equals( primary.getType().getCode() ) )
		        {
		            contained = true;
		        }
		    }
		    
		    return contained; 	    
		}		
		
		protected void attachPurpleHeartWithPHStatusPending( Person person ) throws Exception {
			PurpleHeart purpleHeart = new PurpleHeart();
			purpleHeart.setPhIndicator(Boolean.TRUE);
			purpleHeart.setStatus(lookupService.getDecorationStatusByCode(DecorationStatus.CODE_PENDING.getName()));
			person.addDecoration( purpleHeart );
		}
		protected void attachPurpleHeartWithPHRejectRemark( Person person ) throws Exception {
			PurpleHeart purpleHeart = new PurpleHeart();
			purpleHeart.setPhIndicator(Boolean.FALSE);
			purpleHeart.setRejectionRemark(lookupService.getRejectionRemarkByCode("1"));
			person.addDecoration( purpleHeart );
		}
		protected void attachSignature( Person person ) throws Exception {
			SignatureImage signature = new SignatureImage();
			
			signature.setDateIndex(new Date());
			signature.setImageIdNumber("123");
			signature.setUser("John Smith");
			//signature.setSignatureIndicator((SignatureIndicator)lookupService.getByCode(SignatureIndicator.class, SignatureIndicator.CODE_YES.getName()));
			signature.setSiteSubmittingImage(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
	        setPrivateFieldData(signature, "gov.va.med.fw.model.AbstractVersionedEntity", "setModifiedOn", new Date ());
	        setIdentifier(signature,getNextId());
	        
			person.addSignatureImage(signature);
		}
		protected void attachPOW( Person person ) throws Exception {
			PrisonerOfWar pow = new PrisonerOfWar();
			person.setPrisonerOfWar(pow);
			//pow.setPowIndicator( Boolean.TRUE );
			
			POWEpisode powEpisode = new POWEpisode();
			powEpisode.setConfinementLocation(lookupService.getPrisonerOfWarLocationByCode(PrisonerOfWarLocation.CODE_KOREA.getName()));
			powEpisode.setCaptureDate(createImpreciseDate(1944,12,1));
			powEpisode.setReleaseDate(createImpreciseDate(1944,12,2));
			powEpisode.setDataSource((POWDataSource)lookupService.getByCode(POWDataSource.class, POWDataSource.CODE_VAMC.getName()));
			//powEpisode.setDocumentType((POWDocumentType)lookupService.getByCode(POWDocumentType.class, POWDocumentType.CODE_UNKNOWN.getName()));
			powEpisode.setCampLocation((POWCampLocation)lookupService.getByCode(POWCampLocation.class, POWCampLocation.CODE_STALAG1A.getName()));
			powEpisode.setDateEntered(createCalendar(2005,12,3).getTime());
			
			pow.addEpisode(powEpisode);
		}
		
		protected void attachEligibilityVerification ( Person person, String source ) throws Exception {
			EligibilityVerification eligibilityVerification = new EligibilityVerification();
			eligibilityVerification.setEligibilityStatusDate(null);
			eligibilityVerification.setVerificationMethod("01");
			
			eligibilityVerification.setVerificationSource( lookupService.getEligibilityVerificationSourceByCode(source));
			eligibilityVerification.setVerificationFacility(null);
			person.setEligibilityVerification( eligibilityVerification );	    
		}
		
		
	    protected MonetaryBenefit getMonetaryBenefit(String type, Person person)
	            throws Exception  {

	        MonetaryBenefit mb = null;
	        MonetaryBenefitType mbt = null;
	        MonetaryBenefitAward mba = person.getMonetaryBenefitAward();

	        if (mba != null) {
	           mbt = lookupService.getMonetaryBenefitTypeByCode( type ); 
	           mb = getMonetaryBenefit(mbt, person);
	        }

	        return mb;
	    }
	    
	    protected Date getDate(int year, int month, int day) {
	        Calendar calendar = Calendar.getInstance();
	        calendar.set(1989, 7, 25);
	        return calendar.getTime();
	    }    
	    
	    protected FinancialStatement getFinancialStatement(Person person) {
	        // input person has one financial statement.
	        if (person.getFinancialStatements() != null
	                && !person.getFinancialStatements().isEmpty()) {
	            return (FinancialStatement) person.getFinancialStatements()
	                    .values().iterator().next();
	        }
	        return null;
	    }    
	    
	    protected MonetaryBenefit getMonetaryBenefit(
	            MonetaryBenefitType monetaryBenefitType, Person person) {
	        MonetaryBenefitAward mba = person.getMonetaryBenefitAward();
	        MonetaryBenefit mb = null;
	        Set monBens = (mba != null) ? mba.getMonetaryBenefits() : null;

	        if (mba != null && monBens != null && monBens.size() > 0) {
	            Iterator iterMonBens = monBens.iterator();

	            while (iterMonBens.hasNext()) {
	                MonetaryBenefit mbTemp = (MonetaryBenefit) iterMonBens.next();
	                if (mbTemp != null && mbTemp.getType() != null) {
	                    if (mbTemp.getType().getCode().equals(
	                            monetaryBenefitType.getCode())) {
	                        mb = mbTemp;
	                        break;
	                    }
	                }
	            }
	        }

	        return mb;
	    }    
		
		
	   /**
	    * @return Returns the personHelperService.
	    */
	   public PersonHelperService getPersonHelperService()
	   {
	      return personHelperService;
	   }
	   /**
	    * @param personHelperService The personHelperService to set.
	    */
	   public void setPersonHelperService(PersonHelperService personHelperService)
	   {
	      this.personHelperService = personHelperService;
	   }
	   /**
	    * @return Returns the lookupService.
	    */
	   public LookupService getLookupService()
	   {
	      return this.lookupService;
	   }
	   /**
	    * @param lookupService The lookupService to set.
	    */
	   public void setLookupService(LookupService lookupService)
	   {
	      this.lookupService = lookupService;
	   }	
	   
	   public PersonService getPersonService ()
	   {
	   		return personService;
	   }
	   
	   public void  setPersonService (PersonService personService)
	   {
	   		this.personService = personService;
	   }

       public EGTService getEgtService()
       {
           return egtService;
       }

       public void setEgtService(EGTService egtService)
       {
           this.egtService = egtService;
       }

	   public void afterPropertiesSet() throws Exception {
        //super.afterPropertiesSet();
        
        if( getLookupService() == null || getPersonService() == null || getPersonHelperService() == null) {
            throw new ServiceConfigurationException( "Missing required LookupService/PersonService/PersonHelperService");
        }
    } 
	   
    ///////////////////new create methods added /////////////////////////////////
	protected EligibilityVerification createEligibilityVerification () throws Exception{
		EligibilityVerification elVer = new EligibilityVerification();
		elVer.setEligibilityStatus(lookupService.getEligibilityStatusByCode(
				EligibilityStatus.CODE_VERIFIED.getName()));
		elVer.setEligibilityStatusDate(new ImpreciseDate(createCalendar(2004,12,1).getTime()));
		elVer.setVerificationMethod("DD-214");
		elVer.setVerificationSource( lookupService.getEligibilityVerificationSourceByCode(
            EligibilityVerificationSource.CODE_VBA.getName()));
		elVer.setVerificationFacility(getVaFacility(VAFacility.CODE_HEC.getName()));
		return elVer;
	}

    public EnrollmentDetermination createEnrollmentDetermination () throws Exception {
        EnrollmentDetermination enrollDet = new EnrollmentDetermination();
        Eligibility eligibility = new Eligibility ();
        eligibility.setType(lookupService.getEligibilityTypeByCode(
            EligibilityType.SERVICE_CONNECTED_50_TO_100_PERCENT.getName()));
        enrollDet.setPrimaryEligiblity(eligibility);

        //get secondary eligibilities and add
        Eligibility secEligibility1 = new Eligibility ();
        secEligibility1.setType(lookupService.getEligibilityTypeByCode(
            EligibilityType.CHAMPVA.getName()));
        enrollDet.addSecondaryEligibility(secEligibility1);

        Eligibility secEligibility2 = new Eligibility ();
        secEligibility2.setType(lookupService.getEligibilityTypeByCode(
            EligibilityType.EMPLOYEE.getName()));
        enrollDet.addSecondaryEligibility(secEligibility2);

        //get other eligibilities and add
        Eligibility otherEligibility1 = new Eligibility ();
        otherEligibility1.setFactor(lookupService.getEligibilityFactorByCode(
            EligibilityFactor.CATASTROPHICALLY_DISABLED.getName()));
        enrollDet.addOtherEligibility(otherEligibility1);

        Eligibility otherEligibility2 = new Eligibility ();
        otherEligibility2.setFactor(lookupService.getEligibilityFactorByCode(
        		EligibilityFactor.MILITARY_SEXUAL_TRAUMA.getName()));
        enrollDet.addOtherEligibility(otherEligibility2);

        Eligibility otherEligibility3 = new Eligibility ();
        otherEligibility3.setFactor(lookupService.getEligibilityFactorByCode(
        		EligibilityFactor.NOSE_THROAT_RADIUM.getName()));
        enrollDet.addOtherEligibility(otherEligibility3);

        enrollDet.setEnrollmentDate(createCalendar(2003,2,1).getTime());
        enrollDet.setEndDate(createCalendar(2006,1,1).getTime());
        enrollDet.setCalculationSource(lookupService.getEnrollmentSourceByCode(
            EnrollmentSource.CODE_VAMC.getName()));
        enrollDet.setPriorityGroup(lookupService.getEnrollmentPriorityGroupByCode(
            EnrollmentPriorityGroup.GROUP_1.getName()));
        enrollDet.setEndDate(createCalendar(2006,2,1).getTime());
        enrollDet.setFacilityReceived(lookupService.getVaFacilityByCode((VAFacility.CODE_HEC.getName())));

        EGTSetting egt = egtService.getCurrentEGTSetting();
/*
        EGTSetting egt = new EGTSetting();
        egt.setType(lookupService.getEgtSettingTypeByCode(EGTSettingType.CODE_ANNUAL_FISCAL_YEAR.getName()));
        egt.setPriorityGroup(lookupService.getEnrollmentPriorityGroupByCode(
            EnrollmentPriorityGroup.GROUP_1.getName()));
        egt.setYear(new Integer(2005));
        egt.setDisable(Boolean.FALSE);
*/
        enrollDet.setEgtSetting(egt);

            /*private EnrollmentPrioritySubGroup prioritySubGroup = null;
            private EGTSetting egtSetting = null;
            private DataSource dataSource = null;
            private EnrollmentSource calculationSource = null;
            */
        /*setPrivateFieldData(enrollDet, "gov.va.med.fw.model.AbstractVersionedEntity",
                "setModifiedOn", new Date ());*/
        enrollDet.setEffectiveDate(createCalendar(2004,6,1).getTime());
        enrollDet.setEnrollmentStatus(lookupService.getEnrollmentStatusByCode(
            EnrollmentStatus.CODE_VERIFIED.getName()));
        return enrollDet;
    }

	public Address createPrimaryAddress () throws Exception {
		return createAddress(AddressType.CODE_PERMANENT_ADDRESS.getName());
	}
	public Address createTemporayCorrespondanceAddress () throws Exception {
        Address address = createAddress(AddressType.CODE_TEMPORARY_CORRESPONDENCE_ADDRESS.getName());
        address.setStartDate(new ImpreciseDate(createCalendar(2004,5,31)));
		address.setEndDate(new ImpreciseDate(createCalendar(2004,8,31)));
		return address;
	}
	public Address createConfidentialAddress () throws Exception {
        Address address = createAddress(AddressType.CODE_CONFIDENTIAL_ADDRESS.getName());
		address.setEndDate(new ImpreciseDate(createCalendar(2006,8,31)));
		return address;
	}
	public Address createAddress(String addressTypeCode) throws Exception {
		Address address = new Address ();
		address.setLine1("123 Chain Bridge Road");
		address.setLine2("Suite 220");
		address.setLine3("Attn: David Henry");
		address.setCity("Herndon");
		address.setZipCode("20171");
        address.setZipPlus4("1234");
		address.setState("VA");
		address.setCountry(Country.CODE_USA.getName());
		address.setCounty("Fairfax");
		address.setBadAddressReason(lookupService.getBadAddressReasonByCode(
            BadAddressReason.CODE_UNDELIVERABLE.getName()));
		address.setType(lookupService.getAddressTypeByCode(addressTypeCode));
		address.setStartDate(new ImpreciseDate(createCalendar(2005,6,1)));
		address.setChangeSource(lookupService.getAddressChangeSourceByCode(
            AddressChangeSource.CODE_VAMC.getName()));
		address.setChangeSite(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
        address.setChangeDate(new Date());
        //setPrivateFieldData(address, "gov.va.med.fw.model.AbstractVersionedEntity", "setModifiedOn", new Date ());
		address.setPhoneNumber("703-713-2848");
		return address;
	}
	
	public Phone createBusinessPhone (BigDecimal id) throws Exception {
		return createPhone (PhoneType.CODE_BUSINESS.getName(), id);
	}
	public Phone createHomePhone (BigDecimal id) throws Exception {
		return createPhone (PhoneType.CODE_HOME.getName(), id);
	}
	/** 
	 * create phone for person 
	 * @param phoneTypeCode
	 * @param id
	 * @return
	 * @throws Exception
	 */
	public Phone createPhone(String phoneTypeCode, BigDecimal id) throws Exception {
		Phone phone = new Phone ();
		phone.setPhoneNumber("703-733-7254");
		phone.setType(lookupService.getPhoneTypeByCode(phoneTypeCode));
		if (id != null) {
			setIdentifier(phone,id);
		}
        phone.setSourceOfChange(lookupService.getPhoneSourceOfChangeByCode(PhoneSourceOfChange.CODE_VAMC.getName()));
		phone.setSiteOfChange(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
        setPrivateFieldData(phone, "gov.va.med.fw.model.AbstractVersionedEntity", "setModifiedOn", new Date ());
		return phone;
	}
	
	/**
	 * create email address for person
	 * @param id
	 * @throws Exception
	 */
	
	public Email createPrivateEmail(BigDecimal id) throws Exception {
		return createEmail("privatemail@eds.com",EmailType.CODE_PRIVATE.getName(), id);
	}
	public Email createBusinessEmail(BigDecimal id) throws Exception {
		return createEmail("business@eds.com",EmailType.CODE_BUSINESS.getName(), id);
	}
	public Email createEmail(String emailAddress, String emailTypeCode, BigDecimal id) throws Exception {
		Email email = new Email ();
		email.setAddress(emailAddress == null ? "email@eds.com" : emailAddress);
		email.setType(lookupService.getEmailTypeByCode(emailTypeCode));
		if (id != null) {
			setIdentifier(email,id);
		}
        email.setSourceOfChange(lookupService.getEmailSourceOfChangeByCode(EmailSourceOfChange.CODE_VAMC.getName()));
        email.setSiteOfChange(getVaFacility(VAFacility.CODE_NEW_BEDFORD_CBOC.getName()));
        setPrivateFieldData(email, "gov.va.med.fw.model.AbstractVersionedEntity", "setModifiedOn", new Date ());
		return email;
	}
	
	/**
	 * Attache patient visit summaries
	 * @param person
	 */
	protected void attachPatientVisitSummaries (Person person) throws Exception{
		VAFacility facility = getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName());
		PatientVisitSummary visitSummary = new PatientVisitSummary();
		visitSummary.setLastVisitDate(createCalendar(2005,6,1).getTime ());
		Integer incomeYear = new Integer(2004);
        visitSummary.setIncomeYear(incomeYear);
		visitSummary.setInpatientDays(new Integer(10));
		visitSummary.setOutpatientDays(new Integer(20));
		//use the setPrivateFieldData to set data in private fields		
		setPrivateFieldData(visitSummary,"gov.va.med.esr.common.model.financials.PatientVisitSummary","setFacilityVisited",facility);
        person.setPatientVisitSummary(incomeYear, facility,visitSummary);

        FeeBasis feeBasis = new FeeBasis();
        feeBasis.setAuthorizationFromDate(new Date());
        feeBasis.setAuthorizationToDate(new Date());
        feeBasis.setReportDate(new Date());
        feeBasis.setReportSite(facility);
        feeBasis.setProgram(lookupService.getFeeBasisProgramByCode(
            FeeBasisProgram.CODE_HOME_HEALTH_SERVICES.getName()));
        feeBasis.setTreatmentType(lookupService.getFeeBasisTreatmentTypeByCode(
            FeeBasisTreatmentType.CODE_SHORT_TERM_FEE_STATUS.getName()));
        person.addFeeBasis(feeBasis);

        facility = getVaFacility(VAFacility.CODE_IRON_MOUNTAIN_VA_MEDICAL_CENTER.getName());
        visitSummary = new PatientVisitSummary();
        visitSummary.setLastVisitDate(createCalendar(2005,6,1).getTime ());
        visitSummary.setIncomeYear(incomeYear);
        visitSummary.setInpatientDays(new Integer(30));
        visitSummary.setOutpatientDays(new Integer(40));
        //use the setPrivateFieldData to set data in private fields
        setPrivateFieldData(visitSummary,"gov.va.med.esr.common.model.financials.PatientVisitSummary","setFacilityVisited",facility);
		person.setPatientVisitSummary(incomeYear, facility,visitSummary);

        feeBasis = new FeeBasis();
        feeBasis.setAuthorizationFromDate(new Date());
        feeBasis.setAuthorizationToDate(new Date());
        feeBasis.setReportDate(new Date());
        feeBasis.setReportSite(facility);
        feeBasis.setProgram(lookupService.getFeeBasisProgramByCode(
            FeeBasisProgram.CODE_HOME_HEALTH_SERVICES.getName()));
        feeBasis.setTreatmentType(lookupService.getFeeBasisTreatmentTypeByCode(
            FeeBasisTreatmentType.CODE_SHORT_TERM_FEE_STATUS.getName()));
        person.addFeeBasis(feeBasis);
	}
	
	protected void attachInsurance(Person person)  throws Exception
	{
	    Set ins = person.getInsurances();
	    if(ins.isEmpty()) 
	    {
	    	InsurancePolicy insurance = createPrivateInsurance();
			insurance.setReportSite(getVaFacility(VAFacility.CODE_HEC.getName()));
	        person.addInsurance(insurance);
	        
	        insurance = createMedicare();
			insurance.setReportSite(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
	        person.addInsurance(insurance);
	        
	        //insurance = createMedicare();
			//insurance.setReportSite(getVaFacility(VAFacility.CODE_HEC.getName()));
	        //person.addInsurance(insurance);
	    }
	}
	
	protected InsurancePolicy createMedicare() throws Exception
 	{
		Medicare ins = new Medicare();
		setIdentifier(ins,getNextId());
		ins.setGroupName("A");
		ins.setGroupNumber("987"); 
		ins.setEnrolledInPartA(Boolean.TRUE);
		// ins.setPartAEffectiveDate(new Date());
		ins.setPartAEffectiveDate(new ImpreciseDate( Calendar.getInstance()) );
		ins.setEnrolledInPartB(Boolean.FALSE);
		ins.setMedicareClaimNumber("987654321");
		ins.setNameOnMedicareCard("Test Veteran");
		ins.setAddress(createAddress(AddressType.CODE_PERMANENT_ADDRESS.getName()));
		ins.setBusinessPhone(createBusinessPhone(getNextId()));
				
		InsurancePlan plan = new InsurancePlan();
		//setIdentifier(plan,getNextId());
		plan.setPrecertificationRequired(Boolean.TRUE);
		plan.setPlanType(lookupService.getInsurancePlanTypeByCode(InsurancePlanType.CODE_MEDICARE_M.getName()));
		ins.setInsurancePlan(plan);
		
		InsuranceSubscriber sub = new InsuranceSubscriber();
		//setIdentifier(sub,getNextId());
		sub.setName("Test Person B");
		sub.setRelationship(lookupService.getInsuredRelationshipByCode(InsuredRelationship.CODE_VETERAN.getName()));
		ins.setSubscriber(sub);
		
		return ins;
 	}
	
	protected InsurancePolicy createPrivateInsurance() throws Exception
 	{
		PrivateInsurance ins = new PrivateInsurance();
		setIdentifier(ins,getNextId());
		ins.setCompanyName("Cigna");
		ins.setGroupName("GroupName123");
		ins.setGroupNumber("123");
		ins.setPolicyNumber("123456789");
		ins.setPolicyEffectiveDate(new ImpreciseDate("01/01/2005"));
		ins.setPolicyExpirationDate(new ImpreciseDate("12/31/2005"));
		ins.setAddress(createAddress(AddressType.CODE_PERMANENT_ADDRESS.getName()));
		ins.setBusinessPhone(createBusinessPhone(getNextId()));
		
		InsurancePlan plan = new InsurancePlan();
		//setIdentifier(plan,getNextId());
		plan.setPrecertificationRequired(Boolean.TRUE);
		plan.setPlanType(lookupService.getInsurancePlanTypeByCode(InsurancePlanType.CODE_PPO.getName()));
		ins.setInsurancePlan(plan );
		
		InsuranceSubscriber sub = new InsuranceSubscriber();
		//setIdentifier(sub,getNextId()); setIdentifier is not valid on this object
		sub.setName("Test Person A");
        sub.setRelationship(lookupService.getInsuredRelationshipByCode(InsuredRelationship.CODE_VETERAN.getName()));
		ins.setSubscriber(sub);
		
		return ins;
 	}

	protected void attachFinancialInfo(Person person, Integer year)  throws Exception
	{
		FinancialStatement fin = person.getFinancialStatement(year);
	    if(fin == null) 
	    {	
	    	//BeneficiaryTravel
	    	BeneficiaryTravel bf = new BeneficiaryTravel();
	    	bf.setAmountCertified(new BigDecimal(666));
	    	bf.setClaimDate(new Date());
	    	//bf.setFacilityVisited(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
	    	
	    	//Add some income test information
			IncomeTest incomeTest = this.createIncomeTest(IncomeTestType.CODE_MEANS_TEST.getName(),
		        MeansTestStatus.MT_STATUS_EXEMPT_LTC_COPAY_EXEMPTION_TEST.getName(), year );
			incomeTest.setSiteConductingTest(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()));
			incomeTest.setEffectiveDate(this.createCalendar(1995, 2, 1).getTime());
			//incomeTest.setBeneficiaryTravel(getVaFacility(VAFacility.CODE_CONNECTICUT_HCS.getName()), bf);
			person.setIncomeTest(year, incomeTest );
			
			Income i = new Income();
			i.setAmount(new BigDecimal(3000));
			
			//add financial info
			FinancialStatement fstmt = new FinancialStatement();
            fstmt.setIsPost2005Format(new Boolean(true));

			setIdentifier(fstmt,getNextId());
			
			fstmt.setContributionToSpouse(new BigDecimal(9988));
			fstmt.setMarriedLastCalendarYear(Boolean.TRUE);
			fstmt.setNumberOfDependentChildren(new Integer(5));
			fstmt.setIncome(lookupService.getIncomeTypeByCode(IncomeType.INCOME_TYPE_UNEMPLOYMENT_COMPENSATION.getName()), i);
			fstmt.setDebt(new Debt(new BigDecimal(333)));
			
	 	    NameType nameType = getLookupService().getNameTypeByCode(NameType.LEGAL_NAME.getName());
	 	    Name name =  new Name();
	 	    name.setFamilyName("LastName");
	 	    name.setGivenName("Spouse");
	 	    name.setMiddleName("V");
	 	    name.setPrefix("Mrs.");
	 	    name.setType(nameType);
	 	    
	 	    Spouse spouse = new Spouse();
	 	    setIdentifier(spouse,getNextId());
	 	    spouse.setName(name);
	 	    spouse.addSsn(createSSN("222222222",SSNType.CODE_ACTIVE.getName()));
	 	    spouse.setDob(new ImpreciseDate("01/01/1969"));
	 	    spouse.setRelationship(getLookupService().getRelationshipByCode("2"));
			
			SpouseFinancials sp = new SpouseFinancials();
			setIdentifier(sp,getNextId());
			sp.setLivedWithPatient(Boolean.TRUE);
			Income si = new Income();
			si.setAmount(new BigDecimal(2000));
			sp.setIncome(lookupService.getIncomeTypeByCode(IncomeType.INCOME_TYPE_TOTAL_INCOME_FROM_EMPLOYMENT.getName()), si);
			sp.setDebt(new Debt(new BigDecimal(222)));
			sp.setReportedOn(spouse);
			fstmt.addSpouseFinancials(sp);
			
	 	    NameType nameTypeD = getLookupService().getNameTypeByCode(NameType.LEGAL_NAME.getName());
	 	    Name depName =  new Name();
	 	    depName.setFamilyName("LastName");
	 	    depName.setGivenName("Dependent");
	 	    depName.setMiddleName("P");
	 	    depName.setPrefix("Ms.");
	 	    depName.setType(nameTypeD);
	 	    
	 	    Dependent dep = new Dependent();
	 	    setIdentifier(dep,getNextId());
	 	    dep.setName(depName);
	 	    dep.addSsn(createSSN("444444444",SSNType.CODE_ACTIVE.getName()));
	 	    dep.setDob(new ImpreciseDate("01/01/1999"));
	 	    dep.setRelationship(getLookupService().getRelationshipByCode("4"));

			DependentFinancials dp = new DependentFinancials();
			setIdentifier(dp,getNextId());
			dp.setAttendedSchool(Boolean.FALSE);
			dp.setHasIncome(Boolean.TRUE);
			Income di = new Income();
			di.setAmount(new BigDecimal(1500));
			dp.setDebt(new Debt(new BigDecimal(111)));
		
	
			dp.setIncome(lookupService.getIncomeTypeByCode(IncomeType.INCOME_TYPE_TOTAL_INCOME_FROM_EMPLOYMENT.getName()), di);
			dp.setIncomeAvailableToPatient(Boolean.TRUE);
			dp.setLivedWithPatient(Boolean.TRUE);
			dp.setReportedOn(dep);
			fstmt.addDependentFinancials(dp);
			
			person.setFinancialStatement(year, fstmt);
	    }
	}
	
	/**
	 * Generic methods to set identifiers and private member variables
	 * using Java reflection
	 */
	//generic methods to set data for private member variables 
	private static void setIdentifier(Object obj, BigDecimal id) {
		if (!setIdentifierActive) {
			log.error("setIdentifier is disabled:" + obj);
			return;
		}
		try {			
			Class abstractKeyedEntity = Class.forName("gov.va.med.fw.model.AbstractKeyedEntity");
			final Method[] methods = abstractKeyedEntity.getDeclaredMethods();
			for (int i=0; i<methods.length; i++)
			{
				if (methods[i].getName().equals("setIdentifier"))
				{
					final Object params[] = new Object[1];
					params[0] = id;
					methods[i].setAccessible(true);
					methods[i].invoke(obj,params);
				}
			}
		}
		catch (Exception e) {
			log.error("Exception in setIdentifier",e);
		}
	}	
	
	/**
	 * set private field data of an object by modufying the method accessibility 
	 * in the loaded class object
	 * @param obj
	 * @param className
	 * @param methodName
	 * @param data
	 */
	private static void setPrivateFieldData(Object obj, String className, String methodName, Object data)
	{
		try {		
			Class objectClass = Class.forName(className);
			final Method[] methods = objectClass.getDeclaredMethods();
			
			for (int i=0; i<methods.length; i++)
			{
				if (log.isDebugEnabled()) {
					log.debug(" Method:" +	methods[i].getName ());
					Class[] parTypes = methods[i].getParameterTypes();
					for (int j=0; j<parTypes.length; j++) 
						log.debug("Parameters: " + parTypes[j].getName());
				}
				
				if (methods[i].getName().equals(methodName))
				{
					final Object params[] = new Object[1];
					params[0] = data;
					methods[i].setAccessible(true);
					methods[i].invoke(obj,params);
				}
			}
		}
		catch (Exception e) {		
			log.error("Exception in setPrivateFieldData ",e);			
		}
	}
	
	public static Timestamp createTimestamp (int year, int month, int day) {
		Calendar calender = Calendar.getInstance();
		calender.clear ();
		calender.set(Calendar.YEAR,year);
		calender.set(Calendar.MONTH,month-1); //month starts from 0
		calender.set(Calendar.DAY_OF_MONTH,day);
		return new Timestamp(calender.getTime().getTime());		
	}
	public static Set getChangeEvents (EntityKey entityKey) {
		Set changeEvents = new HashSet ();
		changeEvents.add(new ChangeEvent(createTimestamp(2005,8,1),entityKey));
		changeEvents.add(new ChangeEvent(createTimestamp(2005,6,1),entityKey));
		changeEvents.add(new ChangeEvent(createTimestamp(2005,4,1),entityKey));
		changeEvents.add(new ChangeEvent(createTimestamp(2005,3,1),entityKey));
		changeEvents.add(new ChangeEvent(createTimestamp(2005,1,1),entityKey));
		return changeEvents;				
	}
	
	//Key generation initial value
	private static BigDecimal identifier = new BigDecimal(1000);
	
	private static boolean setIdentifierActive = true;
	private static void enableSetIdentifier() {
		setIdentifierActive = true;
	}
	private static void disableSetIdentifier() {
		setIdentifierActive = false;
	}
	
	//increments the current key value and returns the new value
	public  static synchronized BigDecimal getNextId() {
		identifier = identifier.add(new BigDecimal(1));
		return identifier;
	}
	
	//FIXME temp methods added to avoid compilation errors
	private void assertTrue(boolean value) {}
	private void assertFalse(boolean value) {}
	private void assertNull(boolean value) {}
	private void assertNotNull(boolean value) {}
	

}
