/********************************************************************
 * Copyright  2006 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.ui.registry.action;

// Java classes
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

// ESR classes
import gov.va.med.esr.common.infra.ImpreciseDate;
import gov.va.med.esr.common.model.ee.PrisonerOfWar;
import gov.va.med.esr.common.model.ee.POWEpisode;
import gov.va.med.esr.common.model.ee.PurpleHeart;
import gov.va.med.esr.common.model.ee.PurpleHeartDocument;
import gov.va.med.esr.common.model.ee.SHADDocument;
import gov.va.med.esr.common.model.person.SSN;
import gov.va.med.esr.common.model.registry.Registry;
import gov.va.med.esr.common.model.registry.RegistryTrait;
import gov.va.med.esr.common.model.registry.RegistryTraitDetail;
import gov.va.med.esr.common.model.ee.SHAD;
import gov.va.med.esr.common.model.lookup.DecorationStatus;
import gov.va.med.esr.common.model.lookup.Indicator;
import gov.va.med.esr.common.model.lookup.PHDataSource;
import gov.va.med.esr.common.model.lookup.PHDocumentType;
import gov.va.med.esr.common.model.lookup.POWCampLocation;
import gov.va.med.esr.common.model.lookup.POWDataSource;
import gov.va.med.esr.common.model.lookup.POWDocumentType;
import gov.va.med.esr.common.model.lookup.PrisonerOfWarLocation;
import gov.va.med.esr.common.model.lookup.RegistryType;
import gov.va.med.esr.common.model.lookup.RejectionRemark;
import gov.va.med.esr.common.model.lookup.SHADDocumentType;
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.ui.common.util.DateUtils;
import gov.va.med.esr.ui.common.util.JspUtils;
import gov.va.med.esr.ui.conversion.UIConversionServiceImpl;
import gov.va.med.fw.conversion.ConversionServiceException;
import gov.va.med.fw.model.AbstractKeyedEntity;
import gov.va.med.fw.model.lookup.Lookup;
import gov.va.med.fw.security.Principal;
import gov.va.med.fw.util.StringUtils;

/**
 * The conversion service for association add/update actions.
 *
 * @author Muddaiah Ranga
 * @version 3.0
 */
public class RegistryConversionService extends UIConversionServiceImpl {
    
	public Object convert(Object source, Class targetClass)
	throws ConversionServiceException {
	    return null;
	}
	
    protected void convertBean(Object source, Object target) 
	throws ConversionServiceException {
        
        // Convert RegistryInfoForm to Registry
        if(source instanceof RegistryInfoForm && target instanceof Registry) {
            RegistryInfoForm form = (RegistryInfoForm)source;
            String registryType = form.getRegistryType();
            if(StringUtils.equals(RegistryType.CODE_PH_REGISTRY.getName(),registryType)) { 
                convertPHFormToPurpleHeart((RegistryPHInfoForm)form,(PurpleHeart)target);
            } else if(StringUtils.equals(RegistryType.CODE_POW_REGISTRY.getName(),registryType)) {
                convertPOWFormToPrisonerOfWar((RegistryPOWInfoForm)form,(PrisonerOfWar)target);
            } else if(StringUtils.equals(RegistryType.CODE_SHAD_REGISTRY.getName(),registryType)) {
                convertSHADFormToSHADRegistry((RegistrySHADInfoForm)form,(SHAD)target);
            }
        // Convert Registry to RegistryInfoForm
        } else if(source instanceof Registry && target instanceof RegistryInfoForm) {
            Registry registry = (Registry)source;
            String registryType = (registry != null && registry.getRegistryTrait() != null && registry.getRegistryTrait().getRegistryType() != null) ? registry.getRegistryTrait().getRegistryType().getCode()  : null;
            if(StringUtils.equals(RegistryType.CODE_PH_REGISTRY.getName(),registryType)) {
                convertPurpleHeartToPHForm((PurpleHeart)source,(RegistryPHInfoForm)target);
            } else if(StringUtils.equals(RegistryType.CODE_POW_REGISTRY.getName(),registryType)) {
                convertPrisonerOfWarToPOWForm((PrisonerOfWar)source,(RegistryPOWInfoForm)target);
            } else if(StringUtils.equals(RegistryType.CODE_SHAD_REGISTRY.getName(),registryType)) {
                convertSHADRegistryToSHADForm((SHAD)source, (RegistrySHADInfoForm)target);
            }
        }
	}
    
    private void convertPHFormToPurpleHeart(RegistryPHInfoForm form, PurpleHeart registry) 
    throws ConversionServiceException {
        
        // Set PH indicator only for updates
        if(registry.getEntityKey() != null) {
            registry.setPhIndicator(this.getTrueFalse(form.getPurpleHeartIndicator()));
        }
        
        populateCommon(form,registry);
        
        DecorationStatus status = StringUtils.isNotEmpty(form.getPurpleHeartStatus()) ? (DecorationStatus)getLookup(DecorationStatus.class,form.getPurpleHeartStatus()) : null;
        RejectionRemark rejectionRemark = StringUtils.isNotEmpty(form.getRejectedRemarks()) ? (RejectionRemark)getLookup(RejectionRemark.class,form.getRejectedRemarks()) : null;
        VAFacility facility = StringUtils.isNotEmpty(form.getPurpleHeartSiteNumber()) ? (VAFacility)getLookup(VAFacility.class,form.getPurpleHeartSiteNumber()) : null;
        registry.setStatus(status);
        registry.setRejectionRemark(rejectionRemark);
        registry.setFacility(facility);
        registry.setAssignedContactUser(StringUtils.isBlank(form.getAssignedLAS()) ? null : form.getAssignedLAS());
        
        // Update PH documents
        Map existingPHDocumentsMap = getMapFromCollection(registry.getDocuments());
        registry.removeAllDocuments();
        List documentIds = form.getDocumentIdList();
        if(documentIds != null && !documentIds.isEmpty()) {
            for(int i=0; i<documentIds.size(); i++) {
                String phDocumentId = (String)documentIds.get(i);
                if(StringUtils.isEmpty(phDocumentId)) {
                    if(!form.isPHDocumentEmpty(i)) {
                        //add a new document
                        PurpleHeartDocument newPHDoc = ctrateNewPHDocument(form,i);
                        form.setDocumentHashCode(i,String.valueOf(newPHDoc.hashCode()));
                        registry.addDocument(newPHDoc);
                    }
                    continue;
                }
                if(!form.isPHDocumentEmpty(i)) {
                    PurpleHeartDocument phDoc = (PurpleHeartDocument)existingPHDocumentsMap.get(phDocumentId);
                    //update PH document
                    phDoc = (phDoc == null) ? ctrateNewPHDocument(form,i) : updatePHDocument(form,i,phDoc);
                    form.setDocumentHashCode(i,String.valueOf(phDoc.hashCode()));
                    registry.addDocument(phDoc);
                }
            }
        }
    }
    
    private void convertPurpleHeartToPHForm(PurpleHeart registry, RegistryPHInfoForm form) 
    throws ConversionServiceException {
        form.setPurpleHeartIndicator(this.getYesNo(registry.getPhIndicator()));

        form.setPurpleHeartStatus(this.getLookupCode(registry.getStatus()));
        form.setStatusUpdateDate(DateUtils.format(registry.getStatusLastEditDate(),DateUtils.MMDDYYYY));

        form.setRejectedRemarks(this.getLookupCode(registry.getRejectionRemark()));
        form.setPurpleHeartSiteNumber(this.getLookupCode(registry.getFacility()));
        
        populateCommon(registry,form);
        form.setAssignedLAS(registry.getAssignedContactUser());
        Set documents = registry.getDocuments();
        if(documents != null) {
            PurpleHeartDocument document = null;
            String id,documentType,dataSource,documentReceivedDate,documentLogInDate,documentLoggedInBy,documentAcceptable,unacceptableRemark,hashCode = null;
            for(Iterator iter=documents.iterator(); iter.hasNext();) {
                document = (PurpleHeartDocument)iter.next();
                if(document != null) {
                    id = document.getEntityKey().getKeyValueAsString();
                    documentType = this.getLookupCode(document.getDocumentType());
                    dataSource = this.getLookupCode(document.getDataSource());
                    documentReceivedDate = DateUtils.format(document.getReceivedDate(),DateUtils.MMDDYYYY);
                    documentLogInDate = DateUtils.format(document.getLoginDate(),DateUtils.MMDDYYYY);
                    documentLoggedInBy = document.getLoggedInUser();
                    documentAcceptable = this.getYesNo(document.getAcceptable());
                    unacceptableRemark = document.getUnacceptableRemarks();
                    hashCode = String.valueOf(document.hashCode());
                    form.addPHDocument(id,documentType,dataSource,documentReceivedDate,documentLogInDate,documentLoggedInBy,documentAcceptable,unacceptableRemark,hashCode);
                }
            }
        }
    }
    
    private void convertPOWFormToPrisonerOfWar(RegistryPOWInfoForm form, PrisonerOfWar registry) 
    throws ConversionServiceException {

        if(registry.getEntityKey() != null) {
            Indicator indicator = (Indicator)getLookup(Indicator.class,this.getIndicatorCode(form.getPowIndicator()));
            registry.setPowIndicator(indicator);
        }
        
        populateCommon(form,registry);
        RegistryTrait trait = registry.getRegistryTrait();
        if(trait != null) {
            ImpreciseDate dateWithTime = DateUtils.getImpreciseDate(form.getDateDeath(), true);
            if (dateWithTime != null) {
            String impreciseString = dateWithTime.toStandardFormat();
            if (impreciseString.length() > 8) {
                 if(dateWithTime.equals(trait.getDeathDate()) == false) {
                   	impreciseString = impreciseString.substring(0, 8);
                   	ImpreciseDate dateWithoutTime = new ImpreciseDate(impreciseString);
                   	trait.setDeathDate(dateWithoutTime);
                 }
           	}
            else {
            	trait.setDeathDate(DateUtils.getImpreciseDate(form.getDateDeath(), true));
            }
            }
            else {
            	trait.setDeathDate(DateUtils.getImpreciseDate(form.getDateDeath(), true));
            }
        }
        
        registry.setIcn(StringUtils.isBlank(form.getIcn()) ? null : form.getIcn());
        
        // Update PH documents
        Map existingPOWEpisodesMap = getMapFromCollection(registry.getEpisodes());
        registry.removeAllEpisodes();
        List episodetIds = form.getEpisodeIdList();
        if(episodetIds != null && !episodetIds.isEmpty()) {
            for(int i=0; i<episodetIds.size(); i++) {
                String powEpisodeId = (String)episodetIds.get(i);
                if(StringUtils.isEmpty(powEpisodeId)) {
                    if(!form.isPOWEpisodeEmpty(i)) {
                        //add a new episode
                        POWEpisode newPOWEpisode = ctrateNewPOWEpisode(form,i);
                        form.setEpisodeHashCode(i,String.valueOf(newPOWEpisode.hashCode()));
                        registry.addEpisode(newPOWEpisode);
                    }
                    continue;
                }
                if(!form.isPOWEpisodeEmpty(i)) {
                    POWEpisode powEpisode = (POWEpisode)existingPOWEpisodesMap.get(powEpisodeId);
                    //update POW episode
                    powEpisode = (powEpisode == null) ? ctrateNewPOWEpisode(form,i) : updatePOWEpisode(form,i,powEpisode);
                    form.setEpisodeHashCode(i,String.valueOf(powEpisode.hashCode()));
                    registry.addEpisode(powEpisode);
                }
            }
        } 
    }
    
    private void convertPrisonerOfWarToPOWForm(PrisonerOfWar registry, RegistryPOWInfoForm form) 
    throws ConversionServiceException {       
        populateCommon(registry,form);
        String powIndicatorCode=registry!=null && registry.getPowIndicator()!=null?registry.getPowIndicator().getCode():null;
        form.setPowIndicator(powIndicatorCode!=null&& powIndicatorCode.equals(Indicator.NO.getCode())?RegistryInfoForm.N:RegistryInfoForm.Y);
        
        ImpreciseDate deathDate = (registry.getRegistryTrait() != null) ? registry.getRegistryTrait().getDeathDate() : null;
        form.setDateDeath(DateUtils.formatForEdit(deathDate, null));
        
        form.setIcn(registry.getIcn());
        form.setDateEntered(DateUtils.format(registry.getDateEntered(),DateUtils.MMDDYYYY));
        
        Set episodes = registry.getEpisodes();
        if(episodes != null) {
            POWEpisode episode = null;
            String id,documentType,dataSource,captureDate,releaseDate,daysInCaptivity,confinementLocation,campLocation,enteredDate,enteredBy,hashCode = null;
            for(Iterator iter=episodes.iterator(); iter.hasNext();) {
                episode = (POWEpisode)iter.next();
                if(episode != null) {
                    id = episode.getEntityKey().getKeyValueAsString();
                    documentType = this.getLookupCode(episode.getDocumentType());
                    dataSource = this.getLookupCode(episode.getDataSource());
                    ImpreciseDate powCaptureDate = episode.getCaptureDate();
                    ImpreciseDate powReleaseDate = episode.getReleaseDate();
                    captureDate = DateUtils.format(powCaptureDate,DateUtils.MMDDYYYY);
                    releaseDate = DateUtils.format(powReleaseDate,DateUtils.MMDDYYYY);
                    daysInCaptivity = (episode.getNumberOfDaysInCaptivity() != null) ? episode.getNumberOfDaysInCaptivity().toString() : null;
                    if(StringUtils.isEmpty(daysInCaptivity) 
                    	&& powCaptureDate != null
						&& powReleaseDate != null ) {
                        Long days = DateUtils.getDays(powCaptureDate,powReleaseDate, true);
                        daysInCaptivity = (days != null) ? days.toString() : "";
                    }
                    confinementLocation = (episode.getConfinementLocation() != null) ? episode.getConfinementLocation().getCode() : null;
                    campLocation = (episode.getCampLocation() != null) ? episode.getCampLocation().getCode() : null;
                    enteredDate = DateUtils.format(episode.getDateEntered(),DateUtils.MMDDYYYY);
                    enteredBy = (episode.getCreatedBy() != null) ? episode.getCreatedBy().getName() : "";
                    hashCode = String.valueOf(episode.hashCode());
                    form.addPOWEpisode(id,documentType,dataSource,captureDate,releaseDate,daysInCaptivity,confinementLocation,campLocation,enteredDate,enteredBy,hashCode);
                }
            }
        }
    }
    
    private void convertSHADFormToSHADRegistry(RegistrySHADInfoForm form, SHAD registry) 
    throws ConversionServiceException {
        
        if(registry.getEntityKey() != null) {
            Indicator indicator = (Indicator)getLookup(Indicator.class,this.getIndicatorCode(form.getShadIndicator()));
            registry.setShadIndicator(indicator);
        }
        
        populateCommon(form,registry);

        registry.setClaimNumber(StringUtils.isBlank(form.getClaimNumber()) ? null : form.getClaimNumber());
        
        // Update PH documents
        Map existingSHADDocumentsMap = getMapFromCollection(registry.getDocuments());
        registry.removeAllDocuments();
        List documentIds = form.getDocumentIdList();
        if(documentIds != null && !documentIds.isEmpty()) {
            for(int i=0; i<documentIds.size(); i++) {
                String documentId = (String)documentIds.get(i);
                if(StringUtils.isEmpty(documentId)) {
                    if(!form.isSHADDocumentEmpty(i)) {
                        //add a new document
                        SHADDocument newSHADDoc = ctrateNewSHADDocument(form,i);
                        form.setDocumentHashCode(i,String.valueOf(newSHADDoc.hashCode()));
                        registry.addDocument(newSHADDoc);
                    }
                    continue;
                }
                if(!form.isSHADDocumentEmpty(i)) {
                    SHADDocument shadDoc = (SHADDocument)existingSHADDocumentsMap.get(documentId);
                    //update SHAD document
                    shadDoc = (shadDoc == null) ? ctrateNewSHADDocument(form,i) : updateSHADDocument(form,i,shadDoc);
                    form.setDocumentHashCode(i,String.valueOf(shadDoc.hashCode()));
                    registry.addDocument(shadDoc);
                }
            }
        }
    }
    
    private void convertSHADRegistryToSHADForm(SHAD registry, RegistrySHADInfoForm form) 
    throws ConversionServiceException {
        form.setShadIndicator(this.getYesNo(registry.getShadIndicator()));
        populateCommon(registry,form);
        form.setClaimNumber(registry.getClaimNumber());
        
        Set documents = registry.getDocuments();
        if(documents != null) {
            SHADDocument document = null;
            String id,documentType,documentReceivedDate,hashCode = null;
            for(Iterator iter=documents.iterator(); iter.hasNext();) {
                document = (SHADDocument)iter.next();
                if(document != null) {
                    id = document.getEntityKey().getKeyValueAsString();
                    documentType = this.getLookupCode(document.getDocumentType());
                    documentReceivedDate = DateUtils.format(document.getReceivedDate(),DateUtils.MMDDYYYY);
                    hashCode = String.valueOf(document.hashCode());
                    form.addSHADDocument(id,documentType,documentReceivedDate,hashCode);
                }
            }
        }
    }
    
    private PurpleHeartDocument ctrateNewPHDocument(RegistryPHInfoForm form, int index)
    throws ConversionServiceException
    {
        PurpleHeartDocument newPHDoc = new PurpleHeartDocument();
        return updatePHDocument(form,index,newPHDoc);
    }
    
    private PurpleHeartDocument updatePHDocument(RegistryPHInfoForm form, int index,PurpleHeartDocument phDoc)
    throws ConversionServiceException
    {
        PHDocumentType documentType = (PHDocumentType)getLookup(PHDocumentType.class,form.getDocumentType(index));
        PHDataSource dataSource = (PHDataSource)getLookup(PHDataSource.class,form.getDataSource(index));
        phDoc.setDocumentType(documentType);
        phDoc.setDataSource(dataSource);
        phDoc.setReceivedDate(DateUtils.getDate(form.getDocumentReceivedDate(index)));
        phDoc.setAcceptable(getTrueFalse(form.getDocumentAcceptable(index)));
        phDoc.setUnacceptableRemarks(StringUtils.isBlank(form.getUnacceptableRemark(index)) ? null :
            form.getUnacceptableRemark(index));
        return phDoc;
    }
    
    private POWEpisode  ctrateNewPOWEpisode(RegistryPOWInfoForm form, int index)
    throws ConversionServiceException
    {
        POWEpisode newPOWEpisode = new POWEpisode();
        return updatePOWEpisode(form,index,newPOWEpisode);
    }
    
    private POWEpisode updatePOWEpisode(RegistryPOWInfoForm form, int index,POWEpisode powEpisode)
    throws ConversionServiceException
    {
        POWDocumentType documentType = (POWDocumentType)getLookup(POWDocumentType.class,form.getDocumentType(index));
        powEpisode.setDocumentType(documentType);
        
        POWDataSource dataSource = (POWDataSource)getLookup(POWDataSource.class,form.getDataSource(index));
        powEpisode.setDataSource(dataSource);
        
        powEpisode.setCaptureDate(DateUtils.getImpreciseDate(form.getPowCaptureDate(index)));
        powEpisode.setReleaseDate(DateUtils.getImpreciseDate(form.getPowReleaseDate(index)));
        
        Long days = DateUtils.getDays(form.getPowCaptureDate(index),form.getPowReleaseDate(index), true);
        powEpisode.setNumberOfDaysInCaptivity((days !=null) ? new Integer(days.intValue()) : null);
        
        POWCampLocation campLocation = (POWCampLocation)getLookup(POWCampLocation.class,form.getCampLocation(index));
        powEpisode.setCampLocation(campLocation);
        
        PrisonerOfWarLocation powLocation = (PrisonerOfWarLocation)getLookup(PrisonerOfWarLocation.class,form.getConfinementLocation(index));
        powEpisode.setConfinementLocation(powLocation);
        
        return powEpisode;
    }
    
    private SHADDocument ctrateNewSHADDocument(RegistrySHADInfoForm form, int index)
    throws ConversionServiceException
    {
        SHADDocument newSHADDoc = new SHADDocument();
        return updateSHADDocument(form,index,newSHADDoc);
    }
    
    private SHADDocument updateSHADDocument(RegistrySHADInfoForm form, int index,SHADDocument shadDoc)
    throws ConversionServiceException
    {
        SHADDocumentType documentType = (SHADDocumentType)getLookup(SHADDocumentType.class,form.getDocumentType(index));
        shadDoc.setDocumentType(documentType);
        shadDoc.setReceivedDate(DateUtils.getDate(form.getDocumentReceivedDate(index)));
        return shadDoc;
    }
    
    private void populateCommon(RegistryInfoForm form, Registry registry) 
    throws ConversionServiceException {
        if(registry != null) {
            RegistryTrait registryTrait = registry.getRegistryTrait();
            if(registryTrait == null) {
                registryTrait = new RegistryTrait();
                registry.setRegistryTrait(registryTrait);
            }
            RegistryType type = (RegistryType)getLookup(RegistryType.class,form.getRegistryType());
            registryTrait.setRegistryType(type);
            
            String ssn = SSN.formatSSN(form.getSsn());
            registryTrait.setSsn(StringUtils.isNotEmpty(ssn) ? ssn : null);
            
            String prefix = StringUtils.isNotEmpty(form.getPrefix()) ? form.getPrefix().toUpperCase().trim() : null;
            registryTrait.setPrefix(prefix);
            
            String firstName = StringUtils.isNotEmpty(form.getFirstName()) ? form.getFirstName().toUpperCase().trim() : null;
            registryTrait.setFirstName(firstName);
            
            String middleName = StringUtils.isNotEmpty(form.getMiddleName()) ? form.getMiddleName().toUpperCase().trim() : null;
            registryTrait.setMiddleName(middleName);
            
            String lastName = StringUtils.isNotEmpty(form.getLastName()) ? form.getLastName().toUpperCase().trim() : null;
            registryTrait.setLastName(lastName);
            
            String suffix = StringUtils.isNotEmpty(form.getSuffix()) ? form.getSuffix().toUpperCase().trim() : null;
            registryTrait.setSuffix(suffix);
            
            String degree = StringUtils.isNotEmpty(form.getDegree()) ? form.getDegree().toUpperCase().trim() : null;
            registryTrait.setEducationalDegree(degree);
            
            //Add Military Service Numbers.
            registryTrait.removeAllRegistryTraitDetails();
            List militaryServiveNumbers = form.getMilitaryServiceNumbersList();
            if(militaryServiveNumbers != null) {
                for(Iterator iter=militaryServiveNumbers.iterator(); iter.hasNext();) {
                    String militaryServiveNumber = (String)iter.next();
                    if(StringUtils.isNotEmpty(militaryServiveNumber)) {
                        RegistryTraitDetail detail = new RegistryTraitDetail();
                        detail.setMilitaryServiceNumber(militaryServiveNumber.trim());
                        registryTrait.addRegistryTraitDetail(detail);
                    }
                }
            }
        }
    }
    
    private void populateCommon(Registry registry, RegistryInfoForm form) 
    throws ConversionServiceException {
        form.setRegistryId(registry.getEntityKey().getKeyValueAsString());
        RegistryTrait trait = registry.getRegistryTrait();
        if(trait != null) {
            form.setSsn(JspUtils.displaySSN(trait.getSsn()));
            form.setPrefix(trait.getPrefix());
            form.setFirstName(trait.getFirstName());
            form.setMiddleName(trait.getMiddleName());
            form.setLastName(trait.getLastName());
            form.setSuffix(trait.getSuffix());
            form.setDegree(trait.getEducationalDegree());
            form.setRegistryType(this.getLookupCode(trait.getRegistryType()));
            Set traitDetails  = trait.getRegistryTraitDetails();
            
            if(traitDetails != null && !traitDetails.isEmpty()) {
                RegistryTraitDetail traitDetail  = null;
                List militaryServiceNumbers = form.getMilitaryServiceNumbersList();
                for(Iterator iter=traitDetails.iterator(); iter.hasNext();) {
                    traitDetail  = (RegistryTraitDetail)iter.next();
                    if(traitDetail != null) {
                        militaryServiceNumbers.add(traitDetail.getMilitaryServiceNumber());
                    }
                }
            }
        }
        
        Date updatedate = (registry.getModifiedOn() != null) ? registry.getModifiedOn() : registry.getCreatedOn();
        form.setUpdatedDate(DateUtils.format(updatedate,DateUtils.MMDDYYYYHHMMSS));
        
        Principal updatedBy = (registry.getModifiedBy() != null) ? registry.getModifiedBy() : registry.getCreatedBy();
        form.setUpdatedBy((updatedBy != null) ? updatedBy.getName() : null);
    }
    
//    private Lookup getLookup(Class clazz,String code)
//    throws ConversionServiceException {
//        Lookup lookup = null;
//        try {
//            if(StringUtils.isNotEmpty(code)) {
//                lookup = getLookupCacheService().getByCodeFromCache(clazz,code);
//            }
//        } catch(Exception ex) {
//            StringBuffer buffer = new StringBuffer();
//            buffer.append("Error while getting lookup object for ");
//            buffer.append(" type = " + clazz.getName());
//            buffer.append(" code = " + code);
//            throw new ConversionServiceException(buffer.toString(),ex);
//        }
//        return lookup;
//    }
    
    private String getYesNo(Boolean flag) {
        if(flag == null) return null;
        return flag.booleanValue() ? RegistryInfoForm.Y : RegistryInfoForm.N;
    }
    
    private String getYesNo(Indicator indicator) {
        if(indicator == null) return null;
        return this.getYesNo(indicator.toBoolean());
    }
    
    private String getIndicatorCode(String flag) {
        if(flag == null) return Indicator.UNKNOWN.getCode();
        return StringUtils.equals(flag,RegistryInfoForm.Y) ? Indicator.YES.getCode() : Indicator.NO.getCode();
    }
    
    private Boolean getTrueFalse(String flag) {
        if(StringUtils.isEmpty(flag)) return null;
        return StringUtils.equals(flag,RegistryInfoForm.Y) ? Boolean.TRUE :Boolean.FALSE;
    }
    
    private Map getMapFromCollection(Set set) {
        Map map = new HashMap();
        if(set != null) {
           AbstractKeyedEntity keyedEntity = null;
           for(Iterator iter = set.iterator(); iter.hasNext();) {
              keyedEntity = (AbstractKeyedEntity)iter.next();
              map.put(keyedEntity.getEntityKey().getKeyValueAsString(),keyedEntity);
           }
        }
        return map;
    }
}