/********************************************************************
 * Copyright  2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.ui.egt.action;

// Java Classes

// Library Classes

// Framework Classes
import gov.va.med.fw.conversion.ConversionServiceException;
import gov.va.med.fw.util.StringUtils;

// ESR Classes
import gov.va.med.esr.common.model.ee.EGTSetting;
import gov.va.med.esr.common.model.lookup.EGTSettingType;
import gov.va.med.esr.common.model.lookup.EnrollmentPriorityGroup;
import gov.va.med.esr.common.model.lookup.EnrollmentPrioritySubGroup;
import gov.va.med.esr.ui.common.service.LookupCacheService;
import gov.va.med.esr.ui.common.util.DateUtils;
import gov.va.med.esr.ui.conversion.UIConversionServiceImpl;

/**
 * The conversion service for the EGT Setting screens.
 *
 * @author Andrew Pach
 * @version 3.0
 */
public class EGTSettingConversionService extends UIConversionServiceImpl
{
    /**
     * Default constructor.
     */
    public EGTSettingConversionService()
    {
        super();
    }

    /**
     * Converts between an EGTSetting object and the EGTSettingInfoForm.
     *
     * @param source the source object
     * @param target the target object
     *
     * @throws gov.va.med.fw.conversion.ConversionServiceException if any errors were encountered during the
     * conversion.
     */
    protected void convertBean(Object source, Object target) throws ConversionServiceException
    {
        EGTSetting egtSetting = null;
        EGTSettingInfoForm form = null;

        if (source instanceof EGTSetting)
        {
            egtSetting = (EGTSetting)source;
            if (target instanceof EGTSettingInfoForm)
            {
                form = (EGTSettingInfoForm)target;
            }
            else
            {
                throw new ConversionServiceException("Target object is of type " + target.getClass().getName() +
                    " and must be of type EGTSettingInfoForm.");
            }

            // Convert from the Person to the form
            convertBusinessObjectToForm(egtSetting, form);
        }
        else
        {
            if (source instanceof EGTSettingInfoForm)
            {
                form = (EGTSettingInfoForm)source;
                if (target instanceof EGTSetting)
                {
                    egtSetting = (EGTSetting)target;
                }
                else
                {
                    throw new ConversionServiceException("Target object is of type " + target.getClass().getName() +
                        " and must be of type EGTSetting.");
                }

                // Convert from the Form to Person
                convertFormToBusinessObject(form, egtSetting);
            }
            else
            {
                throw new ConversionServiceException("Source object is of type " + target.getClass().getName() +
                    " and must be of type EGTSetting or EGTSettingInfoForm.");
            }
        }
    }

    /**
     * Converts the EGT Setting data from the EGTSetting object to the form.
     *
     * @param egtSetting The EGTSetting
     * @param form The form
     *
     * @throws gov.va.med.fw.conversion.ConversionServiceException If there were any problems converting the person to the form.
     */
    protected void convertBusinessObjectToForm(EGTSetting egtSetting, EGTSettingInfoForm form) throws ConversionServiceException
    {
        try
        {
            form.setEgtSettingId(egtSetting.getEntityKey().getKeyValueAsString());
            form.setEgtType(egtSetting.getType() != null ? egtSetting.getType().getCode() : null);
            form.setEgtPriorityGroup(egtSetting.getPriorityGroup() != null ?
                egtSetting.getPriorityGroup().getCode() : null);
            form.setEgtPrioritySubGroup(egtSetting.getPrioritySubGroup() != null ?
                egtSetting.getPrioritySubGroup().getCode() : null);
            form.setEgtEffectiveDate(DateUtils.format(egtSetting.getEffectiveDate(), null));
            form.setFederalRegisterDate(DateUtils.format(egtSetting.getFederalRegisterDate(), null));
            form.setComments(egtSetting.getComments());
        }
        catch (Exception ex)
        {
            throw new ConversionServiceException("Unable to convert EGTSetting to EGTSettingInfoForm.", ex);
        }
    }

    /**
     * Converts the form data to the EGTSetting object.  Note that the disable flag is set to True in this method.
     *
     * @param form The form
     * @param egtSetting The EGTSetting
     *
     * @throws gov.va.med.fw.conversion.ConversionServiceException If there were any problems converting the form to the person.
     */
    protected void convertFormToBusinessObject(EGTSettingInfoForm form, EGTSetting egtSetting) throws ConversionServiceException
    {
        LookupCacheService lookupCacheService = getLookupCacheService();

        try
        {
            egtSetting.setType((EGTSettingType)lookupCacheService.getByCodeFromCache(EGTSettingType.class,
                form.getEgtType()));
            egtSetting.setPriorityGroup((EnrollmentPriorityGroup)lookupCacheService.getByCodeFromCache(
                EnrollmentPriorityGroup.class, form.getEgtPriorityGroup()));
            egtSetting.setPrioritySubGroup((EnrollmentPrioritySubGroup)lookupCacheService.getByCodeFromCache(
                EnrollmentPrioritySubGroup.class, form.getEgtPrioritySubGroup()));
            egtSetting.setEffectiveDate(DateUtils.getDate(form.getEgtEffectiveDate()));
            egtSetting.setFederalRegisterDate(DateUtils.getDate(form.getFederalRegisterDate()));
            egtSetting.setComments(StringUtils.isBlank(form.getComments()) ? null : form.getComments());
            egtSetting.setDisable(Boolean.FALSE);
        }
        catch (Exception ex)
        {
            throw new ConversionServiceException("Unable to convert EGTSettingInfoForm to EGTSetting.", ex);
        }
    }
}