/********************************************************************
 * Copyright  2006 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.ui.common.dao;

// Java Classes
import java.util.List;

import gov.va.med.fw.model.lookup.Lookup;

// ESR Classes
import gov.va.med.esr.service.UnknownLookupTypeException;
import gov.va.med.esr.service.LookupService;
import gov.va.med.esr.service.UnknownLookupCodeException;


/**
 * Abstract base class for all UI lookup DAOs'.
 *
 * @author Andrew Pach
 * @version 3.0
 */
public abstract class UIAbstractLookupDAO
{
    // Lookup service bean
    private LookupService lookupService = null;

    /**
     * Looks up and returns all objects for this DAO.
     *
     * @return The list of objects.
     * @throws UnknownLookupTypeException
     * @throws ClassNotFoundException
     */
    public abstract List findAll() throws UnknownLookupTypeException, ClassNotFoundException;

    /**
     * Returns the base lookup class that this UI DAO extends.  Sub-classes should override this method if they extend a
     * base DAO and filter results.  This method returns null by default.
     *
     * @return null
     */
    public Class getBaseLookupClass()
    {
        return null;
    }

    /**
     * If a base lookup class is defined, then the code is looked up from the lookup service using this base class
     * lookup.  Otherwise, null is returned.
     *
     * @param className The classname
     * @param code the code to lookup
     *
     * @return the lookup associated with the code
     * @throws UnknownLookupTypeException if the lookup type is unknown
     * @throws ClassNotFoundException if the class was not found
     * @throws UnknownLookupCodeException if the code was not found
     */
    public Lookup getByCode(String className, String code)
        throws UnknownLookupTypeException, ClassNotFoundException, UnknownLookupCodeException
    {
        return (getBaseLookupClass() == null) ? null : lookupService.getByCode(getBaseLookupClass(), code);
    }

    public LookupService getLookupService()
    {
        return lookupService;
    }

    public void setLookupService(LookupService lookupService)
    {
        this.lookupService = lookupService;
    }
}
