/********************************************************************
 * Copyright  2004-2006 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.ui.common.action;

// Java Classes
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;



// Library Classes
import org.apache.commons.lang.Validate;



// Framework Classes
import gov.va.med.fw.model.EntityKey;

// ESR Classes
import gov.va.med.esr.common.model.CommonEntityKeyFactory;
import gov.va.med.esr.common.model.ee.AgentOrangeExposure;
import gov.va.med.esr.common.model.ee.CampLejeuneVerificationMethod;
import gov.va.med.esr.common.model.ee.CancelDecline;
import gov.va.med.esr.common.model.ee.EnrollmentDetermination;
import gov.va.med.esr.common.model.ee.EnvironmentalContaminationExposure;
import gov.va.med.esr.common.model.ee.MedalOfHonor;
import gov.va.med.esr.common.model.ee.RadiationExposure;
import gov.va.med.esr.common.model.ee.SpecialFactor;
import gov.va.med.esr.common.model.financials.DependentFinancials;
import gov.va.med.esr.common.model.financials.FinancialStatement;
import gov.va.med.esr.common.model.financials.IncomeTest;
import gov.va.med.esr.common.model.financials.SpouseFinancials;
import gov.va.med.esr.common.model.lookup.CampLejeuneVerificationType;
import gov.va.med.esr.common.model.lookup.EmergencyResponse;
import gov.va.med.esr.common.model.lookup.EnrollmentPriorityGroup;
import gov.va.med.esr.common.model.lookup.EnrollmentPrioritySubGroup;
import gov.va.med.esr.common.model.person.EmergencyResponseIndicator;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.common.model.person.Spouse;
import gov.va.med.esr.common.model.ee.CampLejeuneVerification;

/**
 * Generic Action class to implement history screens
 * @author DNS   KATIKM and Andrew Pach
 * @version 3.0
 */
public class PersonHistoryAction extends HistoryAction
{
    public PersonHistoryAction()
    {
        super();
    }



    /**
     * Return the Person entity key that is needed to get history on.
     *
     * @param request The HttpServletRequest
     *
     * @return The PersonEntityKey for the currently selected Person.
     * @throws Exception if any problems occur.
     */
    public EntityKey getEntityKey(HttpServletRequest request, String className) throws Exception
    {
        String personId = getSelectedPersonId(request);
        Validate.notNull(personId);
        return CommonEntityKeyFactory.createPersonIdEntityKey(personId);
    }

    /**
     * Returns true.
     *
     * @return true.
     */
    protected boolean requiresSelectedPerson()
    {
        return true;
    }

    /* Eligibility Enrollment */

    /**
     * Custom conversion method for Enrollment Ppriority
     *
     * @param enroll
     *
     * @return the enrollment priority
     */
    public String getEnrollmentPriority(EnrollmentDetermination enroll)
    {
        if (enroll != null && enroll.getPriorityGroup() != null)
        {
            String priority = enroll.getPriorityGroup().getDescription();

            if (enroll.getPrioritySubGroup() != null)
                priority += enroll.getPrioritySubGroup().getDescription();
            priority = priority.replaceAll("Enrollment Priority sub-Group", "");
            return priority.replaceAll("'", "");
        }
        return null;
    }

    /**
     * Custom conversion method for EGT Ppriority
     *
     * @param enrol
     *
     * @return the EGT Priority
     */
    public String getEGTPriority(EnrollmentDetermination enrol)
    {
        if (enrol != null && enrol.getEgtSetting() != null)
        {
            EnrollmentPriorityGroup group = enrol.getEgtSetting().getPriorityGroup();
            EnrollmentPrioritySubGroup subgroup = enrol.getEgtSetting().getPrioritySubGroup();
            String priority = group != null ? group.getDescription() : "";
            if (subgroup != null)
                priority += subgroup.getDescription();

            priority = priority.replaceAll("Enrollment Priority sub-Group '", "");
            return priority.replaceAll("'", "");
        }
        return null;
    }

    /**
     * Custom conversion method for Health Benefit Plans
     *
     * @param hbpSet
     *
     * @return the current number of Health Benefit Plans
     */

    public Integer getCount(Set hbpSet)
    {
    	return hbpSet.size();
    }


    /**
     * Gets the cancel/decline indicator
     *
     * @param person
     *
     * @return The cancel/decline indicator
     */
    public String getCancelDeclineIndicator(Person person)
    {
        CancelDecline cancelDecline = person.getCancelDecline();
        if (cancelDecline == null || cancelDecline.getCancelDeclineIndicator() == null)
            return TEXT_NO_DATA;
        return Boolean.TRUE.equals(cancelDecline.getCancelDeclineIndicator()) ? TEXT_YES : TEXT_NO;
    }
    /* Finanlcials */

    /**
     * Get the income test for the selected year
     *
     * @param person
     *
     * @return the income test
     */
    public IncomeTest getIncomeTest(Person person)
    {
        if (person == null)
        {
            return null;
        }

        Map incomeTests = person.getIncomeTests();
        //Return last calander year by default
        if (incomeTests != null && incomeTests.size() > 0)
        {
            Set keys = incomeTests.keySet();
            Object incomeYear = keys.iterator().next();
            return (IncomeTest)incomeTests.get(incomeYear);
        }

        return null;
    }

    /**
     * Get the financial statement for the selected year
     *
     * @param person
     *
     * @return the financial statement
     */
    public FinancialStatement getFinancialStatement(Person person)
    {
        if (person == null)
        {
            return null;
        }

        Map FinancialStatements = person.getFinancialStatements();
        //Return last calander year by default
        if (FinancialStatements != null && FinancialStatements.size() > 0)
        {
            Set keys = FinancialStatements.keySet();
            Object incomeYear = keys.iterator().next();
            return (FinancialStatement)FinancialStatements.get(incomeYear);
        }

        return null;
    }

    public Set getSpouses(Person person)
    {
        Set spouses = new HashSet();
        if (person != null) {
            Set sfs = getSpouseFinancials(person);
            for (Iterator i= sfs.iterator(); i.hasNext();){
                spouses.add(((SpouseFinancials)i.next()).getReportedOn());
            }
        }
        return spouses;
    }

    public Set getSpouseFinancials(Person person)
    {
        if (person == null) return new HashSet();
        FinancialStatement stmt = getFinancialStatement(person);
        return (stmt != null) ? stmt.getSpouseFinancials() : new HashSet();
    }

    public Set getDependents(Person person)
    {
        Set dependents = new HashSet();
        if (person == null)
        {
            return dependents;
        }

        Set depFinancials = getDependentFinancials(person);
        for (Iterator i = depFinancials.iterator(); i.hasNext();)
        {
            DependentFinancials financials = (DependentFinancials)i.next();
            if (financials.getReportedOn() != null)
                dependents.add(financials.getReportedOn());
        }
        return dependents;
    }

    public Set getDependentFinancials(Person person)
    {
        if (person == null)
        {
            return new HashSet();
        }
        FinancialStatement stmt = getFinancialStatement(person);
        return (stmt != null) ? stmt.getDependentFinancials() : new HashSet();
    }

    public Set getBeneficiaryTravels(Person person)
    {
        Map map = person.getBeneficiaryTravels();
        if (map == null || map.size() == 0) return null;
        return new HashSet(map.values());
    }

    public Set getIncomeTestStatuses(Person person)
    {
        IncomeTest incomeTest = getIncomeTest(person);
        return incomeTest == null ? null : incomeTest.getStatuses();
    }

    public EmergencyResponse getEmergencyResponse(Person person) {
        if(person == null) return null;

        Set emerResps = person.getEmergencyResponseIndicators();
        EmergencyResponseIndicator emerRespInd = (emerResps != null && !emerResps.isEmpty()) ? (EmergencyResponseIndicator)emerResps.iterator().next() : null;
        return (emerRespInd != null) ? emerRespInd.getEmergencyResponse() : null;
    }

    /* Military Service special factors */
    public SpecialFactor getAgentOrangeExposure(Person person)
    {
        if (person == null)
        {
            return null;
        }
        return person.getSpecialFactorByType(AgentOrangeExposure.class);
    }

    public SpecialFactor getRadiationExposure(Person person)
    {
        if (person == null)
        {
            return null;
        }
        return person.getSpecialFactorByType(RadiationExposure.class);
    }

    public String isSHAD(Person person)
    {
        return (person != null && person.getShad() != null && person.getShad().getShadIndicator() != null) ? person.getShad().getShadIndicator().getName() : null;
    }

    public MedalOfHonor isMedalOfHonor(Person person)
    {
        return (person != null && person.getMedalOfHonor() != null ? person.getMedalOfHonor() : null);
    }


    public String isEnvironmentalContaminationExposure(Person person)
    {
        if (person == null)
        {
            return null;
        }
        EnvironmentalContaminationExposure env = (EnvironmentalContaminationExposure)person.getSpecialFactorByType(EnvironmentalContaminationExposure.class);
        return (env != null && env.getEnvironmentalContaminationExposureIndicator() != null) ? env.getEnvironmentalContaminationExposureIndicator().getName() : null;
    }

	 public String getDeathVerificationDisplay(Person person)
	  {
		 String deathVerificationDisplay = null;
	    	 if (person.getDeathRecord().getDodVerifiedInd() == null || person.getDeathRecord().getDodVerifiedInd())
	    	 {
	    		  	deathVerificationDisplay = "VERIFIED";
	    		  } else {
	    			  deathVerificationDisplay = "UNVERIFIED";
	    		  }
	    	 return deathVerificationDisplay;
	    }

}
