/********************************************************************
 * Copyright  2005 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.ui.common.action;

// Java Classes

// Library Classes

// Framework Classes
import gov.va.med.fw.util.StringUtils;
import gov.va.med.fw.conversion.ConversionServiceException;

// ESR Classes
import gov.va.med.esr.common.model.lookup.Country;
import gov.va.med.esr.common.model.party.Address;
import gov.va.med.esr.ui.conversion.UIConversionServiceImpl;

/**
 * This class provides methods to convert the Address data between an Address object and an AddressForm.
 *
 * @author Andrew Pach
 * @version 3.0
 */
public class AddressConversionService extends UIConversionServiceImpl
{
    /**
     * Default constructor.
     */
    public AddressConversionService()
    {
    }

    /**
     * Converts between an Address object and the AddressForm.
     *
     * @param source the source object
     * @param target the target object
     *
     * @throws gov.va.med.fw.conversion.ConversionServiceException if any errors were encountered during the
     * conversion.
     */
    protected void convertBean(Object source, Object target) throws ConversionServiceException
    {
        Address address = null;
        AddressForm form = null;

        if (source instanceof Address)
        {
            address = (Address)source;
            if (AddressForm.class.isAssignableFrom(target.getClass()))
            {
                form = (AddressForm)target;
            }
            else
            {
                throw new ConversionServiceException("Target object is of type " + target.getClass().getName() +
                    " and must be of type AddressForm.");
            }

            // Convert from the Address to the form
            convertAddressToForm(address, form);
        }
        else
        {
            if (source instanceof AddressForm)
            {
                form = (AddressForm)source;
                if (Address.class.isAssignableFrom(target.getClass()))
                {
                    address = (Address)target;
                }
                else
                {
                    throw new ConversionServiceException("Target object is of type " + target.getClass().getName() +
                        " and must be of type Address.");
                }

                // Convert from the Form to Person
                convertFormToAddress(form, address);
            }
            else
            {
                throw new ConversionServiceException("Source object is of type " + target.getClass().getName() +
                    " and must be of type Address or AddressForm.");
            }
        }
    }

    /**
     * Converts the Address data onto the form.
     *
     * @param address The address
     * @param form The form
     */
    protected void convertAddressToForm(Address address, AddressForm form)
    {
        if (address != null)
        {
            // Convert the standard fields
            form.setAddressLine1(address.getLine1());
            form.setAddressLine2(address.getLine2());
            form.setAddressLine3(address.getLine3());
            form.setCity(address.getCity());
            if (address.getCountry() != null)
            {
                form.setCountry(address.getCountry());
            }

            // Convert US and non-US specific fields
            if (form.isUSAddress())
            {
                form.setState(address.getState());
                form.setCounty(address.getCounty());
                form.setPrevZip(address.getZipCode());
                form.setZipCode(address.getZipCode());
                form.setZipPlus4(address.getZipPlus4());
            }
            else
            {
                form.setProvince(address.getProvince());
                form.setPostalCode(address.getPostalCode());
            }
        }
    }

    /**
     * Converts the address form data onto the address object.
     *
     * @param form The form
     * @param address The address
     */
    protected void convertFormToAddress(AddressForm form, Address address)
    {
        // Convert the standard fields
        address.setCountry(StringUtils.isBlank(form.getCountry()) ? null : form.getCountry());
        address.setLine1(StringUtils.isBlank(form.getAddressLine1()) ? null : form.getAddressLine1());
        address.setLine2(StringUtils.isBlank(form.getAddressLine2()) ? null : form.getAddressLine2());
        address.setLine3(StringUtils.isBlank(form.getAddressLine3()) ? null : form.getAddressLine3());
        address.setCity(StringUtils.isBlank(form.getCity()) ? null : form.getCity());
        
        try {
	        String countryCode = form.getCountry();
	        Country newCountry = (Country)this.getLookupCacheService().getLookupService().getCountryByCode(countryCode);
			String countryShortName = newCountry.getShortName();
			Country countryObject = new Country();
			countryObject.setShortName(countryShortName);
			address.setCountryObject(countryObject);
        } catch (Exception e) {
			// This should never happen
        	logger.error("lookup service is not avaliable. Error= " + e);
		}

        // Convert US and non-US specific fields and null out the fields that don't apply
        if (form.isUSAddress())
        {
            address.setCounty(StringUtils.isBlank(form.getCounty()) ? null : form.getCounty());
            address.setState(StringUtils.isBlank(form.getState()) ? null : form.getState());
            address.setZipCode(StringUtils.isBlank(form.getZipCode()) ? null : form.getZipCode());
            address.setZipPlus4(StringUtils.isBlank(form.getZipPlus4()) ? null : form.getZipPlus4());
            address.setProvince(null);
            address.setPostalCode(null);
        }
        else
        {
            address.setCounty(null);
            address.setState(null);
            address.setZipCode(null);
            address.setZipPlus4(null);
            address.setProvince(StringUtils.isBlank(form.getProvince()) ? null : form.getProvince());
            address.setPostalCode(StringUtils.isBlank(form.getPostalCode()) ? null : form.getPostalCode());
        }
    }
}