/********************************************************************
 * Copyright  2004 VHA. All rights reserved
 ********************************************************************/
package gov.va.med.esr.ui.admin.action;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.Validate;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;

import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.model.EntityKeyFactory;

import gov.va.med.esr.common.model.security.ESRRolePrincipalImpl;

import gov.va.med.esr.ui.common.beans.FieldHistoryInfo;
import gov.va.med.esr.ui.common.util.DateUtils;

public class RolesAction extends AbstractUserProfileAction {

    public static final String FWD_DISPLAY_ROLES = "displayRoles";
    public static final String FWD_EDIT = "editRole";
    public static final String FWD_CONFIRM = "confirm";
    public static final String PAR_ROLE_ID = "roleId";
    public static final String ATTR_ROLES = "Roles";
    public static final String ATTR_DELETED_ROLES = "DeletedRoles";
    
    public static final String MSG_ASSIGNED_CAN_NOT_BE_DELETED ="error.roleisAssigned";
    public static final String MSG_ROLE_REQUIRES = "error.roleRequires";
    public static final String MSG_ROLE_NAME_DUPLICATE = "error.roleNameDuplicate";
    public static final String FIELD_NAME = "name";
    public static final String MSG_ACTIVE_DATE_INVALID = "error.activeDate";
    public static final String FLD_ACTIVE_DATE = "activeDate";
    public static final String FLD_INACTIVE_DATE = "inactiveDate";

        
    public RolesAction() {
        super();
    }
    
    public ActionForward display(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        clearSessionData(request);
        //sort roles by name
        List allRoles= getUserAdminService().getAllRoles();
        Collections.sort(allRoles);
        List deletedRoles= getUserAdminService().findDeletedRoles();
        Collections.sort(deletedRoles);        
        request.setAttribute(ATTR_ROLES,allRoles);
        request.setAttribute(ATTR_DELETED_ROLES,deletedRoles);
        return mapping.findForward(FWD_DISPLAY_ROLES);
    }
    
    public ActionForward addNew(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        //we don't need to reset the form as it is request scoped        
        RoleForm roleForm = (RoleForm) form;
        
        //set available capabilities and sets 
        roleForm.setAvailableCapabilities(getUserAdminService().getAllCapabilities());
        roleForm.setAvailableCapabilitySets(getUserAdminService().getAllCapabilitySets());
        populateReferenceData(request);
        return mapping.findForward(FWD_EDIT);
    }
    
    public ActionForward edit(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        //we don't need to reset the form as it is request scoped        
        String roleId = request.getParameter(PAR_ROLE_ID);
        Validate.notNull(roleId,"Role Id can not be null");
        
        //Get Admin service and Retrieve the Capability Set
        //Create Entity Key
        EntityKey entityKey = EntityKeyFactory.createEntityKey(
            new BigDecimal(roleId), ESRRolePrincipalImpl.class);
        ESRRolePrincipalImpl role = 
            (ESRRolePrincipalImpl) getUserAdminService().getRoleById(entityKey);

        //place the pristine entity in the session
        setPristineEntity(request,role);

        //Convert the Role to the form
        conversionService.convert(role,form);
        populateReferenceData(request);
        return mapping.findForward(FWD_EDIT);
    }    
    
    public ActionForward add(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        //validate from if required 
        //Validate whether the name already exists in the database
        
        String roleName = ((RoleForm)form).getName().trim();
        if (getUserAdminService().getRoleByName(roleName) != null) {
            addActionMessageForField(request,MSG_ROLE_NAME_DUPLICATE,FIELD_NAME);
            return mapping.getInputForward();
        }        
        ESRRolePrincipalImpl newRole = new ESRRolePrincipalImpl();
        conversionService.convert(form,newRole);
        
        setPristineEntity(request,null);
        setUpdatedEntity(request,newRole);
        if (isValid(request,newRole)) {
            //Forward to confirmation page
            return confirmation(mapping,form,request,response);           
        }
        return mapping.getInputForward();
    }
    
    public ActionForward delete(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        //we don't need to reset the form as it is request scoped        
        String roleId = request.getParameter(PAR_ROLE_ID);
        Validate.notNull(roleId,"Role Id can not be null");
        
        //Create Entity key from Id and get CapabilitySet from admin service
        EntityKey entityKey = EntityKeyFactory.createEntityKey(
            new BigDecimal(roleId), ESRRolePrincipalImpl.class);
        ESRRolePrincipalImpl role = 
            (ESRRolePrincipalImpl) getUserAdminService().getRoleById(entityKey);

        //Verify that Capability set is not assigned to any user
        if (Boolean.TRUE.equals(role.getAssigned())) {
            //Set error message that it cann't be deleted
            addActionMessage(request,MSG_ASSIGNED_CAN_NOT_BE_DELETED);
            return mapping.getInputForward();            
        }
        else {
            //delete the role
            getUserAdminService().deleteRole(role);
            return mapping.findForward(FWD_DISPLAY_DELETE_MSG);
        }        
    }
    
    public ActionForward update(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        //validate from
        RoleForm roleForm = (RoleForm) form;
        
        ESRRolePrincipalImpl pristine = (ESRRolePrincipalImpl) getPristineEntity(request);
        ESRRolePrincipalImpl updated = (ESRRolePrincipalImpl) pristine.clone();
        
        //convert
        conversionService.convert(roleForm, updated);
        setUpdatedEntity(request,updated);

        if (isValid(request,updated)) {
            //Forward to confirmation page
            return confirmation(mapping,form,request,response);           
        }
        return mapping.getInputForward();   
    }
    public ActionForward accept(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        //Get the updated capability set from the session
        ESRRolePrincipalImpl role = (ESRRolePrincipalImpl)getUpdatedEntity(request);
        
        if (role.getEntityKey() == null ) {
            //add
            getUserAdminService().addRole(role);
            return mapping.findForward(FWD_DISPLAY_ADD_MSG);
        }
        else {
            //update
            getUserAdminService().updateRole(role);
            return mapping.findForward(FWD_DISPLAY_UPDATE_MSG);
        }
    }
    
    public ActionForward revise(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        //Retrive the capability set from the session and forward to edit screen
        ESRRolePrincipalImpl role = (ESRRolePrincipalImpl)getUpdatedEntity(request);        
        conversionService.convert(role,form);           
        return mapping.findForward(FWD_EDIT);
    }

    public ActionForward discard(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) throws Exception
    {
        //return the display list discarding all changes     
        return mapping.findForward(FWD_DISPLAY_ROLES);
    }
    
    public void afterPropertiesSet() throws Exception {
    }
    
    private boolean isValid(HttpServletRequest request, ESRRolePrincipalImpl role) {
        boolean valid = true;
        //Check for Minimum number of required capabilities and or sets
        /*  No requirement for minimum number Capabilities or Capability Sets        
        int items = role.getCapabilities().size() + role.getCapabilitySets().size();
        if ( items == 0) {
            addActionMessage(request,MSG_ROLE_REQUIRES);
            valid = false;
        }  */      
        Date current = DateUtils.getCurrentDate();
        Date prevActiveDate = null;
        Date prevInactiveDate = null;
        if (role.getEntityKey() != null) {
        	ESRRolePrincipalImpl pristine = (ESRRolePrincipalImpl) getPristineEntity(request);
        	prevActiveDate = pristine.getActiveDate();
        	prevInactiveDate = pristine.getInactiveDate();
        }
        //date is not null and is different from previous value
        if (role.getActiveDate() != null && !role.getActiveDate().equals(prevActiveDate)) {        	
            if (role.getActiveDate().before(current)){
            	valid = false;
            	addActionMessageForField(request,ERRORS_ACTIVE_DATE_PAST,FLD_ACTIVE_DATE);   
            }
        }
        //date is not null and is different from previous value
        if (role.getInactiveDate() != null && !role.getInactiveDate().equals(prevInactiveDate)) { 
            if (role.getInactiveDate().before(current)){
            	valid = false;
            	addActionMessageForField(request,ERRORS_INACTIVE_DATE_PAST,FLD_INACTIVE_DATE);  
            }
        }
        
        //active date should be before the inactive date
        if (role.getActiveDate() != null && role.getInactiveDate() != null &&
                role.getActiveDate().after(role.getInactiveDate())) {
            valid = false;
            addActionMessageForField(request,MSG_ACTIVE_DATE_INVALID,FLD_ACTIVE_DATE);
            addActionMessageForField(request,MSG_ACTIVE_DATE_INVALID,FLD_INACTIVE_DATE);                        
        }
        return valid;
    }
    
    protected ActionForward confirmation(ActionMapping mapping, ActionForm form,
            HttpServletRequest request, HttpServletResponse response) {
        //convert to confirmation properties
        ESRRolePrincipalImpl pristine = 
            (ESRRolePrincipalImpl) getPristineEntity(request);
        ESRRolePrincipalImpl current = 
            (ESRRolePrincipalImpl) getUpdatedEntity(request);
                        
        List propList = new ArrayList ();
        
        //User Attributes (name) and Functional Group
        propList.add(new FieldHistoryInfo("label.name",            
            (current == null ? "" : current.getName()),
            (pristine == null ? "" : pristine.getName())));
        propList.add(new FieldHistoryInfo("label.description",            
                (current == null ? "" : current.getDescription()),
                (pristine == null ? "" : pristine.getDescription())));        

        propList.add(new FieldHistoryInfo("label.activeDate",
            (current == null ? "" : DateUtils.format(current.getActiveDate(),null)),
            (pristine == null ? "" : DateUtils.format(pristine.getActiveDate(),null))));
            
        propList.add(new FieldHistoryInfo("label.inactivedate",
           (current == null ? "" : DateUtils.format(current.getInactiveDate(),null)),
           (pristine == null ? "" : DateUtils.format(pristine.getInactiveDate(),null)))); 

        List list = matchSets(current.getCapabilitySets(),
                pristine == null ? null : pristine.getCapabilitySets());
        propList.addAll(processCapabilitySets(list));  
        
        list = matchSets(current.getCapabilities(),
                pristine == null ? null : pristine.getCapabilities());
        propList.addAll(processCapabilities(list));        
        
        //setForm.setCapabilityChangeList(orderedList);
        //Set attributes on the request for action,title and mode
        request.setAttribute("changeList",propList);
                
        request.setAttribute("confirmationActionName","editRole");
        request.setAttribute("confirmationEntityId",
                pristine == null ? "" : pristine.getEntityKey().getKeyValueAsString());       
        if (pristine == null) {
            request.setAttribute("confirmationTitle","label.addRole");
        }
        else {
            request.setAttribute("confirmationTitle","label.editRole");
        }

        return mapping.findForward(FWD_CONFIRM);
    }
    
    protected Map getKeyMethodMap()
    {
        Map map = new HashMap();
        map.put("button.display", "display");
        map.put("button.add", "add");
        map.put("button.update", "update");
        map.put("button.delete", "delete");
        map.put("button.cancel", "discard");
        map.put("button.addRole", "addNew");
        map.put("button.edit", "edit");
        map.put("button.revise", "revise");
        map.put("button.discardChanges", "discard");
        map.put("button.acceptChanges", "accept");
        map.put("button.returnToEdit", "returnToEdit");
        return map;
    }
}
