/********************************************************************
 * Copyright  2004 EDS. All rights reserved
 ********************************************************************/
//Package
package gov.va.med.esr.messaging.service.retransmission;

//Java Classes
import java.util.List;

import gov.va.med.esr.common.model.lookup.MessageStatus;
import gov.va.med.esr.common.model.system.SystemParameter;
import gov.va.med.esr.common.persistent.messaging.MessageLogEntryDAO;
import gov.va.med.esr.service.SystemParameterService;
import gov.va.med.fw.persistent.DAOException;
import gov.va.med.fw.service.ServiceException;

//Common Classes

/**
 * Calls the Messa geLogEntryDAO to retieve all MessageLogEntry records that are
 * awaiting acknowledgement for more than 2 days and have not been retransmitted
 * more than a specified number of times. The number of days waiting
 * acknowledgment and the retranmisison count are configurable
 * 
 * 
 * @author Alex Yoon
 * @version 1.0
 * 
 */
public class RetransmissionMessageLogFinder implements MessageLogFinder {
	private MessageLogEntryDAO messageLogEntryDAO;

	private boolean isEGTRetransmission;

	private SystemParameterService systemParameterService;

	/**
	 * Default constructor.
	 */
	public RetransmissionMessageLogFinder() {
		super();
	}

	/**
	 * @return Returns the systemParameterService.
	 */
	public SystemParameterService getSystemParameterService() {
		return systemParameterService;
	}

	/**
	 * @param systemParameterService
	 *           The systemParameterService to set.
	 */
	public void setSystemParameterService(
			SystemParameterService systemParameterService) {
		this.systemParameterService = systemParameterService;
	}

	/**
	 * @return Returns the isEGTRetransmission.
	 */
	public boolean isEGTRetransmission() {
		return isEGTRetransmission;
	}

	/**
	 * @param isEGTRetransmission
	 *           The isEGTRetransmission to set.
	 */
	public void setEGTRetransmission(boolean isEGTRetransmission) {
		this.isEGTRetransmission = isEGTRetransmission;
	}

	/**
	 * Method to find message log entries.
	 * 
	 * @throws ServiceException
	 * @return Iterator that contains MessageLogEntry objects.
	 */
	public List find() throws ServiceException {
		// how long the system shall wait for an acknowledgement before
		// re-transmitting the message.
		String retransmitDays = null;
		// total number of times the system shall re-transmit a message that has
		// not been acknowledged
		String maxRetransmissionCount = null;

		try {
			if (isEGTRetransmission()) {
				retransmitDays = systemParameterService.getByName(
						SystemParameter.RETRANSMIT_DAYS_EGT).getValue();
				maxRetransmissionCount = systemParameterService.getByName(
						SystemParameter.MAXIMUM_RETRANSMITION_COUNT_EGT).getValue();
			}
			else {
				retransmitDays = systemParameterService.getByName(
						SystemParameter.RETRANSMIT_DAYS).getValue();
				maxRetransmissionCount = systemParameterService.getByName(
						SystemParameter.MAXIMUM_RETRANSMITION_COUNT).getValue();
			}

			List messageLogEntries = messageLogEntryDAO
					.findByRetransmissionWaitPeriodAndCount(new Integer(
							retransmitDays), new Integer(maxRetransmissionCount),
							MessageStatus.AWAITING_ACKNOWLEDGEMENT.getCode(),
							isEGTRetransmission());

			return messageLogEntries;

		}
		catch (DAOException e) {
			throw new ServiceException(
					"Exception getting list of MessageLogEntryDAOs by RetransmissionWaitPeriod and Count",
					e);
		}
	}

	/**
	 * @return Returns the messageLogEntryDAO.
	 */
	public MessageLogEntryDAO getMessageLogEntryDAO() {
		return messageLogEntryDAO;
	}

	/**
	 * @param messageLogEntryDAO
	 *           The messageLogEntryDAO to set.
	 */
	public void setMessageLogEntryDAO(MessageLogEntryDAO messageLogEntryDAO) {
		this.messageLogEntryDAO = messageLogEntryDAO;
	}

}