/********************************************************************
 * Copyright  2004 EDS. All rights reserved
 ********************************************************************/
//Package
package gov.va.med.esr.messaging.service.retransmission;

// Java classes
import java.util.Properties;

import gov.va.med.fw.service.AbstractComponent;
import gov.va.med.fw.service.ConfigurationConstants;
import gov.va.med.fw.service.ServiceConfigurationException;
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.service.jms.QueueMessageProducer;

import gov.va.med.esr.common.model.messaging.MessageLogEntry;
import gov.va.med.esr.service.trigger.RetransmitTriggerEvent;

// Messaging classes

/**
 * Publishes a controler ID extracted from a message log entry.  An ID 
 * represents an ID of a message that needs to be retrasmitted to a 
 * client.
 * 
 * @author Vu Le
 * @version 1.0
 */
public class RetransmissionIDPublisher extends AbstractComponent implements MessageResender {
   
    /**
     * A message producer that uses JMS Message producer
     * implementation class to send a message
     */
    private QueueMessageProducer producer = null;
    
    private Properties publisherProperties = null; 
    	
    
    /**
     * Default constructor.
     */
    public RetransmissionIDPublisher() {
        super();
    }
    
    /**
     * Packages a message log's control id in a JMS message then send it to a JMS 
     * message queue. 
     *  
     * @param messageLog The MessageLogEntry object to use for resending a message.
     * @exception Thrown RetransmissionException if failed to resend
     */
    public void resend( MessageLogEntry messageLog ) throws ServiceException {
        
        String id = (messageLog != null ? messageLog.getControlIdentifier() : null);
        
        if( id == null ) {
            throw new RetransmissionException( "Invalid control id to send" + id );
        }
        
        try {
        	setPublisherproperties(TransmissionConstants.TRANSMISSION_ID_DESCRIPTOR);        	
        	producer.send( id, publisherProperties);
        }
        catch( ServiceException e ) {
            throw new RetransmissionException( "Failed to publish a message id to a queue", e );
        }
    }
    
    public void resend( String controlId ) throws ServiceException {
        if( controlId == null ) {
            throw new RetransmissionException( "Invalid control id to send" + controlId );
        }
        
        try {
        	setPublisherproperties(TransmissionConstants.TRANSMISSION_ID_DESCRIPTOR);        	
            producer.send( controlId, publisherProperties);
        }
        catch( ServiceException e ) {
            throw new RetransmissionException( "Failed to publish a message id to a queue", e );
        }    	
    }    
    
    public void resend( RetransmitTriggerEvent triggerEvent ) throws ServiceException {    
    	
        if( triggerEvent == null || (triggerEvent != null && triggerEvent.getMessageLogEntryId() == null)) {
            throw new RetransmissionException( "Invalid retransmit trigger event id to send" + triggerEvent );
        }
        
        try {
        	setPublisherproperties(TransmissionConstants.RETRANSMISSION_FROM_UI_DESCRIPTOR);
            producer.send( triggerEvent, publisherProperties);
        }
        catch( ServiceException e ) {
            throw new RetransmissionException( "Failed to publish a message id to a queue", e );
        }
    }
    
    /**
     * @return Returns the producer.
     */
    public QueueMessageProducer getProducer() {
        return producer;
    }
    
    /**
     * @param producer The producer to set.
     */
    public void setProducer( QueueMessageProducer producer) {
        this.producer = producer;
    }
    
    /**
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    public void afterPropertiesSet() throws Exception {
        if( this.producer == null ) {
            throw new ServiceConfigurationException( "Missing a required producer", null, getClass() );
        }
        
    }
    
    private void setPublisherproperties(String retransmissionType)
    {
        publisherProperties = new Properties();
        publisherProperties.remove(ConfigurationConstants.DEFAULT_MESSAGE_TYPE);
    	publisherProperties.put( ConfigurationConstants.DEFAULT_MESSAGE_TYPE, retransmissionType);
    }
}