/********************************************************************
 * Copyright  2004-2006 EDS. All rights reserved
 ********************************************************************/
package gov.va.med.esr.messaging.service.outbound;

// Java Classes
import java.util.Iterator;
import java.util.List;

// Framework Classes
import gov.va.med.fw.service.ServiceException;
import gov.va.med.fw.util.InvalidConfigurationException;

// ESR Classes
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.service.LookupService;
import gov.va.med.esr.service.trigger.EGTTriggerEvent;

/**
 * Adaptor to send out a MFN-ZEG message to a list of Vista sites to update changes to the EGT (Enrollment Group Threshold) setting.
 *
 * @author Alex Yoon
 * @author Andrew Pach
 * @version 3.0
 */
public class OutboundMessageMasterFileAdaptor extends AbstractOutboundServiceAdaptor
{
    private LookupService lookupService = null;

    /**
     * Default constructor.
     */
    public OutboundMessageMasterFileAdaptor()
    {
        super();
    }

    /**
     * Method to process an outbound message.
     *
     * @param input The input used to build a message.
     *
     * @throws ServiceException
     */
    public void processMessage(Object input) throws ServiceException
    {
        Object[] args = (Object[])input;
        this.processMessage(args[0], (String)args[1]);
    }

    /**
     * Method to process an outbound message. This method gets corresponding OutboundProcessService for message type and delegates actual creation of outbound messages to the appropriate service.
     *
     * @param obj The object used to build a message.
     * @param messageType The messageType used to build a message.
     *
     * @throws ServiceException
     */
    public void processMessage(Object obj, String messageType)
        throws ServiceException
    {
        if (! (obj instanceof EGTTriggerEvent))
        {
            throw new IllegalArgumentException("Invalid Trigger Event to send MFNZEG message");
        }

        OutboundMessageMasterFileService service = this.getOutboundMasterFileService(messageType);

        if (logger.isDebugEnabled())
        {
            logger.debug("Got a Service " + service.getClass().getName()
                + " for Message Type " + messageType);
        }

        // Get the list of MFN-ZEG facilities to send the message to
        List facilities = this.lookupService.getAllMfnzegRecipients();
        if ((facilities == null) || (facilities.size() == 0))
        {
            throw new ServiceException("No sites found for this outbound message: " + messageType);
        }

        // Process each message one at a time
        for (Iterator iter = facilities.iterator(); iter.hasNext();)
        {
            VAFacility facility = (VAFacility)iter.next();
            service.processMessage(((EGTTriggerEvent)obj).getEGTSetting(), facility);
        }
    }

    /**
     * Method to OutboundMessageMasterFileService bean through Mapping.
     *
     * @param messageType The input used to find a corresponding OutboundMessageMasterFileService.
     */
    private OutboundMessageMasterFileService getOutboundMasterFileService(
        String messageType)
    {
        OutboundMessageMasterFileService service = (OutboundMessageMasterFileService)outboundProcessServices
            .get(messageType);

        if (service == null)
        {
            throw new InvalidConfigurationException(messageType
                + " not mapped to an OutboundMessageMasterFileService");
        }

        return service;
    }

    public LookupService getLookupService()
    {
        return lookupService;
    }

    public void setLookupService(LookupService lookupService)
    {
        this.lookupService = lookupService;
    }
}
