/********************************************************************
 * Copyright  2004 EDS. All rights reserved
 ********************************************************************/
//Package
package gov.va.med.esr.messaging.builder.message;

//Java Classes
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

//Framework Classes
import gov.va.med.fw.hl7.Segment;
import gov.va.med.fw.hl7.segment.ZRD;
import gov.va.med.fw.hl7.builder.HL7MetaData;
import gov.va.med.fw.hl7.constants.SegmentConstants;
import gov.va.med.fw.util.builder.BuilderException;

//ESR Classes
import gov.va.med.esr.common.model.lookup.Disability;
import gov.va.med.esr.common.model.ee.RatedDisability;
import gov.va.med.esr.common.model.ee.ServiceConnectionAward;
import gov.va.med.esr.common.model.person.Person;

/**
 * Class to build the ZRD segment.
 * 
 * @author Alex Yoon
 * @version 1.0
 */
public class ZRDBuilder extends AbstractSegmentBuilder
{  
  
    /**
    * An instance of serialVersionUID 
    */
   private static final long serialVersionUID = 4500473786960249887L;

   /**
     * Default contructor.
     */
    public ZRDBuilder()
    {
        super();
    }

    /**
     * Method to build the ZRD segment.
     * 
     * @param metaData
     *            The parameter object used to build the a ZRD segment.
     * @return The ZRD segment.
     */
    public Object build(HL7MetaData metaData) throws BuilderException
    {
        if ((metaData == null) || (metaData.getEntity() == null)
                || !(metaData.getEntity() instanceof Person))
        {
            throw new BuilderException(
                    "Invalid input parameter to build a segment");
        }

        List segments = new ArrayList();
        Person person = (Person) metaData.getEntity();

        try
        {
            int setID = 1;

            ServiceConnectionAward scAward = person.getServiceConnectionAward();
            if (scAward != null)
            {
                Set ratedDisabilities = scAward.getRatedDisabilities();

                if ((ratedDisabilities != null)
                        && (ratedDisabilities.size() > 0))
                {
                    Iterator iterRatedDisabilities = ratedDisabilities
                            .iterator();
                    while (iterRatedDisabilities.hasNext())
                    {
                        RatedDisability theRatedDisability = (RatedDisability) iterRatedDisabilities
                                .next();
                        if (theRatedDisability != null)
                        {
                            //Get Disability object.
                            Disability disability = theRatedDisability
                                    .getDisability();
                            if (disability != null)
                            {
                                ZRD segment = new ZRD();
                                setDefaultValues(segment);
                                
                                segment.setSetID(String.valueOf(setID));
                                segment.setDisabilityCondition(ElementFormatter
                                        .formatCodedElement(disability
                                                .getCode(), disability
                                                .getName()));
                                segment
                                        .setDisabilityPercentage((theRatedDisability
                                                .getPercentage() != null) ? theRatedDisability
                                                .getPercentage().toString()
                                                : null);
                                segment
                                        .setDiagnosticExtremity(theRatedDisability
                                                .getDiagnosticExtremity() == null ? null
                                                : theRatedDisability
                                                        .getDiagnosticExtremity()
                                                        .getCode());
                                if (theRatedDisability
                                        .getOriginalEffectiveRatingDate() != null)
                                {
                                    segment
                                            .setRatedDisabilityOrigEffectiveDate(DateFormatter
                                                    .formatDate(theRatedDisability
                                                            .getOriginalEffectiveRatingDate()));
                                }
                                if (theRatedDisability.getRatingDate() != null)
                                {
                                    segment
                                            .setRatedDisabilityCurrentEffectiveDate(DateFormatter
                                                    .formatDate(theRatedDisability
                                                            .getRatingDate()));
                                }
                                segments.add(segment);
                                setID += 1;
                            }
                        }
                    }
                }
            }

            //If no ZRD data, build a empty segment
            if (setID == 1)
            {
                ZRD segment = new ZRD();
                segment.setSetID("1");
                setDefaultValues(segment);
                
                segments.add(segment);
            }
        } catch (Exception e)
        {
            throw new BuilderException(
                    "Failed to build ZRD Segment due to an exception ", e);
        }

        return segments;
    }
    
    protected void setDefaultValues(Segment segment)
    {
        ZRD zrd = (ZRD) segment;
        zrd.setDisabilityCondition(SegmentConstants.DEFAULT_VALUE);
        zrd.setDisabilityPercentage(SegmentConstants.DEFAULT_VALUE);
        zrd.setDiagnosticExtremity(SegmentConstants.DEFAULT_VALUE);
        zrd.setRatedDisabilityOrigEffectiveDate(SegmentConstants.DEFAULT_VALUE);
        zrd.setRatedDisabilityCurrentEffectiveDate(SegmentConstants.DEFAULT_VALUE);

    }
}