package gov.va.med.esr.messaging.builder.message;

import gov.va.med.esr.common.model.lookup.NameType;
import gov.va.med.esr.common.model.messaging.SiteIdentity;
import gov.va.med.esr.common.model.person.Name;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.service.trigger.PersonTrigger;
import gov.va.med.esr.service.trigger.PersonTriggerEvent;
import gov.va.med.esr.service.trigger.PersonTriggerIdentity;
import gov.va.med.fw.hl7.Segment;
import gov.va.med.fw.hl7.constants.SegmentConstants;
import gov.va.med.fw.hl7.segment.PID;
import gov.va.med.fw.util.builder.BuilderException;

/**
 * Builds the PID segment for Z05 message based on different
 * datatypes.
 * 
 * @author Rajiv Patnaik Created on Sep 8, 2005
 * @version 1.0
 * 
 * Copyright  2005 VHA. All rights reserved
 */
public class PIDZ05Builder extends AbstractPIDBuilder
{

    /**
    * An instance of serialVersionUID 
    */
   private static final long serialVersionUID = 6947624802711523259L;

   /**
     * Method to build the PID segment.
     * 
     * @param metaData
     *            The parameter object used to build the a PID  segment.
     * @return The PID segment.
     */
    public Object build(PIDMetaData metaData,
            PersonTriggerEvent triggerEvent) throws BuilderException
    {

        if ((metaData == null) || (metaData.getEntity() == null)
                || !(metaData.getEntity() instanceof Person)
                || (metaData.getSiteIdentity() == null) || triggerEvent == null)
        {
            throw new BuilderException(
                    "Invalid input parameter to build a Z05 PID segment");
        }

        PID segment = new PID();

        try
        {
            PersonTriggerIdentity personTriggerIdentity = (PersonTriggerIdentity) triggerEvent
                    .getTriggerIdentity();
            PersonTrigger.DataType dataType = (PersonTrigger.DataType) personTriggerIdentity
                    .getApplicableDataTypes().iterator().next();

            SiteIdentity siteIdentity = metaData.getSiteIdentity();
            Person person = (Person) metaData.getEntity();

            //Sequences 1-3,5,7,8,19 are common. Set them first
            //Sequence 1
            segment.setSetID("1");

            //Sequence 2
            String VPIDKey = person.getVPIDEntityKey() == null ? null : person
                    .getVPIDEntityKey().getVPID();             
            segment.setPrimaryID(getPersonHelperService().getICNChecksum(VPIDKey));

            //Sequence 3
            segment.setPatientIdentifierList(getPatientIdentifierList(person,
                    siteIdentity));

            //Sequence 5
            segment.setPatientName(ElementFormatter.formatName(Name
                    .getNameOfType(person.getNames(), NameType.LEGAL_NAME
                            .getName())));
            //Sequence 6
            //Stop Sending Messages for Mothers Maiden Name per ESR 3.7_CodeCR11435
            //segment.setMotherMaidenName(person.getMothersMaidenName());
            
            //Sequence 7
            if (person.getBirthRecord() != null)
            {
                segment.setDOB(DateFormatter.formatDate(person.getBirthRecord()
                        .getBirthDate()));
            }

            //Sequence 8
            segment.setSex(buildGender(person.getGender()));

            //Sequence 10
            if ( person.getRaces() != null && person.getRaces().size() > 0 )
            {
            	setRace(person, segment);
            }
            
            //sequence 16 
            if ( person.getMaritalStatus() != null)
            {
            	segment.setMaritalStatus(person.getMaritalStatus().getCode());
            }
            //sequence 17
            if ( person.getReligion() != null)
            {
            	segment.setReligion(person.getReligion().getCode());
            }
            
            //Sequence 19
            setSSN(person, segment);

            //Sequence 22
            if ( person.getEthnicity() != null )
            {
            	setEthnicity(person,segment);
            }
            
            //Sequence 24
            if ( person.getBirthRecord() != null )
            {
            	segment.setMultipleBirthIndicator(buildBooleanFor0136(person.getBirthRecord().getMultipleBirth()));
            }
            
            setSSN(person, segment);
            
            
            if (dataType == PersonTrigger.DataType.ADDRESS
                    || dataType == PersonTrigger.DataType.GUARDIAN_INFO)
            {
                //Sequence 11,12 (Address and County code)
                setAddress(person, segment);

                //Sequence 13
                setPhonesAndEmails(person, segment); 
            }

        } catch (BuilderException e)
        {
            throw new BuilderException(
                    "Failed to build PID Segment for Z05 due to an exception ",
                    e);
        }
        return segment;
    }
    
    protected void setDefaultValues(Segment segment)
    {
        PID pid = (PID)segment;
        super.setDefaultValues(pid);
        
        pid.setPatientAddress(SegmentConstants.DEFAULT_VALUE);
        pid.setCountyCode(SegmentConstants.DEFAULT_VALUE);
        pid.setHomePhoneNumber(SegmentConstants.DEFAULT_VALUE);
    }    
}