/********************************************************************
 * Copyright  2004 EDS. All rights reserved
 ********************************************************************/
//Package
package gov.va.med.esr.messaging.builder.message;

//Java Classes
import java.util.ArrayList;
import java.util.List;

import gov.va.med.fw.hl7.InvalidMessageException;
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Message;
import gov.va.med.fw.hl7.builder.HL7MetaData;
import gov.va.med.fw.model.EntityKey;
import gov.va.med.fw.util.builder.BuilderException;

import gov.va.med.esr.common.model.messaging.SiteIdentity;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.messaging.constants.HL7Constants;

/**
 * Class to build the ORUZ04 message from a person object. 
 * 
 * @author Alex Yoon
 * @version 1.0
 */
public class ORUZ04Builder extends AbstractUnsolicitedMessageBuilder {
    /**
    * An instance of serialVersionUID 
    */
   private static final long serialVersionUID = 8080690352436585481L;
   private IN1Builder in1Builder;
    private ZIVBuilder zivBuilder;
    
    /**
     * Default constructor  
     */
    public ORUZ04Builder()
    {
        super();
    }
    
    public IN1Builder getIn1Builder()
    {
        return this.in1Builder;
    }

    public void setIn1Builder(IN1Builder in1Builder)
    {
        this.in1Builder = in1Builder;
    }
    
    public ZIVBuilder getZivBuilder()
    {
        return this.zivBuilder;
    }

    public void setZivBuilder(ZIVBuilder zivBuilder)
    {
        this.zivBuilder = zivBuilder;
    }
        
    /**
     * Method to build the ORUZ04 message. 
     * 
     * @param object array
     * 			The person who is source of message content.
     * @param siteIdentity
     * 			The siteIdentity that is a destination of message.
     * @return The ORUZ04 message.
     * @throws BuilderException
     */
    public Message build(Object[] args) throws BuilderException
    {
        return this.build((Person)args[0], (SiteIdentity)args[1], (EntityKey)args[2]);
    }
    
    /**
     * Method to build the ORUZ04 message. 
     * 
     * @param 	The person who is source of message content.
     * @param siteIdentity
     * 			The siteIdentity that is a destination of message.
     * @return 	The ORUZ04 message.
     * @throws 	BuilderException
     */
    public Message build(Person person, SiteIdentity siteIdentity, EntityKey entityKey) throws BuilderException {
        
        if (person == null || siteIdentity == null)
        {
            throw new BuilderException(
                    "Invalid input parameters to build an object");
        }
        
        if(super.logger.isInfoEnabled())
        {
            super.logger.info("Building outbound ORUZ04");

            if(person.getEntityKey() != null)
            {
                super.logger.info("Person id = "
                    + person.getEntityKey());
            }
            
            if(siteIdentity.getVaFacility() != null)
            {
                super.logger.info("Station Number = "
                    + siteIdentity.getVaFacility().getStationNumber());
            }
        }
        
        Message message = null;
        List segments = new ArrayList();
	    
        try {
            
            HL7MetaData metaData = new HL7MetaData(person);
            
            segments.add(super.buildMSHSegment(siteIdentity, HL7Constants.ORUZ04)); 	
            segments.add(super.buildPIDSegment(person, siteIdentity));
            segments.add(this.in1Builder.build(metaData, entityKey));
            segments.add(this.zivBuilder.build(metaData, entityKey));
            
            message = new Message(segments, HL7Constants.ORUZ04,true);

            if(super.logger.isInfoEnabled())
            {
                super.logger.info("Built outbound ORUZ04 " + message.getMessageID());
            }
        }
        catch(InvalidSegmentException me)
        {
            throw new BuilderException(
                "Failed to build ORUZ04 message due to an segment exception ",
                me);
        }
        catch(InvalidMessageException me)
        {
            throw new BuilderException(
                "Failed to build ORUZ04 message due to an exception ", me);
        }
        
                
        return message;
    }
}
