package gov.va.med.esr.messaging.builder.message;

import gov.va.med.esr.common.model.lookup.AckType;
import gov.va.med.esr.common.model.messaging.SiteIdentity;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.messaging.constants.HL7Constants;
import gov.va.med.esr.service.trigger.TargetedPersonTriggerEvent;
import gov.va.med.fw.hl7.InvalidMessageException;
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Message;
import gov.va.med.fw.util.builder.BuilderException;

import java.util.ArrayList;
import java.util.List;

/**
 * Builds the outbound ORFZ11 message in response to a QRYZ11 query.
 *  
 * @author Rajiv Patnaik Created on Jan 10, 2006
 * @version 1.0
 * 
 * Copyright  2005 VHA. All rights reserved
 */
public class ORFZ11Builder extends AbstractZ11Builder
{
    /**
    * An instance of serialVersionUID 
    */
   private static final long serialVersionUID = 4837952081062167603L;

   /**
     * Method to build the ORFZ11 message.
     * 
     * @param object
     *            array The person who is source of message content.
     * @param siteIdentity
     *            The siteIdentity that is a destination of message.
     * @return The ORFZ11 message.
     * @throws BuilderException
     */
    public Message build(Object[] args) throws BuilderException
    {
        return this
                .build((Person) args[0], (SiteIdentity) args[1],
                        (args[3] == null ? null
                                : (TargetedPersonTriggerEvent) args[3]));
    }

    public Message build(Person person, SiteIdentity siteIdentity,
            TargetedPersonTriggerEvent triggerEvent) throws BuilderException
    {

        if (person == null || siteIdentity == null || triggerEvent == null)
        {
            throw new BuilderException(
                    "Invalid input parameters to build an object");
        }
        
        Message message = null;

        List segments = new ArrayList();

        try
        {
            segments.add(super.buildMSHSegment(siteIdentity,
                    HL7Constants.ORFZ11));

            segments.add(super.buildMSASegment(AckType.CODE_AA.getName(),
                    triggerEvent.getIncomingMessageInfo()
                            .getReferenceMessageId(), null) );

            String ssn = person.getOfficialSsn() == null ? null : person
                    .getOfficialSsn().getFormattedSsnText();
            
            segments.add(super.buildQRDSegment(person, siteIdentity.getDfn(),
                    ssn, triggerEvent.getIncomingMessageInfo()
                            .getQueryDateTime(),
                    HL7Constants.WHAT_SUBJECT_FILTER_OTH,
                    HL7Constants.WHAT_DEPARTMENT_DATA_CODE_ENROLLMENT));

            segments.add(super.buildQRFSegment(person));

            super.buildCommonZ11Segments(segments, person, siteIdentity);

            message = new Message(segments, HL7Constants.ORFZ11, true);

            if (super.logger.isInfoEnabled())
            {
                super.logger.info("Built outbound ORFZ11 with message ID " + message.getMessageID()); 
            }
            
        } catch (InvalidSegmentException me)
        {
            throw new BuilderException(
                    "Failed to build ORFZ11 message due to an segment exception ",
                    me);
        } catch (InvalidMessageException me)
        {
            throw new BuilderException(
                    "Failed to build ORFZ11 message due to an exception ", me);
        }


        return message;
    }

}