package gov.va.med.esr.messaging.builder.message;

import java.util.ArrayList;
import java.util.List;

import gov.va.med.esr.common.model.lookup.AckType;
import gov.va.med.esr.common.model.messaging.SiteIdentity;
import gov.va.med.esr.common.model.person.Person;
import gov.va.med.esr.messaging.constants.HL7Constants;
import gov.va.med.esr.service.trigger.IncomeYearTriggerEvent;
import gov.va.med.esr.service.trigger.PersonTriggerEvent;
import gov.va.med.fw.hl7.InvalidMessageException;
import gov.va.med.fw.hl7.InvalidSegmentException;
import gov.va.med.fw.hl7.Message;
import gov.va.med.fw.util.builder.BuilderException;

/**
 * Builds the outbound ORFZ10 message in response to a QRYZ10 query.
 * 
 * @author Rajiv Patnaik Created on Dec 22, 2005
 * @version 1.0
 * 
 * Copyright  2005 VHA. All rights reserved
 */
public class ORFZ10Builder extends AbstractZ10Builder
{

    /**
    * An instance of serialVersionUID 
    */
   private static final long serialVersionUID = 632956614942470376L;

   /**
     * Method to build the ORFZ10 message.
     * 
     * @param object
     *            array The person who is source of message content.
     * @param siteIdentity
     *            The siteIdentity that is a destination of message.
     * @return The ORFZ10 message.
     * @throws BuilderException
     */
    public Message build(Object[] args) throws BuilderException
    {
        return this.build((Person) args[0], (SiteIdentity) args[1],
                (args[3] == null ? null : (PersonTriggerEvent) args[3]));
    }

    /**
     * @param person
     * @param siteIdentity
     * @param triggerEvent
     * @return
     * @throws BuilderException
     */
    public Message build(Person person, SiteIdentity siteIdentity,
            PersonTriggerEvent triggerEvent) throws BuilderException
    {
        if (person == null || siteIdentity == null || triggerEvent == null
                || !(triggerEvent instanceof IncomeYearTriggerEvent))
        {
            throw new BuilderException(
                    "Invalid input parameters to build an object");
        }

        IncomeYearTriggerEvent incomeYearEvent = (IncomeYearTriggerEvent) triggerEvent;

        Message message = null;

        List segments = new ArrayList();

        try
        {
            segments.add(super.buildMSHSegment(siteIdentity,
                    HL7Constants.ORFZ10));

            segments.add(super.buildMSASegment(AckType.CODE_AA.getName(),
                    incomeYearEvent.getIncomingMessageInfo()
                            .getReferenceMessageId(), null));

            String whatDepartmentCode = incomeYearEvent.getIncomeYear()
                    + "0000";

            String ssn = person.getOfficialSsn() == null ? null : person
                    .getOfficialSsn().getFormattedSsnText();

            segments.add(super.buildQRDSegment(person, siteIdentity.getDfn(),
                    ssn, incomeYearEvent.getIncomingMessageInfo()
                            .getQueryDateTime(),
                    HL7Constants.WHAT_SUBJECT_FILTER_FIN, whatDepartmentCode));

            segments.add(super.buildQRFSegment(person));

            super.buildCommonZ10Segments(segments, person, siteIdentity,
                    incomeYearEvent);

            message = new Message(segments, HL7Constants.ORFZ10, true);

            if (super.logger.isInfoEnabled())
            {
                super.logger.info("Built outbound ORFZ10 " +message.getMessageID()); 
            }
            
        } catch (InvalidSegmentException me)
        {
            throw new BuilderException(
                    "Failed to build ORFZ10 message due to an segment exception ",
                    me);
        } catch (InvalidMessageException me)
        {
            throw new BuilderException(
                    "Failed to build ORFZ10 message due to an exception ", me);
        }


        return message;
    }

}