/********************************************************************
 * Copyright  2004 EDS. All rights reserved
 ********************************************************************/
//Package
package gov.va.med.esr.messaging.builder.message;

//Java Classes
import java.util.Calendar;
import java.util.Date;

//Framework Classes
import gov.va.med.fw.hl7.Message;
import gov.va.med.fw.hl7.InvalidMessageException;
import gov.va.med.fw.hl7.builder.HL7MetaData;
import gov.va.med.fw.model.AbstractEntity;

//EDB Classes
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.messaging.constants.HL7Constants;


/**
 * Class that functions as parameter object to build MSH segment.
 * 
 * @author Alex Yoon
 * @version 1.0
 */
public class MSHMetaData extends HL7MetaData {
	/**
    * An instance of serialVersionUID 
    */
   private static final long serialVersionUID = -55914733321420169L;
   private String receivingApplication = null;
	private String receivingFacility = null;
	private String messageType = null;
	private String controlID = null;
	private Date creationDatetime = null;
	private String applicationACKType = null;
	
	/**
	 * Contructor for initializing object using a message.
	 * 
	 * @param message
	 * 			The message object.
	 */
	public MSHMetaData(Message message) {
		super(message);
		
		try {
		    if(message.getSendingApplication().equals(HL7Constants.HEADER_SENDING_APPLICATION_MVR))
		    {
		        this.receivingApplication = HL7Constants.HEADER_RECEIVING_APPLICATION_MVR;
		        this.receivingFacility = (VAFacility.CODE_MVR.getName() + HL7Constants.HEADER_RECEIVING_FACILITY_MVR);
		    }else{
		        this.receivingApplication = message.getSendingApplication();
		        this.receivingFacility = message.getSendingFacility();
		    }
			

		    this.controlID = message.getMessageID();
			this.messageType = message.getType();
			this.creationDatetime = Calendar.getInstance().getTime();
			this.applicationACKType = HL7Constants.APPLICATION_ACK_TYPE_NE;
		}
		catch (InvalidMessageException e) {
		   if( logger.isDebugEnabled() ) {
		      logger.debug( "Unexpected error occured", e );
		   }
		}
	}

	/**
	    * Constructor that takes entity object to build a segment.
	    * 
	    * @param entity
	    *           The entity that is source of segment data.
	    */
	   public MSHMetaData (AbstractEntity entity) {
	       super(entity);
	   }
	   
	/**
	 * Return the receiving application.
	 * 
	 * @return The receiving application.
	 */
	public String getReceivingApplication() {
		return this.receivingApplication;
	}
	
	/**
	 * Set the receiving application.
	 * 
	 * @param receivingApplication
	 * 			The receiving application.
	 */
	public void setReceivingApplication(String receivingApplication) {
		this.receivingApplication = receivingApplication;
	}
	
	/**
	 * Return the receiving facility.
	 * 
	 * @return The receiving facility.
	 */
	public String getReceivingFacility() {
		return this.receivingFacility;
	}
	
	/**
	 * Set the receiving facility.
	 * 
	 * @param receivingFacility
	 * 			The receiving facility to set.
	 */
	public void setReceivingFacility(String receivingFacility) {
		this.receivingFacility = receivingFacility;
	}
	
	/**
	 * Return the message type.
	 * 
	 * @return The message type.
	 */
	public String getMessageType() {
		return this.messageType;
	}
	
	/**
	 * Set the message type.
	 * 
	 * @param messageType
	 * 			The message type to set.
	 */
	public void setMessageType(String messageType) {
		this.messageType = messageType;
	}
	
	/**
	 * Return the control id of the message.
	 * 
	 * @return The control id.
	 */
	public String getControlID() {
		return this.controlID;
	}
	
	/**
	 * Set the control id of the message.
	 * 
	 * @param controlID
	 * 			The control id to use for the message.
	 */
	public void setControlID(String controlID) {
		this.controlID = controlID;
	}
	
	/**
	 * Return the date/time when message was created.
	 * 
	 * @return The date/time when message was created.
	 */
	public Date getCreationDatetime() {
		return this.creationDatetime;
	}
	
	/**
	 * Set the creation date/time of message.
	 * 
	 * @param creationDatetime
	 * 			The date/time when message was created.
	 */
	public void setCreationDatetime(Date creationDatetime) {
		this.creationDatetime = creationDatetime;
	}
	
	/**
	 * Return the application ACK type.
	 * 
	 * @return The application ACK type.
	 */
	public String getApplicationACKType() {
		return this.applicationACKType;
	}
	
	/**
	 * Set the application ACK type.
	 * 
	 * @param applicationACKType
	 * 			Only for ACK message - application ack is AL or NE
	 */
	public void setApplicationACKType(String applicationACKType) {
		this.applicationACKType = applicationACKType;
	}
}
