/********************************************************************
 * Copyright  2004 EDS. All rights reserved
 ********************************************************************/
//Package
package gov.va.med.esr.messaging.builder.message;

//Java Classes
import java.util.Date;
import java.util.Calendar;

//Framework Classes
import gov.va.med.esr.common.model.lookup.VAFacility;
import gov.va.med.esr.messaging.constants.HL7Constants;
import gov.va.med.fw.hl7.BatchMessage;
import gov.va.med.fw.hl7.builder.HL7MetaData;
import gov.va.med.fw.hl7.InvalidMessageException;
import gov.va.med.fw.model.AbstractEntity;

/**
 * The class that serves as parameter object for building a BHS segment.
 * 
 * @author Alex Yoon
 * @version 1.0
 */
public class BHSMetaData extends HL7MetaData
{

    /**
    * An instance of serialVersionUID 
    */
   private static final long serialVersionUID = 107817524517812822L;

   private String receivingApplication = null;

    private String receivingFacility = null;

    private String messageType = null;

    private String controlID = null;

    private Date creationDatetime = null;

    private String comment = null; //Only used for ACK

    /**
     * Constructor that initializes object using a batch message.
     * 
     * @param message
     *            The batch message that is source of segment data.
     */
    public BHSMetaData(BatchMessage message)
    {
        super(message);

        try
        {

            if (message.getSendingApplication().equals(
                    HL7Constants.HEADER_SENDING_APPLICATION_MVR))
            {
                this.receivingApplication = HL7Constants.HEADER_RECEIVING_APPLICATION_MVR;
                this.receivingFacility = (VAFacility.CODE_MVR.getName() + HL7Constants.HEADER_RECEIVING_FACILITY_MVR);
            } else
            {
                this.receivingApplication = message.getSendingApplication();
                this.receivingFacility = message.getSendingFacility();
            }

            this.controlID = message.getMessageID();
            this.messageType = message.getType();
            this.creationDatetime = Calendar.getInstance().getTime();
        } catch (InvalidMessageException e)
        {
            if (logger.isDebugEnabled())
            {
                logger.debug("Unexpected exception occured", e);
            }
        }
    }

    /**
     * Constructor that takes entity object to build a segment.
     * 
     * @param entity
     *            The entity that is source of segment data.
     */
    public BHSMetaData(AbstractEntity entity)
    {
        super(entity);
    }

    /**
     * Returns the receiving application.
     * 
     * @return The receiving application.
     */
    public String getReceivingApplication()
    {
        return this.receivingApplication;
    }

    /**
     * Set the receiving application.
     * 
     * @param receivingApplication
     *            The receiving application of the message.
     */
    public void setReceivingApplication(String receivingApplication)
    {
        this.receivingApplication = receivingApplication;
    }

    /**
     * Returns the receiving facility.
     * 
     * @return The receiving facility.
     */
    public String getReceivingFacility()
    {
        return this.receivingFacility;
    }

    /**
     * Set the receiving facility.
     * 
     * @param receivingFacility
     *            The receiving facility of the message.
     */
    public void setReceivingFacility(String receivingFacility)
    {
        this.receivingFacility = receivingFacility;
    }

    /**
     * Return the message type.
     * 
     * @return The message type.
     */
    public String getMessageType()
    {
        return this.messageType;
    }

    /**
     * Set the message type.
     * 
     * @param messageType
     *            The message type of the message.
     */
    public void setMessageType(String messageType)
    {
        this.messageType = messageType;
    }

    /**
     * Return the control id of the message.
     * 
     * @return The control id of the message.
     */
    public String getControlID()
    {
        return this.controlID;
    }

    /**
     * Set the control id of the message.
     * 
     * @param controlID
     *            The control id of the message.
     */
    public void setControlID(String controlID)
    {
        this.controlID = controlID;
    }

    /**
     * Return the date/time the message was created.
     * 
     * @return The date/time.
     */
    public Date getCreationDatetime()
    {
        return this.creationDatetime;
    }

    /**
     * Set the creation data and time.
     * 
     * @param creationDatetime
     *            The date and time the message was created.
     */
    public void setCreationDatetime(Date creationDatetime)
    {
        this.creationDatetime = creationDatetime;
    }

    /**
     * Returns the comment.
     * 
     * @return The comment.
     */
    public String getComment()
    {
        return this.comment;
    }

    /**
     * Sets the comment.
     * 
     * @param comment
     *            Only for ACK message - ACK is for AA or AE
     */
    public void setComment(String comment)
    {
        this.comment = comment;
    }
}