package gov.va.med.esr.messaging.builder.message;

import gov.va.med.fw.hl7.segment.ZPD;
import gov.va.med.fw.util.builder.BuilderException;

import gov.va.med.esr.common.model.lookup.Indicator;
import gov.va.med.esr.common.model.person.DeathRecord;
import gov.va.med.esr.common.model.person.Person;

/**
 * @author Rajiv Patnaik Created on Sep 9, 2005
 * @version 1.0 Copyright  2005 VHA. All rights reserved
 */
public abstract class AbstractZPDBuilder extends AbstractSegmentBuilder {

    private static final long serialVersionUID = -1370346746640850300L;
    /**
    * Method that sets birth city and state to the segment.
    * 
    * @param person
    *           The Person Object.
    * @param segment
    *           The ZPD segment.
    */
   protected void setBirthRecord(Person person, ZPD segment) {
      String birthCity = null;
      String birthState = null;

      if( person.getBirthRecord() != null ) {
         birthCity = person.getBirthRecord().getCity();
         birthState = person.getBirthRecord().getState();
      }

   }

   /**
    * Method that sets eligible for medicaid and last asked date to the segment.
    * 
    * @param person
    *           The Person Object.
    * @param segment
    *           The ZPD segment.
    */
   protected void setMedicaid(Person person, ZPD segment) throws BuilderException {
      if( person.getMedicaidFactor() != null ) {
         Boolean eligible = person.getMedicaidFactor().isEligibleForMedicaid();

         // Set segment attributes
         segment.setEligibleForMedicaid(buildBooleanForVA001(eligible));
         // not populated
         segment.setMedicaidLastAskedDate(DateFormatter.formatDateTime(person.getMedicaidFactor().getLastUpdateDate(), DateFormatter.DATETIME_ZONE_FORMAT));
      }

   }

   /**
    * Method that sets incompetent ruling and dates to the segment.
    * 
    * @param person
    *           The Person Object.
    * @param segment
    *           The ZPD segment.
    */
   protected void setRatedIncompetent(Person person, ZPD segment) throws BuilderException {

      // Set Incompetent Indicator
      Boolean indicator = person.getIncompetenceRuling() == null ? null : person
            .getIncompetenceRuling().isIncompetent();
      // Set segment attributes
      segment.setRatedIncompetent(buildBooleanForVA001(indicator));

   }

   /**
    * Method that sets death date and source to the segment.
    * 
    * @param person
    *           The Person Object.
    * @param segment
    *           The ZPD segment.
    */
   protected void setDeathRecord(Person person, ZPD segment) {
      String deathDate = null;
      String deathSource = null;
      String lastModifiedDate = null;

      DeathRecord deathRecord = person.getDeathRecord();
      if( deathRecord != null ) {
    	  //Populate with Lazarus date if it is not null. Rest of the sequences will contain double quotes
    	  if(deathRecord.getLazarusDate() != null)
    	  {
              lastModifiedDate = DateFormatter.formatDateWithTime(deathRecord.getLazarusDate());    		  
    	  }
          else {
             lastModifiedDate = DateFormatter.formatDate(deathRecord.getDeathReportDate(),
             "yyyyMMddHHmmssZ");
          }
          
    	  deathDate = DateFormatter.formatDateWithTime(deathRecord.getDeathDate());
    	  deathSource = deathRecord.getDataSource() == null ? null : deathRecord.getDataSource().getCode();
      }
      // Set segment attributes.
      segment.setDeathDate(deathDate);
      segment.setNotificationSource(deathSource);
      segment.setLastModifiedDate(lastModifiedDate);
   }

   /**
    * Method that gets POW indicator.
    * 
    * @param person
    *           The Person Object.
    * @return POW indicator.
    */
   protected String getPOWIndicator(Person person) throws BuilderException {
      Indicator powIndicator = person.getPrisonerOfWar() == null ? null : person
            .getPrisonerOfWar().getPowIndicator();

      return buildBooleanFor0136FromIndicator(powIndicator);
   }

}